/***************************************************************************//**
 * @file mp3vdac.c
 * @brief MP3 Player audio output through VDAC
 * @version  1.00

 *******************************************************************************
 * # License
 * <b>Copyright 2017 Silicon Laboratories, Inc. http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/

#include "mp3config.h"

#if (AUDIO_OUTPUT_SELECT == 0)
// External variables
extern LDMA_TransferCfg_t descConf;
extern LDMA_Descriptor_t descLink[LIST_SIZE];
extern MP3FrameInfo mp3FrameInfo;
extern short outBuf[DECODEBUF_SIZE * 2];

/**************************************************************************//**
 * @brief
 *   VDAC Setup.
 * @details
 *   Setup VDAC in stereo mode and triggered by PRS from TIMER.
 *****************************************************************************/
void setupVdac(void)
{
  VDAC_Init_TypeDef        init        = VDAC_INIT_DEFAULT;
  VDAC_InitChannel_TypeDef initChannel = VDAC_INITCHANNEL_DEFAULT;

  // Setup VDAC clock close to 1 MHz with HFPERCLK
  init.prescaler = VDAC_PrescaleCalc(VDAC_CLOCK, true, 0);

  // Initialize the VDAC
  VDAC_Init(VDAC0, &init);

  // Enable PRS to trigger samples at the right time with the timer
  initChannel.prsSel = VDAC_PRS_SEL;
  initChannel.trigMode = vdacTrigModePrs;

  // Both channels can be configured the same and be triggered by the same PRS-signal
  VDAC_InitChannel(VDAC0, &initChannel, 0);
  VDAC_InitChannel(VDAC0, &initChannel, 1);
  
  // Setup outputs for VDAC channels 0 & 1 and VDAC PRS source
  VDAC0->OPA[1].OUT &= ~VDAC_OPA_OUT_MAINOUTEN;                               
  VDAC0->OPA[1].OUT |= VDAC_OPA1_ALT;   
  PRS_SourceSignalSet(VDAC_PRS_CH, TIMER_PRS, TIMER_SRC, prsEdgeOff);

  // Use peripheral transfer configuration macro for VDAC
  descConf = (LDMA_TransferCfg_t)LDMA_TRANSFER_CFG_PERIPHERAL(VDAC_DMAREQ);
}

/**************************************************************************//**
 * @brief
 *   Setup LDMA in ping pong mode
 * @details
 *   The LDMA is set up to transfer data from memory to the VDAC.
 *   Transfer size is mono/stereo and sampling frequency dependent.
 *****************************************************************************/
void setupVdacDma(void)
{
  if (mp3FrameInfo.nChans == 1)
  {
    // VDAC mono, audio output on VDAC channel 0 (VDAC0->CH0DATA)
    switch (mp3FrameInfo.samprate)
    {
    case 8000:
    case 16000:
    case 11025:
    case 22050:
    case 12000:
    case 24000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &VDAC0->CH0DATA, MAX_NSAMP, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &VDAC0->CH0DATA, MAX_NSAMP, -1);
      break;
      
    case 32000:
    case 44100:
    case 48000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &VDAC0->CH0DATA, MAX_NSAMP * 2, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &VDAC0->CH0DATA, MAX_NSAMP * 2, -1);
      break;
      
    default:
      break;
    }
    VDAC0->CMD = VDAC_CMD_CH0EN;
  }
  else
  {
    // VDAC stereo, audio output on VDAC channel 0 & 1 (VDAC0->COMBDATA)
    switch (mp3FrameInfo.samprate)
    {
    case 8000:
    case 16000:
    case 11025:
    case 22050:
    case 12000:
    case 24000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &VDAC0->COMBDATA, MAX_NSAMP, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &VDAC0->COMBDATA, MAX_NSAMP, -1);
      break;
      
    case 32000:
    case 44100:
    case 48000:
      descLink[0] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf), &VDAC0->COMBDATA, MAX_NSAMP * 2, 1);
      descLink[1] = (LDMA_Descriptor_t)LDMA_DESCRIPTOR_LINKREL_M2P_HALF((uint32_t *)(outBuf + MAX_NSAMP * 4), &VDAC0->COMBDATA, MAX_NSAMP * 2, -1);
      break;
      
    default:
      break;
    }
    // 32 bit transfer for stereo
    descLink[0].xfer.size = ldmaCtrlSizeWord;
    descLink[1].xfer.size = ldmaCtrlSizeWord;
    VDAC0->CMD = VDAC_CMD_CH0EN + VDAC_CMD_CH1EN;
  }
  descLink[0].xfer.link = 1;
  descLink[1].xfer.link = 1;
  
  // Setup TIMER PRS trigger period
  TIMER_TopBufSet(TIMER_VDAC, (CMU_ClockFreqGet(TIMER_CLOCK)/mp3FrameInfo.samprate - 1));
}
#endif
