//-----------------------------------------------------------------------------
// UART_Idle_F30x.c
//-----------------------------------------------------------------------------
// Copyright 2002 Cygnal Integrated Products, Inc.
//
// AUTH: FB
// DATE: 6 NOV 02
//
//
// This example shows how a system can wake from Idle mode upon receiving
// a wakeup signal on the UART RX line.  The system operates on the internal
// oscillator divided by 8 in Normal mode. When in Idle mode, the system uses
// the external oscillator in C-mode as its clock source. This code assumes
// a 33pF capacitor is present between XTAL2 and GND. The capacitor causes
// oscillation between 5kHz and 10kHz when the external oscillator drive
// current (XFCN) is set to its lowest value.
//
// When in Normal mode, the program gets one character from the UART at 2400
// baud, transmits a string containing the character, and goes back into Idle
// mode. The system consumes approximately 4.2 uA in Idle mode and 1.5 mA in
// Normal mode.
//
// The wakeup character must have only one falling edge followed by only one
// rising edge. Since the start bit is a '0' and the stop bit is a '1',
// example wakeup characters are 0x00, 0xFF, and 0xF0 but not 0x0F. Keep in
// mind that characters are sent LSB first.
//
// The text file "FF_H.txt" contains an 0xFF character followed by 'H'. It
// can be sent over UART to wake up the system from Idle mode.
//
// Target: C8051F30x
// Tool chain: KEIL C51 6.03 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <c8051f300.h>                 // SFR declarations
#include <stdio.h>
#include <math.h>

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F30x
//-----------------------------------------------------------------------------

sfr16 DP       = 0x82;                 // data pointer
sfr16 TMR2RL   = 0xca;                 // Timer2 reload value
sfr16 TMR2     = 0xcc;                 // Timer2 counter
sfr16 PCA0CP1  = 0xe9;                 // PCA0 Module 1 Capture/Compare
sfr16 PCA0CP2  = 0xeb;                 // PCA0 Module 2 Capture/Compare
sfr16 PCA0     = 0xf9;                 // PCA0 counter
sfr16 PCA0CP0  = 0xfb;                 // PCA0 Module 0 Capture/Compare

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define INTCLK       24500000 / 8      // Internal Oscillator frequency
                                       // in Hz (divide by 8 mode)
#define EXTCLK       5000              // Frequency of external capacitor
                                       // oscillator

#define BAUDRATE     2400              // Baudrate in bits per second

sbit LED = P0^2;                       // LED='1' means ON
sbit SW2 = P0^3;                       // SW2='0' means switch pressed
sbit TX0_PIN = P0^4;                   // UART TX0 pin
sbit RX0_PIN = P0^5;                   // UART RX0 pin

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void PORT_Init (void);
void UART0_Init (void);
void INT0_ISR (void);


//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

bit UART_ACTIVE = 0;                   // Flag indicating system is in Normal
                                       // mode operating at 3.0625 MHz

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) {

   char c;

   // Disable Watchdog timer
   PCA0MD &= ~0x40;                    // WDTE = 0 (clear watchdog timer
                                       // enable)
   PORT_Init ();                       // initialize crossbar and GPIO
   UART0_Init ();                      // initialize UART0

   IT01CF = 0x05;                      // Configure External Interrupt 0 to
                                       // generate an interrupt on the falling
                                       // edge of P0.5 (UART RX signal)

   EA = 1;                             // Enable global interrupts

   while (1) {

      if(UART_ACTIVE){
         c = getchar();                // Get the next character
         printf("\nThe character you entered was: %c", c);

      }

      printf("\n\nTransmit an (0xFF) to wake up the system.\n");

      UART_ACTIVE = 0;                 // Make device ready for Idle mode

      P0MDOUT &= ~0x10;                // Make TX0_PIN open-drain
      TX0_PIN = 1;                     // Make TX0_PIN high impedance

      REN0 = 0;                        // Disable UART reception

      OSCXCN = 0x50;                   // Start external oscillator in C mode

      RSTSRC = 0x00;                   // Disable missing clock detector

      OSCICN = 0x08;                   // Switch to external oscillator
                                       // and disable internal oscillator

      TR1 = 0;                         // Disable Timer1

      EX0 = 1;                         // Enable External Interrupt 0

      PCON |= 0x01;                    // Go into Idle mode
   }
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// INT0_ISR
//-----------------------------------------------------------------------------
//
// This Interrupt Service Routine is called when a UART character is received
// when the system is in Idle mode.
//
// It enables the UART and sets the system state variable <UART_ACTIVE> to '1'.
//
void INT0_ISR (void) interrupt 0 {


   OSCICN = 0x04;                      // Enable Internal oscillator in divide
                                       // by 8 mode and switch to it

   EX0 = 0;                            // Disable External Interrupt0

   TR1 = 1;                            // Enable Timer1
   REN0 = 1;                           // Enable UART reception

   P0MDOUT |= 0x10;                    // enable TX0 as a push-pull output

   UART_ACTIVE = 1;                    // Indicate UART is ready for communication
}


//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
// P0.0 -
// P0.1 -
// P0.2 - LED (push-pull)
// P0.3 - SW2
// P0.4 - UART TX (push-pull)
// P0.5 - UART RX
// P0.6 -
// P0.7 - C2D
//
void PORT_Init (void)
{
   XBR0    = 0x08;                     // skip XTAL2 in the crossbar assignments
   XBR1    = 0x03;                     // UART0 TX and RX pins enabled
   XBR2    = 0x40;                     // Enable crossbar and weak pull-ups
   P0MDOUT |= 0x10;                    // enable TX0 as a push-pull output
   P0MDIN &= ~0x08;                    // Configure XTAL2 as an analog input


}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//
void UART0_Init (void)
{
   SCON0 = 0x10;                      // SCON0: 8-bit variable bit rate
                                      //        level of STOP bit is ignored
                                      //        RX enabled
                                      //        ninth bits are zeros
                                      //        clear RI0 and TI0 bits
   if (INTCLK/BAUDRATE/2/256 < 1) {
      TH1 = -(INTCLK/BAUDRATE/2);
      CKCON |= 0x10;                  // T1M = 1; SCA1:0 = xx
   } else if (INTCLK/BAUDRATE/2/256 < 4) {
      TH1 = -(INTCLK/BAUDRATE/2/4);
      CKCON &= ~0x13;
      CKCON |=  0x01;                 // T1M = 0; SCA1:0 = 01
   } else if (INTCLK/BAUDRATE/2/256 < 12) {
      TH1 = -(INTCLK/BAUDRATE/2/12);
      CKCON &= ~0x13;                 // T1M = 0; SCA1:0 = 00
   } else {
      TH1 = -(INTCLK/BAUDRATE/2/48);
      CKCON &= ~0x13;
      CKCON |=  0x02;                 // T1M = 0; SCA1:0 = 10
   }

   TL1 = TH1;                         // set Timer1 to reload value
   TMOD &= ~0xf0;                     // TMOD: timer 1 in 8-bit autoreload
   TMOD |=  0x20;
   TR1 = 1;                           // START Timer1
   TI0 = 1;                           // Indicate TX0 ready
}


