//-----------------------------------------------------------------------------
// Data_Logger_RTC.c
//-----------------------------------------------------------------------------
// Copyright 2002 Cygnal Integrated Products, Inc.
//
// AUTH: FB, JM
// DATE: 03 SEP 03
//
//
// This application uses a 22.1184 MHz crystal oscillator to implement a 
// software real-time clock (RTC). PCA Module 0, configured to count Timer 0
// overflows in software timer mode, generates an interrupt every second.
// The interrupt handler records the current time and device temperature
// in a non-volatile log in FLASH. 
//
// With SYSCLK at 49.7664 MHZ, Timer 0 in mode 2 overflows exactly 4050 times 
// every second when clocked by SYSCLK/48. PCA0, clocked by Timer 0 overflows,
// is programmed to generate an interrupt every 4050 Timer 0 overflows, 
// or once every second.
//
// The 112,640 byte log cycles through all 4 code banks recording time and 
// temperature. Each data record is 8 bytes long. The log is capable of storing
// 14080 records over a time period of 3.9 hours. Once the log is full, it
// continues logging at the beginning of log, erasing the FLASH page with 
// the oldest data as it progresses.
//
// When this code is built, the linker generates two multiple call to segments
// warnings. These warnings are generated because the FLASH support routines
// are called from the main routine and from interrupts. These warnings have 
// been accounted for in the code by disabling interrupts before calling any
// FLASH support routines.
//
//
// Target: C8051F12x
// Tool chain: TASKING CC51 7.0 / TASKING EVAL CC51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "regc51f12x.sfr"              // SFR declarations
#include <stdio.h>                     // printf() and getchar()

/* SFR PAGE DEFINITIONS */

#define CONFIG_PAGE       0x0F    /* SYSTEM AND PORT CONFIGURATION PAGE */
#define LEGACY_PAGE       0x00    /* LEGACY SFR PAGE */
#define TIMER01_PAGE      0x00    /* TIMER 0 AND TIMER 1 */
#define CPT0_PAGE         0x01    /* COMPARATOR 0 */
#define CPT1_PAGE         0x02    /* COMPARATOR 1 */
#define UART0_PAGE        0x00    /* UART 0 */
#define UART1_PAGE        0x01    /* UART 1 */
#define SPI0_PAGE         0x00    /* SPI 0 */
#define EMI0_PAGE         0x00    /* EXTERNAL MEMORY INTERFACE */
#define ADC0_PAGE         0x00    /* ADC 0 */
#define ADC2_PAGE         0x02    /* ADC 2 */
#define SMB0_PAGE         0x00    /* SMBUS 0 */
#define TMR2_PAGE         0x00    /* TIMER 2 */
#define TMR3_PAGE         0x01    /* TIMER 3 */
#define TMR4_PAGE         0x02    /* TIMER 4 */
#define DAC0_PAGE         0x00    /* DAC 0 */
#define DAC1_PAGE         0x01    /* DAC 1 */
#define PCA0_PAGE         0x00    /* PCA 0 */
#define PLL0_PAGE         0x0F    /* PLL 0 */
#define MAC0_PAGE         0x03    /* MAC 0 */



typedef union UInt {                   // Byte addressable unsigned int
   unsigned int Int;
   unsigned char Char[2];
} UInt;

typedef union Long {                   // Byte addressable long
   long Long;
   unsigned int Int[2];
   unsigned char Char[4];
} Long;

typedef union ULong {                  // Byte addressable unsigned long
   unsigned long ULong;
   unsigned int Int[2];
   unsigned char Char[4];
} ULong;

typedef struct Record {                // LOG record structure
   char start;
   unsigned int hours;
   unsigned char minutes;
   unsigned char seconds;
   unsigned int ADC_result;
   char end;
} Record;
   
//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
#define TRUE         1
#define FALSE        0

#define EXTCLK       22118400          // External oscillator frequency in Hz
#define SYSCLK       49766400          // Output of PLL derived from 
                                       // (EXTCLK*9/4)
 
#define BAUDRATE     115200            // Baud rate of UART in bps
                                       // Note: The minimum standard baud rate
                                       // supported by the UART0_Init routine
                                       // in this file is 19,200 bps when
                                       // SYSCLK = 49.76MHz. 

#define SAMPLERATE   2000              // The ADC sampling rate in Hz

_sfrbit LED _atbit(P1, 6);             // LED='1' means ON
_sfrbit SW2 _atbit(P3, 7);             // SW2='0' means switch pressed

#define LOG_START 0x04000L             // Starting address of LOG
#define LOG_END   0x1F800L             // Last address in LOG + 1
#define RECORD_LEN 8                   // Record length in bytes
#define START_OF_RECORD ':'            // Start of Record symbol

#define FLASH_PAGESIZE 1024            // Number of bytes in each FLASH page

#define COBANK    0xF0                 // Bit mask for the high nibble of PSBANK

#define COBANK0   0x00                 // These macros define the bit mask values
#define COBANK1   0x10                 // for the PSBANK register used for
#define COBANK2   0x20                 // selecting COBANK. COBANK should always 
#define COBANK3   0x30                 // be cleared then OR-Equaled (|=) with 
                                       // the proper bit mask to avoid changing
                                       // the other bits in the PSBANK register


//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

unsigned char SECONDS = 0;             // global RTC seconds counter
unsigned char MINUTES = 0;             // global RTC minutes counter
unsigned int  HOURS = 0;               // global RTC hours counter

unsigned int ADC_RESULT = 0;           // holds the oversampled and averaged
                                       // result from ADC0               


bit LOG_FLAG = 0;                      // this flag is used to enable 
                                       // and disable logging but does
                                       // not affect the real-time clock

bit LOG_ERASED = 0;                    // this flag indicates that the 
                                       // LOG has been erased.  
//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void main(void);
void RTC_update(void);
void print_menu(void);

// initialization routines
void SYSCLK_Init(void);
void PORT_Init(void);
void UART0_Init (void);
void ADC0_Init (void);
void Timer3_Init(int counts);
void RTC_Init (void);
_interrupt(9) void PCA0_ISR (void);

// FLASH support routines
void FLASH_PageErase (unsigned long addr);
void FLASH_Write (unsigned long dest, char* src, unsigned int numbytes);
void FLASH_ByteWrite (unsigned long dest, char dat);
void FLASH_Read (char* dest, unsigned long src, unsigned int numbytes);
unsigned char FLASH_ByteRead (unsigned long addr);

// LOG support routines

void print_time(void);
void LOG_erase(void);
unsigned long find_current_record(void);
void LOG_print(char all_at_once);
void LOG_update(void);

// Get_Key function: Returns the keystroke
char Get_Key();


//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) 
{

   #define input_str_len 4             // buffer to hold characters entered
   char input_str[input_str_len];      // at the command prompt

   WDTCN = 0xde;                       // disable watchdog timer
   WDTCN = 0xad;

   PORT_Init ();                       // initialize crossbar and GPIO
   SYSCLK_Init ();                     // initialize oscillator
   UART0_Init ();                      // initialize UART0
   ADC0_Init();                        // initialize ADC0
   RTC_Init ();                        // initializes Timer0 and the PCA
   Timer3_Init(SYSCLK/SAMPLERATE);     // initialize Timer3 to overflow
                                       // and generate interrupts at 
                                       // <SAMPLERATE> Hz

                                       // to implement a real-time clock

   EA = 1;                             // enable global interrupts
   
   print_menu();                       // print the command menu
 
   
   while (1){

      SFRPAGE = UART0_PAGE;         
      printf("\nEnter a command > ");
      input_str[0] = Get_Key();
      putchar(input_str[0]);			// Echo keystroke
      putchar('\n');
      switch ( input_str[0] ){
                      
         case '1': LOG_FLAG = 1;
                   SFRPAGE = UART0_PAGE;
                   printf("\nLogging has now started.\n");
                   break;

         case '2': LOG_FLAG = 0;
                   SFRPAGE = UART0_PAGE;
                   printf("\nLogging has now stopped.\n");
                   break;

         case '3': LOG_FLAG = 0;
                   LOG_erase();
                   SFRPAGE = UART0_PAGE;
                   printf("\nThe log has been erased and logging is stopped.\n");
                   break;
         
         case '4': LOG_print(FALSE);
                   print_menu();
                   break;
         
         case '5': LOG_print(TRUE);
                   print_menu();
                   break;

         case '6': print_time();
                   break;

         case '?': print_menu(); 
                   break;  

         default:  if(input_str[0] != 0x03)
                   printf("\nIllegal Command.\n");
                   break;
      }   

   } // end while

}


//-----------------------------------------------------------------------------
// RTC_update
//-----------------------------------------------------------------------------
//
//
void RTC_update(void)
{
   SECONDS++;
   if (SECONDS == 60) {
      SECONDS = 0;
      MINUTES++;
      if (MINUTES == 60) {
         MINUTES = 0;
         HOURS++;
      }
   }

}

//-----------------------------------------------------------------------------
// FLASH Support Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// FLASH_PageErase
//-----------------------------------------------------------------------------
//
// This function erases the FLASH page containing <addr>.
//
void FLASH_PageErase (unsigned long addr)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char PSBANK_SAVE = PSBANK;          // Preserve current code bank
   bit EA_SAVE = EA;                   // Preserve interrupt state
   
   char _xdat *  pwrite;          // FLASH write/erase pointer

   ULong temp_addr;                    // Temporary ULong   
   
   temp_addr.ULong = addr;             // copy <addr> to a byte addressable
                                       // unsigned long

   // Extract address information from <addr>
   pwrite = (char _xdat *) temp_addr.Int[1];
   
   // Extract code bank information from <addr>
   PSBANK &= ~COBANK;                  // Clear the COBANK bits
   
   
   if( temp_addr.Char[1] == 0x00){     // If the address is less than
                                       // 0x10000, the Common area and 
      PSBANK |= COBANK1;               // Bank1 provide a 64KB linear 
                                       // address space   
   } else {                            // Else, Bank2 and Bank3 provide
                                       // a 64KB linear address space
      
      if (temp_addr.Char[2] & 0x80){   // If bit 15 of the address is 
                                       // a '1', then the operation should
         PSBANK |= COBANK3;            // target Bank3, else target Bank2
      
      } else { 
         
         PSBANK |= COBANK2;
         temp_addr.Char[2] |= 0x80;    
         pwrite = (char _xdat *) temp_addr.Int[1];
      }
    }     
                                        
   SFRPAGE = LEGACY_PAGE;
      
   EA = 0;                             // Disable interrupts                 
   FLSCL |= 0x01;                      // Enable FLASH writes/erases
   PSCTL = 0x03;                       // MOVX erases FLASH page

   *pwrite = 0;                        // Initiate FLASH page erase

   FLSCL &= 0xFE;                     // Disable FLASH writes/erases
   PSCTL = 0x00;                       // MOVX targets XRAM

   EA = EA_SAVE;                       // Restore interrupt state
   PSBANK = PSBANK_SAVE;               // Restore current code bank
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// FLASH_Write
//-----------------------------------------------------------------------------
//
// This routine copies <numbytes> from <src> to the FLASH addressed by <dest>.
//
void FLASH_Write (unsigned long dest, char* src, unsigned int numbytes)
{
   
   unsigned int i;                     // Software Counter
      
   for (i = 0; i < numbytes; i++) {
      
      FLASH_ByteWrite( dest++, *src++);
   }
}

//-----------------------------------------------------------------------------
// FLASH_ByteWrite
//-----------------------------------------------------------------------------
//
// This routine writes <dat> to the FLASH byte addressed by <dest>.
//
void FLASH_ByteWrite (unsigned long dest, char dat) 
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char PSBANK_SAVE = PSBANK;          // Preserve current code bank
   bit EA_SAVE = EA;                   // Preserve interrupt state
   
   ULong temp_dest;                    // Temporary ULong 
      
   char _xdat *  pwrite;          // FLASH write/erase pointer
   
   temp_dest.ULong = dest;             // copy <dest> to a byte
                                       // addressable unsigned long

   // Check if data byte being written is 0xFF
   // There is no need to write 0xFF to FLASH since erased
   // FLASH defaults to 0xFF.
   if(dat != 0xFF){

      // Extract address information from <dest>
      pwrite = (char _xdat *) temp_dest.Int[1];
   
      // Extract code bank information from <addr>
      PSBANK &= ~COBANK;               // Clear the COBANK bits
   
   
      if( temp_dest.Char[1] == 0x00){  // If the address is less than
                                       // 0x10000, the Common area and 
         PSBANK |= COBANK1;            // Bank1 provide a 64KB linear 
                                       // address space   
      } else {                         // Else, Bank2 and Bank3 provide
                                       // a 64KB linear address space
      
         if (temp_dest.Char[2] & 0x80){// If bit 15 of the address is 
                                       // a '1', then the operation should
            PSBANK |= COBANK3;         // target Bank3, else target Bank2
      
         } else { 
         
            PSBANK |= COBANK2;
            temp_dest.Char[2] |= 0x80;
            pwrite = (char _xdat *) temp_dest.Int[1];
         }
       }     
                                        
      
      SFRPAGE = LEGACY_PAGE;
      
      EA = 0;                          // Disable interrupts                 
      FLSCL |= 0x01;                   // Enable FLASH writes/erases
      PSCTL = 0x01;                    // MOVX writes FLASH byte

      *pwrite = dat;                   // Write FLASH byte

      FLSCL &= 0xFE;                  // Disable FLASH writes/erases
      PSCTL = 0x00;                    // MOVX targets XRAM
   }
      
   EA = EA_SAVE;                       // Restore interrupt state
   PSBANK = PSBANK_SAVE;               // Restore current code bank
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page      
}

//-----------------------------------------------------------------------------
// FLASH_Read
//-----------------------------------------------------------------------------
//
// This routine copies <numbytes> from FLASH addressed by <src> to <dest>.
//
void FLASH_Read ( char* dest, unsigned long src, unsigned int numbytes)
{
   
   unsigned int i;                     // Software Counter
      
   for (i = 0; i < numbytes; i++) {
   
      *dest++ = FLASH_ByteRead(src++);
   }
}

//-----------------------------------------------------------------------------
// FLASH_ByteRead
//-----------------------------------------------------------------------------
//
// This routine returns to the value of the FLASH byte addressed by <addr>.
//
unsigned char FLASH_ByteRead (unsigned long addr)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Preserve current SFR page
   char PSBANK_SAVE = PSBANK;          // Preserve current code bank

   ULong temp_addr;                    // Temporary ULong   
   char temp_char;                     // Temporary char

   char _rom *  pread;            // FLASH read pointer   

   temp_addr.ULong = addr;             // copy <addr> to a byte addressable
                                       // unsigned long

   // Extract address information from <addr>
   pread = (char _rom *) temp_addr.Int[1];
   
   // Extract code bank information from <addr>
   PSBANK &= ~COBANK;                  // Clear the COBANK bits
   
   
   if( temp_addr.Char[1] == 0x00){     // If the address is less than
                                       // 0x10000, the Common area and 
      PSBANK |= COBANK1;               // Bank1 provide a 64KB linear 
                                       // address space   
   } else {                            // Else, Bank2 and Bank3 provide
                                       // a 64KB linear address space
      
      if (temp_addr.Char[2] & 0x80){   // If bit 15 of the address is 
                                       // a '1', then the operation should
         PSBANK |= COBANK3;            // target Bank3, else target Bank2
      
      } else { 
         
         PSBANK |= COBANK2;
         temp_addr.Char[2] |= 0x80;
         pread = (char _rom *) temp_addr.Int[1];
      }
   }     
                                       
   temp_char = *pread;                 // Read FLASH byte

   PSBANK = PSBANK_SAVE;               // Restore current code bank
   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page      

   return temp_char;
}

//-----------------------------------------------------------------------------
// Support Routines
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// print_menu
//-----------------------------------------------------------------------------
//
// This routine uses prints the command menu to the UART.
//
void print_menu(void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = UART0_PAGE;
   printf("\n\nC8051F12x Data Logging Example\n");
   printf("---------------------------------------\n");
   printf("1. Start Logging\n");
   printf("2. Stop Logging\n");
   printf("3. Erase Log\n");
   printf("4. Print Log (one page at a time - Press CTRL+C to stop)\n");
   printf("5. Print Log (all at once - Press CTRL+C to stop)\n");
   printf("6. Print Elapsed Time Since Last Reset\n");
   printf("?. Print Command List\n");
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// print_time
//-----------------------------------------------------------------------------
//
// This routine uses prints the elapsed time since the last reset to the UART.
//
void print_time(void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   bit EA_SAVE = EA;                // Preserve interrupt state

   SFRPAGE = UART0_PAGE;
   EA = 0;
   printf("%05u:", HOURS);
   printf("%02u:", MINUTES);
   printf("%02u", SECONDS);

   EA = EA_SAVE;
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}
   
//-----------------------------------------------------------------------------
// find_current_record
//-----------------------------------------------------------------------------
//
//
unsigned long find_current_record(void)
{
   
   char SFRPAGE_SAVE = SFRPAGE;           // Save Current SFR page
   bit EA_SAVE = EA;                      // Preserve interrupt state

   unsigned long pRead = LOG_START;       // Pointer used to read from FLASH
   
   unsigned int i;                        // Software counter
   bit record_erased;                     // Temporary flag
   

   // Keep skipping records until an uninitialized record is found or
   // until the end of the log is reached
   while( pRead < LOG_END ){

      EA = 0;   
      // Skip all records that have been initialized
      if(FLASH_ByteRead(pRead) == START_OF_RECORD ){

         // increment pRead to the next record
         pRead += RECORD_LEN;
         EA = EA_SAVE;
         continue;
      }

      // Verify that the Record is uninitialized, otherwise keep 
      // searching for an uninitialized record
      record_erased = 1;
      for(i = 0; i < RECORD_LEN; i++){
         if( FLASH_ByteRead(pRead+i) != 0xFF ){
            record_erased = 0;
         } 
      }
      if(!record_erased){
         // increment pRead to the next record
         pRead += RECORD_LEN;
         EA = EA_SAVE;    
         continue;
      }      
      
      EA = EA_SAVE;

      // When this code is reached, <pRead> should point to the beginning
      // of an uninitialized (erased) record;
      SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
      return pRead;

   }
   
   // This code is reached only when there are no uninitialized records
   // in the LOG. Erase the first FLASH page in the log and return 
   // a pointer to the first record in the log.
   EA = 0;
   FLASH_PageErase(LOG_START);            // Erase the first page of the LOG
   EA = EA_SAVE;
   SFRPAGE = SFRPAGE_SAVE;                // Restore SFR page
   return LOG_START;
}

//-----------------------------------------------------------------------------
// LOG_erase
//-----------------------------------------------------------------------------
//
//
void LOG_erase(void)
{
   unsigned long pWrite = LOG_START;      // pointer used to write to FLASH
   bit EA_SAVE = EA;                      // save interrupt status                                
   
   // Keep erasing pages until <pWrite> reaches the end of the LOG. 
   while( pWrite < LOG_END ){
      
      EA = 0;   
      FLASH_PageErase(pWrite);
      EA = EA_SAVE;

      pWrite += FLASH_PAGESIZE;
      
   }
   
   LOG_ERASED = 1;                        // flag that LOG has been erased 
}

//-----------------------------------------------------------------------------
// LOG_print
//-----------------------------------------------------------------------------
//
//
void LOG_print(char all_at_once)
{
   char SFRPAGE_SAVE = SFRPAGE;           // Save Current SFR page
   char user_command;
   bit EA_SAVE = EA;                      // save interrupt status

   unsigned long pRead = LOG_START;       // Pointer used to read from FLASH 
   
   Record temp_rec;                       // Temporary record

   // Keep printing records until the end of the log is reached
   while( pRead < LOG_END ){
                           
      // Copy a record from at <pRead> from the LOG into the local
      // Record structure <temp_rec> 
      EA = 0;         
      FLASH_Read( (char*) &temp_rec, pRead, RECORD_LEN);
      EA = EA_SAVE;

      // Validate Record
      if(temp_rec.start != ':'){
         SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
         return;
      }

      // Print the Record
      SFRPAGE = UART0_PAGE;
            
      RI0 = 0;                            // Clear UART Receive flag
                                          // to later check for the 
                                          // user pressing CTRL+C            
     
      EA = 0;                             // disable interrupts

      printf("%05u:", temp_rec.hours); 
      printf("%02u:", temp_rec.minutes);
      printf("%02u:", temp_rec.seconds);
      printf("  ADC = 0x%04X\n", temp_rec.ADC_result);
              
      EA = EA_SAVE;                        // restore interrupts
                                           // any pending interrupts will 
                                           // be handled immediatly
     // check if we need to continue

     // if printing all data at once do not stop printing unless
     // the user presses CTRL+C, otherwise print 16 records and
     // then prompt user to press any key

     if(all_at_once){
        // Check if user has pressed CTRL+C                        
        if(RI0 && SBUF0 == 0x03){ 
           RI0 = 0;
           printf("\nLog print terminated.\n");
           SFRPAGE = SFRPAGE_SAVE;         // Restore SFR page
           return;              
        }
         
     // pause every 16 lines 
     } else if( (pRead & ((RECORD_LEN*16)-1)) == 0  &&
                pRead > (LOG_START + RECORD_LEN)) {       
        
        // wait for a key to be pressed then check if user has 
        // pressed CTRL+C (0x03)
        printf("\npress any key to continue\n");
        user_command = Get_Key();
        if(user_command == 0x03) {
           printf("\nLog print terminated.\n");
           SFRPAGE = SFRPAGE_SAVE;         // Restore SFR page
           return;
        }
     }
   
  
     // increment pRead to the next record
     pRead += RECORD_LEN;
         
     SFRPAGE = SFRPAGE_SAVE;               // Restore SFR page
   
  }

}


//-----------------------------------------------------------------------------
// LOG_update
//-----------------------------------------------------------------------------
//
//
void LOG_update(void)
{
   bit EA_SAVE = EA;                 // Preserve interrupt state
   Record temp_record;               // local LOG record structure
   
   static unsigned long pWrite = LOG_START; 
                                     // pointer used to write to the LOG 
   bit record_erased;                // temporary flag
   unsigned int i;                   // temporary integer
   
   // record the time and ADC reading in the LOG if logging is enabled
   if(LOG_FLAG){
      
      if(LOG_ERASED){
         pWrite = LOG_START;
         LOG_ERASED = 0;   
   
      } else {
         
         // find the current record if the record at pWrite is not erased
         record_erased = 1;
         for(i = 0; i < RECORD_LEN; i++){
            EA = 0;   
            if( FLASH_ByteRead(pWrite+i) != 0xFF ){
               record_erased = 0;
            } 
            EA = EA_SAVE;
         }
         if(!record_erased){
            pWrite = find_current_record();
         }      


         // build the temporary record
         temp_record.start = START_OF_RECORD;
         temp_record.hours = HOURS;
         temp_record.minutes = MINUTES;
         temp_record.seconds = SECONDS;
         temp_record.ADC_result = ADC_RESULT;
      
         // write the temporary record to FLASH
         EA = 0;   
         FLASH_Write( pWrite, (char*) &temp_record, RECORD_LEN);     
         EA = EA_SAVE;
         
         // increment record pointer
         pWrite += RECORD_LEN;      
      
         // if <pWrite> is past the end of the LOG, reset to the top
         if(pWrite >= LOG_END){
            pWrite = LOG_START;
         }      
      } // end else     
   } // end if(LOG_FLAG)
   
}


//-----------------------------------------------------------------------------
// Get_Key()
//-----------------------------------------------------------------------------
//
// This routine returns the keystroke as a char
//

char Get_Key()
{
   char c;

   while (!RI0);
   c = SBUF0;
   RI0 = 0;

   return (c);
}


//-----------------------------------------------------------------------------
// Initialization Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use an external 22.1184 MHz 
// crystal oscillator multiplied by a factor of 9/4 using the PLL as its 
// clock source. The resulting frequency is 22.1184 MHz * 9/4 = 49.7664 MHz
//
void SYSCLK_Init (void)
{
   int i;                           // delay counter
   
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = CONFIG_PAGE;           // set SFR page

   OSCXCN = 0x67;                   // start external oscillator with
                                    // 22.1184MHz crystal

   for (i=0; i < 256; i++) ;        // Wait for osc. to start up
   
   while (!(OSCXCN & 0x80)) ;       // Wait for crystal osc. to settle

   CLKSEL = 0x01;                   // Select the external osc. as
                                    // the SYSCLK source

   OSCICN = 0x00;                   // Disable the internal osc.
   
   //Turn on the PLL and increase the system clock by a factor of M/N = 9/4
   SFRPAGE = CONFIG_PAGE;
   
   PLL0CN  = 0x04;                  // Set PLL source as external osc.
   SFRPAGE = LEGACY_PAGE;
   FLSCL   = 0x10;                  // Set FLASH read time for 50MHz clk 
                                    // or less 
   SFRPAGE = CONFIG_PAGE;
   PLL0CN |= 0x01;                  // Enable Power to PLL
   PLL0DIV = 0x04;                  // Set Pre-divide value to N (N = 4)
   PLL0FLT = 0x01;                  // Set the PLL filter register for 
                                    // a reference clock from 19 - 30 MHz
                                    // and an output clock from 45 - 80 MHz 
   PLL0MUL = 0x09;                  // Multiply SYSCLK by M (M = 9)
   
   for (i=0; i < 256; i++) ;        // Wait at least 5us
   PLL0CN  |= 0x02;                 // Enable the PLL
   while(!(PLL0CN & 0x10));         // Wait until PLL frequency is locked
   CLKSEL  = 0x02;                  // Select PLL as SYSCLK source

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// This routine configures the Crossbar and GPIO ports.
//
void PORT_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
      
   SFRPAGE = CONFIG_PAGE;           // set SFR page

   XBR0     = 0x04;                 // Enable UART0
   XBR1     = 0x00;
   XBR2     = 0x40;                 // Enable crossbar and weak pull-up
                                    

   P0MDOUT |= 0x01;                 // Set TX0 pin to push-pull
   P1MDOUT |= 0x40;                 // Set P1.6(LED) to push-pull
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Configure the UART0 using Timer1, for <baudrate> and 8-N-1. In order to
// increase the clocking flexibility of Timer0, Timer1 is configured to count
// SYSCLKs. 
// 
// To use this routine SYSCLK/BAUDRATE/16 must be less than 256. For example,
// if SYSCLK = 50 MHz, the lowest standard baud rate supported by this
// routine is 19,200 bps. 
//
void UART0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = UART0_PAGE;

   SCON0  = 0x50;                   // SCON0: mode 0, 8-bit UART, enable RX
   SSTA0  = 0x10;                   // Timer 1 generates UART0 baud rate and
                                    // UART0 baud rate divide by two disabled
   SFRPAGE = TIMER01_PAGE;
   TMOD   &= ~0xF0;                  
   TMOD   |=  0x20;                 // TMOD: timer 1, mode 2, 8-bit reload
   
   TH1 = -(SYSCLK/BAUDRATE/16);     // Set the Timer1 reload value
                                    // When using a low baud rate, this equation
                                    // should be checked to ensure that the
                                    // reload value will fit in 8-bits.
             
   CKCON |= 0x10;                   // T1M = 1; SCA1:0 = xx
   

   TL1 = TH1;                       // initialize Timer1
   TR1 = 1;                         // start Timer1  
   
   SFRPAGE = UART0_PAGE;
   TI0 = 1;                         // Indicate TX0 ready
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Configure ADC0 to start conversions on Timer3 Overflows and to
// use left-justified output mode. 
//
void ADC0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = ADC0_PAGE;

   ADC0CN = 0x85;                   // ADC0 enabled; normal tracking
                                    // mode; ADC0 conversions are initiated
                                    // on Timer3 overflows; ADC0 data is
                                    // left-justified

   REF0CN = 0x07;                   // enable temp sensor, on-chip VREF,
                                    // and VREF output buffer
   AMX0SL = 0x0F;                   // Select TEMP sens as ADC mux output

   ADC0CF = ((SYSCLK/2500000) << 3);// ADC conversion clock = 2.5MHz

   ADC0CF |= 0x01;                  // PGA gain = 2

   EIE2 |= 0x02;                    // Enable ADC0 End-of-conversion 
                                    // interrupts
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
// This routine initializes Timer3 in auto-reload mode to overflow
// at intervals specified in <counts>.
//
void Timer3_Init ( int counts)
{
   SFRPAGE = TMR3_PAGE;
   
   TMR3CN = 0;                       // STOP timer; set to auto-reload mode
   TMR3CF = 0x08;                    // Timer3 counts SYSCLKs
   RCAP3L = -counts;
   RCAP3H = (-counts) >> 8;			// Set reload value
   TMR3L = RCAP3L;
   TMR3H = RCAP3H;                    // Initialize Timer to reload value
   TR3 = 1;                          // start Timer3
      
}

//-----------------------------------------------------------------------------
// RTC_Init
//-----------------------------------------------------------------------------
//
// This Routine initializes Timer0 and PCA0 to implement a real-time clock.
// Assuming <SYSCLK> is generated from a 22.1184 crystal oscillator, Timer0
// overflows exactly 1800 times per second when configured as an 8-bit timer 
// that uses <SYSCLK>/48 as its timebase. PCA0 is configured to count 
// Timer0 overflows and interrupt every 1800 Timer0 overflows, or every second.
// The PCA0 ISR updates a set of global RTC counters for seconds, minutes, hours,
// and days. 
//
void RTC_Init(void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = TIMER01_PAGE;

   // configure Timer0 in Mode2: 8-bit Timer with Auto-Reload
   TMOD &= 0xF0;                   // Clear Timer0 bits
   TMOD |= 0x02;                    // Mode2 Auto-Reload
   
   // configure Timer0 timebase to <SYSCLK>/48
   CKCON &= 0xF0;                  // Clear bits
   CKCON |= 0x02;                   // Set Timer0 timebase  
   
   // configure PCA0 to count Timer0 overflows 
   PCA0MD = 0x04;
   
   // configure capture/compare module 0 to generate an interrupt when
   // the value of PCA0 reaches 4050 (0x0FD2)  
   //PCA0CP0 = 4050
   PCA0CPH0 = 0x0F;
   PCA0CPL0 = 0xD2;					// Set the value to match
     
   PCA0CPM0 &= ~0xFF;               // Clear bits
   PCA0CPM0 |= 0x49;                // Generate an interrupt when the 
                                    // PCA0 value matches PCA0CP0

   EIE1 |= 0x08;                    // Enable PCA0 interrupts
   
     
   TR0 = 1;                         // Start Timer0
   PCA0CN |= 0x40;                  // Enable PCA0

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// PCA0_ISR
//-----------------------------------------------------------------------------
//
//
_interrupt(9) void PCA0_ISR (void)
{
   
  if (CCF0) {
      CCF0 = 0;                         // clear Module0 capture/compare flag
      
      PCA0L = 0x00;
      PCA0H = 0x00;                     // clear the PCA counter
      RTC_update();                     // update RTC variables
      LOG_update();                     // update LOG if logging is enabled

  } else 
 
  if (CCF1) {
      CCF1 = 0;                         // clear Module1 capture/compare flag
  } else

  if (CCF2) {
      CCF2 = 0;                         // clear Module2 capture/compare flag
  } else

  if (CCF3) {
      CCF3 = 0;                         // clear Module3 capture/compare flag
  } else

  if (CCF4) {
      CCF4 = 0;                         // clear Module4 capture/compare flag
  } else
   
  if (CCF5) {
      CCF5 = 0;                         // clear Module5 capture/compare flag
  } else
   
  if (CF) {
      CF = 0;                           // clear PCA counter overflow flag
  }
  
}

//-----------------------------------------------------------------------------
// ADC0_ISR
//-----------------------------------------------------------------------------
//
// This ISR is called on the end of an ADC0 conversion.
//
_interrupt(15) void ADC0_ISR (void)
{
 
   Long result = {0};              // byte addressable long variable
   int i;
   bit EA_SAVE = EA;
   //accumulate 256 temperature samples
   result.Long += ADC0H;
   result.Long += ADC0L;
   i++;
   
   if( i == 256 ) {

      i = 0;

      // take the average (Divide by 256 = shift right by 8)
      // Do this operation "result.Long >>= 8;" (170 SYSCLK cycles) using
      // three MOV instructions (9 SYSCLK cycles)
      // Assume Most Significant Byte only contains sign information
   
      result.Char[3] = result.Char[2];
      result.Char[2] = result.Char[1];
      result.Char[1] = result.Char[0];
      
      // update global <ADC_RESULT>
      ADC_RESULT = result.Int[1];
   }

}
