//-----------------------------------------------------------------------------
// common.c
//-----------------------------------------------------------------------------
// Copyright 2003 Cygnal Integrated Products, Inc.
//
// AUTH: FB, JM
// DATE: 19 AUG 03 
//
// This example shows how to set up a code banking project using the Cygnal
// IDE and the TASKING 8051 development tools. It uses Timer3 and Timer4 
// interrupts to blink the LED and output a 1 kHz sine wave on DAC1,
// respectively. The code that blinks the LED is located in Bank 3 and the 
// code that outputs a sine wave based on a 256 entry sine table is located
// in Bank 2. Since interrupts must be located in the Common area, both 
// interrupts call a function in one of the banks to perform the desired task.
// 
// The project should be configured for code banking as shown in AN043 before
// this project is built.
// 
// This program uses the the 24.5 MHz internal oscillator multiplied by two 
// for an effective SYSCLK of 49 MHz.
//
// In "Project->Tool Chain Integration," the Command line flags under the
// Linker tab should read:
//
// -lc51s NOCASE RS(256) -banks 4 -common 8000H "BA(3, BANK3_TOGGLE_LED_PR)
// BA(2,BANK2_SET_DAC1_PR)" 
//
// Target: C8051F12x
// Tool chain: TASKING CC51 7.0 / TASKING EVAL CC51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "regc51f12x.sfr"               // SFR declarations
#include <stdio.h>                     // printf() and getchar()

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
#define TRUE         1
#define FALSE        0

#define SYSCLK       49000000          // Output of PLL derived from (INTCLK*2)
#define SAMPLE_RATE_DAC 100000L        // DAC sampling rate in Hz
#define PHASE_PRECISION 65536          // range of phase accumulator
#define FREQUENCY 1000                 // frequency of output waveform in Hz

                                       // <phase_add> is the change in phase
                                       // between DAC1 samples; It is used in 
                                       // the set_DAC1 routine in bank2
unsigned int phase_add = FREQUENCY * PHASE_PRECISION / SAMPLE_RATE_DAC;

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

// Common area functions 
void main(void);
void SYSCLK_Init(void);
void PORT_Init(void);
void DAC1_Init (void);
void Timer3_Init(int counts);
void Timer4_Init(int counts);
_interrupt(14) void Timer3_ISR (void);
_interrupt(16) void Timer4_ISR (void);

// code bank 2 functions
extern void set_DAC1(void);

// code bank 3 functions
extern void toggle_LED(void);

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
//
//Common area code;
//
void main (void) 
{

   WDTCN = 0xde;                       // disable watchdog timer
   WDTCN = 0xad;
   PORT_Init ();                       // initialize crossbar and GPIO
   SYSCLK_Init ();                     // initialize oscillator
   DAC1_Init ();                       // initialize DAC1   

   Timer3_Init(SYSCLK/12/1000);        // initialize Timer3 to overflow
                                       // every millisecond

   Timer4_Init(SYSCLK/SAMPLE_RATE_DAC);// initialize Timer4 to overflow 
                                       // <SAMPLE_RATE_DAC> times per
                                       // second
   EA = 1;                             // enable global interrupts
    while(1);

} 

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Timer3_ISR
//-----------------------------------------------------------------------------
// This routine changes the state of the LED whenever Timer3 overflows 250 times.
//
// NOTE: The SFRPAGE register will automatically be switched to the Timer 3 Page
// When an interrupt occurs.  SFRPAGE will return to its previous setting on exit
// from this routine.
//
_interrupt(14) void Timer3_ISR (void)
{
   static int i;                       // software interrupt counter

   TF3 = 0;                            // clear Timer3 overflow flag  
   i++;                                // increment software counter   

   // toggle the LED every 250ms
   if (i >= 250) {   
      toggle_LED();                    // toggle the green LED
	  i = 0;                           // clear software counter
   }
}
//-----------------------------------------------------------------------------
// Timer4_ISR -- Wave Generator
//-----------------------------------------------------------------------------
//
// This ISR is called on Timer4 overflows.  Timer4 is set to auto-reload mode
// and is used to schedule the DAC output sample rate in this example.
// Note that the value that is written to DAC1 during this ISR call is
// actually transferred to DAC1 at the next Timer4 overflow.
//
_interrupt(16) void Timer4_ISR (void)
{
  TF4 = 0;                             // clear Timer4 overflow flag 
  set_DAC1();
}


//-----------------------------------------------------------------------------
// Initialization Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal oscillator
// at 24.5 MHz multiplied by two using the PLL.
//
void SYSCLK_Init (void)
{
   int i;                           // software timer

   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = 0x0F;					// set SFR page to CONFIG_PAGE

   OSCICN = 0x83;                   // set internal oscillator to run
                                    // at its maximum frequency

   CLKSEL = 0x00;                   // Select the internal osc. as
                                    // the SYSCLK source
   
   //Turn on the PLL and increase the system clock by a factor of M/N = 2
   SFRPAGE = 0x0F;					// Set SFR page to CONFIG_PAGE
   
   PLL0CN  = 0x00;                  // Set internal osc. as PLL source
   SFRPAGE = 0x00;					// Set SFR page to LEGACY_PAGE
   FLSCL   = 0x10;                  // Set FLASH read time for 50MHz clk
                                    // or less
   SFRPAGE = 0x0F;					// Set SFR page to CONFIG_PAGE
   PLL0CN |= 0x01;                  // Enable Power to PLL
   PLL0DIV = 0x01;                  // Set Pre-divide value to N (N = 1)
   PLL0FLT = 0x01;                  // Set the PLL filter register for 
                                    // a reference clock from 19 - 30 MHz
                                    // and an output clock from 45 - 80 MHz 
   PLL0MUL = 0x02;                  // Multiply SYSCLK by M (M = 2)
   
   for (i=0; i < 256; i++) ;        // Wait at least 5us
   PLL0CN  |= 0x02;                 // Enable the PLL
   while(!(PLL0CN & 0x10));         // Wait until PLL frequency is locked
   CLKSEL  = 0x02;                  // Select PLL as SYSCLK source

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}


//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// This routine configures the crossbar and GPIO ports.
//
void PORT_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page
   
   SFRPAGE = 0x0F;					// Set SFR page to CONFIG_PAGE

   XBR0     = 0x00;                 
   XBR1     = 0x00;
   XBR2     = 0x40;                 // Enable crossbar and weak pull-up

   P1MDOUT |= 0x40;                 // Set P1.6(LED) to push-pull
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// DAC1_Init
//-----------------------------------------------------------------------------
//
// Configure DAC1 to update on Timer4 overflows and enable the the VREF buffer.
//
//
void DAC1_Init(void){

   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = 0x01;					// Set SFR page to DAC1_PAGE
   
   DAC1CN = 0x94;                   // Enable DAC1 in left-justified mode
                                    // managed by Timer4 overflows
   SFRPAGE = 0x00;					// Set SFR page to LEGACY_PAGE

   REF0CN |= 0x03;                  // Enable the internal VREF (2.4v) and
                                    // the Bias Generator

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page

}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
//
// Configure Timer3 to auto-reload mode and to generate interrupts 
// at intervals specified by <counts> using SYSCLK/12 as its time base.
//
//
void Timer3_Init (int counts)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = 0x01;					// Set SFR page to TMR3_PAGE

   TMR3CN = 0x00;                   // Stop Timer; Clear overflow flag;
                                    // Set to Auto-Reload Mode

   TMR3CF = 0x00;                   // Configure Timer to increment;
                                    // Timer counts SYSCLKs/12
   
   RCAP3L = -counts;
   RCAP3H = (-counts) >> 8;			// Set reload value
   TMR3L = RCAP3L;
   TMR3H = RCAP3H;                    // Initialize Timer to reload value

   EIE2   |= 0x01;                  // enable Timer3 interrupts
   TR3 = 1;                         // start Timer
   
   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page

}

//-----------------------------------------------------------------------------
// Timer4_Init
//-----------------------------------------------------------------------------
// Configure Timer4 to auto-reload mode and to generate interrupts
// at intervals specified in <counts> using SYSCLK as its time base.
//
void Timer4_Init (int counts)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = 0x02;					// Set SFR page to TMR4_PAGE
   
   TMR4CN = 0x00;                   // Stop Timer4; Clear overflow flag (TF4);
                                    // Set to Auto-Reload Mode

   TMR4CF = 0x08;                   // Configure Timer4 to increment;
                                    // Timer4 counts SYSCLKs

   RCAP4L = -counts;
   RCAP4H = (-counts) >> 8;                 // Set reload value
   TMR4L = RCAP4L;
   TMR4H = RCAP4H;                    // Initialzie Timer4 to reload value

   EIE2 |= 0x04;                    // enable Timer4 interrupts
   TR4 = 1;                         // start Timer4

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}