//-----------------------------------------------------------------------------
// MagStripeReaderF330_3CH.c
//-----------------------------------------------------------------------------
// Copyright 2008 Silicon Laboratories
//
// AUTH: BD
// DATE: 15 AUG 08
// VER:  2.1
//
// This program reads the magnetic stripe from a card written in the standard
// ISO 3-channel format using F2F encoding.  Read data is output to the UART
// after being decoded.
//
// Target: C8051F33x
// Tool chain: KEIL C51 7.06 / KEIL EVAL C51
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <c8051f330.h>                    // SFR declarations for C8051F330

//-----------------------------------------------------------------------------
// 16-bit SFR Definitions for 'F33x
//-----------------------------------------------------------------------------
sfr16 TMR2RL   = 0xca;                    // Timer2 reload value
sfr16 TMR2     = 0xcc;                    // Timer2 counter


//-----------------------------------------------------------------------------
// Conditional Compilation CONSTANTS
//-----------------------------------------------------------------------------

#define DEBUG        0                    // Set to '1' for extra information
#define T3_5BIT      1                    // Set to '1' for T3 5-bit encoding
                                          // Set to '0' for T3 7-bit encoding

// **NOTE** The Track 3 encoding scheme is different for different card types
//          The ISO-4909 standard uses 5-bit Track 3 encoding
//          Many cards now use 7-bit encoding for Track 3


//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

#define SYSCLK       24500000             // SYSCLK frequency in Hz
#define BAUDRATE     115200               // Baud rate of UART in bps
#define SAMPLE_RATE  200000               // Sample rate of ADC

#define T1_SS        0x45                 // Start Sentinel + parity
#define T1_ES        0x1F                 // End Sentinel + parity
#define T1_BITS      7                    // data + parity bit
#define T1_CHPOS     0x08                 // Positive ADC Mux channel
#define T1_CHNEG     0x09                 // Negative ADC Mux channel

#define T2_SS        0x0B                 // Start Sentinel + parity
#define T2_ES        0x1F                 // End Sentinel + parity
#define T2_BITS      5                    // data + parity bit
#define T2_CHPOS     0x0A                 // Positive ADC Mux channel
#define T2_CHNEG     0x0B                 // Negative ADC Mux channel

#if T3_5BIT                               // Use 5-bit encoding on Track 3
#define T3_SS        0x0B                 // Start Sentinel + parity
#define T3_ES        0x1F                 // End Sentinel + parity
#define T3_BITS      5                    // data + parity bit
#endif

#if !T3_5BIT                              // Use 7-bit encoding on Track 3
#define T3_SS        0x45                 // Start Sentinel + parity
#define T3_ES        0x1F                 // End Sentinel + parity
#define T3_BITS      7                    // data + parity bit
#endif

#define T3_CHPOS     0x0C                 // Positive ADC Mux channel
#define T3_CHNEG     0x0D                 // Negative ADC Mux channel

#define THRESHOLD1   7                    // Noise threshold limits
#define THRESHOLD2   7
#define THRESHOLD3   7

#define Z_LIMIT      3                    // Number of Zeros before recording

sbit TK1_GRN_LED   = P0^2;                // GREEN LED TK1
sbit TK1_RED_LED   = P0^3;                // RED LED TK1

sbit TK2_GRN_LED   = P0^6;                // GREEN LED TK2
sbit TK2_RED_LED   = P0^7;                // RED LED TK2

sbit TK3_GRN_LED   = P1^6;                // GREEN LED TK3
sbit TK3_RED_LED   = P1^7;                // RED LED TK3


//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

unsigned char xdata T1RAW[100], T2RAW[100],     // Track 1 and 2 Raw Data
                                T3RAW[100];     // Track 3 Raw Data
unsigned char xdata ASCII_array[128];           // Decoded Information
unsigned char COLLECTED1 = 1, COLLECTED2 = 1,   // Raw data indices
                              COLLECTED3 = 1;

unsigned int bdata Timeout_Counter;       // Bit-Addressable Timeout counter
sbit CLEAR_TIMEOUT = Timeout_Counter ^ 4; // Used to keep from timing out
sbit READ_TIMEOUT = Timeout_Counter ^ 5;  // Indicates when read is finished
sbit CH2_SWITCH = Timeout_Counter ^ 8;    // LSB of counter:
                                          //   If '1', CH2 is sampled
                                          //   If '0', CH2 is skipped

unsigned char bdata Temp_Byte1;           // Bit-Addressable Temporary Storage
sbit Temp1_b0 = Temp_Byte1 ^ 0;           // LSB of Temp_Byte1

unsigned char bdata Temp_Byte2;           // Bit-Addressable Temporary Storage
sbit Temp2_b0 = Temp_Byte2 ^ 0;           // LSB of Temp_Byte2

unsigned char bdata Temp_Byte3;           // Bit-Addressable Temporary Storage
sbit Temp3_b0 = Temp_Byte3 ^ 0;           // LSB of Temp_Byte3


//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);
void ADC0_Init (void);
void UART0_Init (void);
void PORT_Init (void);
void Timer2_Init (int);

unsigned char Swipe_Card(void);

char GetDirection (unsigned char maxindex, unsigned char StartSen,
                  unsigned char EndSen, unsigned char *TrackRAW,
                  unsigned char CharBits);
char DecodeTrackForward (unsigned char maxindex, unsigned char Byte_Offset,
                  unsigned char Bit_Offset, unsigned char *TrackRAW,
                  unsigned char CharBits);

char DecodeTrackBackward (unsigned char Byte_Offset, unsigned char Bit_Offset,
                  unsigned char *TrackRAW, unsigned char CharBits);

char TrackErrorCheck (unsigned char maxindex, unsigned char StartSen,
                  unsigned char EndSen, unsigned char CharBits);

void UART_CharOut (unsigned char c);
void UART_StringOut (unsigned char *c);
void UART_HexOut (unsigned char c);


//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------

void main (void) {

unsigned char idata Return_Code;
unsigned char idata colCount;
bit ERRT1, ERRT2, ERRT3;                  // Track 1, 2, 3 Error Flags

   // Disable Watchdog timer
   PCA0MD &= ~0x40;                       // WDTE = 0 (clear watchdog timer
                                          // enable)
   PORT_Init();                           // Initialize Port I/O
   SYSCLK_Init ();                        // Initialize Oscillator

   ADC0_Init ();                          // Init ADC0
   Timer2_Init(SYSCLK/SAMPLE_RATE);       // Init Timer 2 w/ ADC sample rate
   UART0_Init();

   while (1) {

#if DEBUG
      UART_StringOut("\nTesting");
#endif   // END #if DEBUG

      UART_StringOut("\n");

      Swipe_Card();

// If DEBUG is '1', use verbose mode for output
#if DEBUG

      // Find direction of track1, and decode to character array
      Return_Code = GetDirection(COLLECTED1, T1_SS, T1_ES, T1RAW, T1_BITS);

      if ((Return_Code & 0x80) == 0)               // If no error was detected
      {
         // Check character array for SS, ES, Parity, and LRC
         Return_Code = TrackErrorCheck(Return_Code, T1_SS, T1_ES, T1_BITS);
      }

      if (Return_Code & 0x80)                      // If an error was detected
      {
         ERRT1 = 1;
         UART_StringOut("\nErrors: \n");           // List the errors detected

         if (Return_Code & 0x01)
         {
            UART_StringOut("\tStart Sentinel not found\n");
         }
         if (Return_Code & 0x02)
         {
            UART_StringOut("\tEnd Sentinel not found\n");
         }
         if (Return_Code & 0x04)
         {
            UART_StringOut("\tLRC incorrect\n");
         }
         if (Return_Code & 0x08)
         {
            UART_StringOut("\tParity error(s)\n");
         }

         UART_StringOut("\nDATA CH1:\n");
         for (colCount = 0; colCount < 128; colCount++)
         {
            UART_CharOut(0x20 + (ASCII_array[colCount]&0x3F));
            UART_CharOut(0x30);
         }
         UART_CharOut('\n');
         UART_StringOut("END DATA CH1\n");
      }
      else                                         // No errors, print T1 data
      {
         ERRT1 = 0;
         UART_StringOut("\nDATA CH1:\n");

         for (colCount = 0; colCount < Return_Code; colCount++)
         {
            UART_CharOut(0x20 + (ASCII_array[colCount]&0x3F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
         UART_StringOut("END DATA CH1\n");
      }

      // Print the RAW data for Track 1
      UART_StringOut("\nRAW COLLECTION CH1:\n0x");

      for (colCount = 0; colCount < COLLECTED1; colCount++)
      {
         UART_HexOut (T1RAW[colCount]);
      }

      UART_CharOut('\n');
      UART_StringOut("END RAW CH1\n");

      // Find direction of track2, and decode to character array
      Return_Code = GetDirection(COLLECTED2, T2_SS, T2_ES, T2RAW, T2_BITS);

      if ((Return_Code & 0x80) == 0)               // If no error was detected
      {
         // Check character array for SS, ES, Parity, and LRC
         Return_Code = TrackErrorCheck(Return_Code, T2_SS, T2_ES, T2_BITS);
      }

      if (Return_Code & 0x80)                      // If an error was detected
      {
         ERRT2 = 1;
         UART_StringOut("\nErrors: \n");           // List the errors detected

         if (Return_Code & 0x01)
         {
            UART_StringOut("\tStart Sentinel not found\n");
         }
         if (Return_Code & 0x02)
         {
            UART_StringOut("\tEnd Sentinel not found\n");
         }
         if (Return_Code & 0x04)
         {
            UART_StringOut("\tLRC incorrect\n");
         }
         if (Return_Code & 0x08)
         {
            UART_StringOut("\tParity error(s)\n");
         }

         UART_StringOut("\nDATA CH2:\n");
         for (colCount = 0; colCount < 128; colCount++)
         {
            UART_CharOut(0x30 + (ASCII_array[colCount]&0x0F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
         UART_StringOut("END DATA CH2\n");
      }
      else                                         // No errors, print T2 data
      {
         ERRT2 = 0;
         UART_StringOut("\nDATA CH2:\n");

         for (colCount = 0; colCount < Return_Code; colCount++)
         {
            UART_CharOut(0x30 + (ASCII_array[colCount]&0x0F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
         UART_StringOut("END DATA CH2\n");
      }

      // Print the RAW data for Track 2
      UART_StringOut("\nRAW COLLECTION CH2:\n0x");

      for (colCount = 0; colCount < COLLECTED2; colCount++)
      {
         UART_HexOut (T2RAW[colCount]);
      }

      UART_CharOut('\n');
      UART_StringOut("END RAW CH2\n");

      // Find direction of track3, and decode to character array
      Return_Code = GetDirection(COLLECTED3, T3_SS, T3_ES, T3RAW, T3_BITS);

      if ((Return_Code & 0x80) == 0)               // If no error was detected
      {
         // Check character array for SS, ES, Parity, and LRC
         Return_Code = TrackErrorCheck(Return_Code, T3_SS, T3_ES, T3_BITS);
      }

      if (Return_Code & 0x80)                      // If an error was detected
      {
         ERRT3 = 1;
         UART_StringOut("\nErrors: ");             // List the errors detected
         UART_CharOut('\n');

         if (Return_Code & 0x01)
         {
            UART_StringOut("\tStart Sentinel not found\n");
         }
         if (Return_Code & 0x02)
         {
            UART_StringOut("\tEnd Sentinel not found\n");
         }
         if (Return_Code & 0x04)
         {
            UART_StringOut("\tLRC incorrect\n");
         }
         if (Return_Code & 0x08)
         {
            UART_StringOut("\tParity error(s)\n");
         }

         UART_StringOut("\nDATA CH3:\n");
         for (colCount = 0; colCount < 128; colCount++)
         {
            UART_CharOut(0x30 + (ASCII_array[colCount]&0x0F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
         UART_StringOut("END DATA CH3\n");
      }
      else                                         // No errors, print T3 data
      {
         ERRT3 = 0;
         UART_StringOut("\nDATA CH3:\n");

         for (colCount = 0; colCount < Return_Code; colCount++)
         {
            UART_CharOut(0x30 + (ASCII_array[colCount]&0x0F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
         UART_StringOut("END DATA CH3\n");
      }

      // Print the RAW data for Track 3
      UART_StringOut("\nRAW COLLECTION CH3:\n0x");

      for (colCount = 0; colCount < COLLECTED3; colCount++)
      {
         UART_HexOut (T3RAW[colCount]);
      }

      UART_CharOut('\n');
      UART_StringOut("END RAW CH3\n");


      // Signal Error / OK with LEDs
      if (!ERRT1)
      {
         TK1_RED_LED = 0;
         TK1_GRN_LED = 1;
      }
      else
      {
         TK1_RED_LED = 1;
         TK1_GRN_LED = 0;
      }      
      
      // Signal Error / OK with LEDs
      if (!ERRT2)
      {
         TK2_RED_LED = 0;
         TK2_GRN_LED = 1;
      }
      else
      {
         TK2_RED_LED = 1;
         TK2_GRN_LED = 0;
      }

      // Signal Error / OK with LEDs
      if (!ERRT3)
      {
         TK3_RED_LED = 0;
         TK3_GRN_LED = 1;
      }
      else
      {
         TK3_RED_LED = 1;
         TK3_GRN_LED = 0;
      }

#endif // END #if DEBUG

// If DEBUG is '0', only output valid track info
#if !DEBUG

      // Find direction of track1, and decode to character array
      Return_Code = GetDirection(COLLECTED1, T1_SS, T1_ES, T1RAW, T1_BITS);

      if ((Return_Code & 0x80) == 0)               // If no error was detected
      {
         // Check character array for SS, ES, Parity, and LRC
         Return_Code = TrackErrorCheck(Return_Code, T1_SS, T1_ES, T1_BITS);
      }

      if (Return_Code & 0x80)                      // If an error was detected
      {                                            // set the error bit
         ERRT1 = 1;
      }
      else                                         // Otherwise print Track 1
      {
         ERRT1 = 0;
         for (colCount = 0; colCount < Return_Code; colCount++)
         {
            UART_CharOut(0x20 + (ASCII_array[colCount]&0x3F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
      }


      // Find direction of track2, and decode to character array
      Return_Code = GetDirection(COLLECTED2, T2_SS, T2_ES, T2RAW, T2_BITS);

      if ((Return_Code & 0x80) == 0)               // If no error was detected
      {
         // Check character array for SS, ES, Parity, and LRC
         Return_Code = TrackErrorCheck(Return_Code, T2_SS, T2_ES, T2_BITS);
      }

      if (Return_Code & 0x80)                      // If an error was detected
      {                                            // set the error bit
         ERRT2 = 1;
      }
      else                                         // Otherwise print Track 2
      {
         ERRT2 = 0;
         for (colCount = 0; colCount < Return_Code; colCount++)
         {
            UART_CharOut(0x30 + (ASCII_array[colCount]&0x0F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
      }

      // Find direction of track3, and decode to character array
      Return_Code = GetDirection(COLLECTED3, T3_SS, T3_ES, T3RAW, T3_BITS);

      if ((Return_Code & 0x80) == 0)               // If no error was detected
      {
         // Check character array for SS, ES, Parity, and LRC
         Return_Code = TrackErrorCheck(Return_Code, T3_SS, T3_ES, T3_BITS);
      }

      if (Return_Code & 0x80)                      // If an error was detected
      {                                            // set the error bit
         ERRT3 = 1;
      }
      else                                         // Otherwise print Track 3
      {
         ERRT3 = 0;
         for (colCount = 0; colCount < Return_Code; colCount++)
         {
            UART_CharOut(0x30 + (ASCII_array[colCount]&0x0F));
            ASCII_array[colCount] = 0x30;
         }
         UART_CharOut('\n');
      }

      // Signal Error / OK with LEDs
      if (!ERRT1)
      {
         TK1_RED_LED = 0;
         TK1_GRN_LED = 1;
      }
      else
      {
         TK1_RED_LED = 1;
         TK1_GRN_LED = 0;
      }      
      
      // Signal Error / OK with LEDs
      if (!ERRT2)
      {
         TK2_RED_LED = 0;
         TK2_GRN_LED = 1;
      }
      else
      {
         TK2_RED_LED = 1;
         TK2_GRN_LED = 0;
      }

      // Signal Error / OK with LEDs
      if (!ERRT3)
      {
         TK3_RED_LED = 0;
         TK3_GRN_LED = 1;
      }
      else
      {
         TK3_RED_LED = 1;
         TK3_GRN_LED = 0;
      }
#endif // END #if !DEBUG

   }  // END while(1)
}  // END main()

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Configure the Crossbar and GPIO ports.
//
// P0.0 - VREF Input    (analog, skipped)
// P0.4 - UART TX       (push-pull)
// P0.5 - UART RX       (open drain)
// P1.0 - Channel 1+    (analog, skipped)
// P1.1 - Channel 1-    (analog, skipped)
// P1.2 - Channel 2+    (analog, skipped)
// P1.3 - Channel 2-    (analog, skipped)
// P1.4 - Channel 3+    (analog, skipped)
// P1.5 - Channel 3-    (analog, skipped)
// P1.6 - Green LED     (push-pull, skipped)
// P1.7 - Red LED       (push-pull, skipped)
//

void PORT_Init (void)
{
   P0MDOUT |= 0xDC;                    // enable TX and LEDs as push-pull out

   P0MDIN &= ~0x01;                    // VREF analog in
   P1MDIN &= ~0x3F;                    // Enable P1.0 through 1.5 as analog in
   P1MDOUT |= 0xC0;                    // P1.6, 1.7 Push-pull output

   P0SKIP |= 0xCD;                     // Skip VREF pin and LED Outputs
   P1SKIP |= 0xFF;                     // Skip Analog Inputs and LED Outputs

   XBR0    = 0x01;                     // Enable UART on P0.4(RX) and P0.5(TX)
   XBR1    = 0x40;                     // Enable crossbar and enable
                                       // weak pull-ups

   TK1_RED_LED = 0;                    // Turn all LEDs off
   TK1_GRN_LED = 0;
   TK2_RED_LED = 0;                  
   TK2_GRN_LED = 0;
   TK3_RED_LED = 0;                   
   TK3_GRN_LED = 0;
}

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// This routine initializes the system clock to use the internal oscillator
// at its maximum frequency, enables the Missing Clock Detector and VDD
// monitor.
//

void SYSCLK_Init (void)
{
   OSCICN |= 0x03;                     // Configure internal oscillator for
                                       // its maximum frequency
   RSTSRC = 0x06;                      // Enable missing clock detector and
                                       // VDD Monitor
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Configure ADC0 to use Timer 2 as conversion source, and to initially point
// to Channel 2.  Disables ADC end of conversion interrupt. Leaves ADC
// disabled.
//
void ADC0_Init (void)
{
   ADC0CN = 0x02;                      // ADC0 disabled; Normal tracking
                                       // mode; ADC0 conversions are initiated
                                       // on timer 2

   AMX0P  = T1_CHPOS;                  // Channel 1+
   AMX0N  = T1_CHNEG;                  // Channel 1-

   ADC0CF = (SYSCLK/3000000) << 3;     // ADC conversion clock <= 3MHz

   ADC0CF &= ~0x04;                    // Right-Justify data
   REF0CN = 0x03;                      // VREF = P0.0 internal VREF, bias
                                       // generator is on.

}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//
void UART0_Init (void)
{
   SCON0 = 0x10;                       // SCON0: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits
   if (SYSCLK/BAUDRATE/2/256 < 1) {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON &= ~0x0B;                  // T1M = 1; SCA1:0 = xx
      CKCON |=  0x08;
   } else if (SYSCLK/BAUDRATE/2/256 < 4) {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 01
      CKCON |=  0x01;
   } else if (SYSCLK/BAUDRATE/2/256 < 12) {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 00
   } else {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 10
      CKCON |=  0x02;
   }

   TL1 = TH1;                          // init Timer1
   TMOD &= ~0xf0;                      // TMOD: timer 1 in 8-bit autoreload
   TMOD |=  0x20;
   TR1 = 1;                            // START Timer1
   TI0 = 1;                            // Indicate TX0 ready
}


//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Configure Timer2 to auto-reload at interval specified by <counts> (no
// interrupt generated) using SYSCLK as its time base.
//
void Timer2_Init (int counts)
{
   TMR2CN = 0x00;                      // STOP Timer2; Clear TF2H and TF2L;
                                       // disable low-byte interrupt; disable
                                       // split mode; select internal timebase
   CKCON |= 0x10;                      // Timer2 uses SYSCLK as its timebase

   TMR2RL  = -counts;                  // Init reload values
   TMR2    = TMR2RL;                   // Init Timer2 with reload value
}

//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Swipe_Card
//-----------------------------------------------------------------------------
//
// This routine performs the signal detection and data collection when a card
// is swiped through the reader for Track 1 2 and 3.  Interrupts should be
// turned off when this routine runs for optimal performance.
//

unsigned char Swipe_Card(void)
{
unsigned char data zerocount1;                  // Zero counter - Track 1
unsigned char data bytecount1;                  // Raw data counter - TK 1
unsigned char data zerocount2;                  // Zero counter - Track 2
unsigned char data bytecount2;                  // Raw data counter - TK 2
unsigned char data zerocount3;                  // Zero counter - Track 3
unsigned char data bytecount3;                  // Raw data counter - TK 3

char data runningsum1 = 0, rsum1_div = 0;       // Filtering variables
char data runningsum2 = 0, rsum2_div = 0;
char data runningsum3 = 0, rsum3_div = 0;

                                                // Minimum / Maximum and
                                                //    next peak values
char data localmax1 = 0, localmin1 = 0, next_peak1 = 0;
char data localmax2 = 0, localmin2 = 0, next_peak2 = 0;
char data localmax3 = 0, localmin3 = 0, next_peak3 = 0;

char data ADC_DATA;                             // Raw ADC Data (low byte)

unsigned int data cyclecount1, cyclecount2,     // Cycle counters
                  cyclecount3;

unsigned int maincycle;                         // Main time stamp for
                                                //   ADC conversions

unsigned int data maxtime1, mintime1;           // Min / Max time stamps
unsigned int data maxtime2, mintime2;
unsigned int data maxtime3, mintime3;

unsigned char data cycleindex1;                 // Index for # of cycles
unsigned char data cycleindex2;                 //   present in sum
unsigned char data cycleindex3;
unsigned int data cyclesum1 = 0;                // Sum over 3 cycles
unsigned int data cyclesum2 = 0;
unsigned int data cyclesum3 = 0;
unsigned int data CP75pct1 = 0, CP150pct1 = 0;  // 75% and 150% comparison
unsigned int data CP75pct2 = 0, CP150pct2 = 0;  //   values
unsigned int data CP75pct3 = 0, CP150pct3 = 0;

bit ZERO_WAIT1, FIRST_ONE1, BIT_RECORD1;        // Bits keep track of stages
bit ZERO_WAIT2, FIRST_ONE2, BIT_RECORD2;        //   in the collection
bit ZERO_WAIT3, FIRST_ONE3, BIT_RECORD3;

bit LASTEDGE1 = 0;                              // State of last edges:
bit LASTEDGE2 = 0;                              //   1 = Positive
bit LASTEDGE3 = 0;                              //   0 = Negative

   maincycle = 0;                               // Reset ADC timestamp
   Timeout_Counter = 0;                         // Reset Timeout Variables
   READ_TIMEOUT = 0;                            // (included for clarity)
   CH2_SWITCH = 0;

   T1RAW[0] = 0;                                // Reset Track1 Variables
   COLLECTED1 = 1;
   ZERO_WAIT1 = 1;
   FIRST_ONE1 = 0;
   BIT_RECORD1 = 0;
   zerocount1 = 0;
   bytecount1 = 0;

   T2RAW[0] = 0;                                // Reset Track2 Variables
   COLLECTED2 = 1;
   ZERO_WAIT2 = 1;
   FIRST_ONE2 = 0;
   BIT_RECORD2 = 0;
   zerocount2 = 0;
   bytecount2 = 0;

   T3RAW[0] = 0;                                // Reset Track3 Variables
   COLLECTED3 = 1;
   ZERO_WAIT3 = 1;
   FIRST_ONE3 = 0;
   BIT_RECORD3 = 0;
   zerocount3 = 0;
   bytecount3 = 0;


   AMX0P  = T1_CHPOS;                           // Set up AIN+ channel
   AMX0N  = T1_CHNEG;                           // Set up AIN- channel

   AD0EN = 1;                                   // Enable ADC0
   TR2 = 1;                                     // start Timer2

   // wait for Timer2 overflow flag - 1st conversion begins
   while (!TF2H);
   TF2H = 0;                                    // clear timer overflow flag
   AMX0P = T2_CHPOS;                            // switch AIN+ channel
   AMX0N = T2_CHNEG;                            // switch AIN- channel

   while (!READ_TIMEOUT)
   {
      Timeout_Counter++;                        // Increment counters
      maincycle++;

      if (CH2_SWITCH)                           // check if CH2 is sampled
      {
         // wait for Timer2 overflow flag
         while (!TF2H);
         AMX0P = T3_CHPOS;                      // switch AIN+ channel
         AMX0N = T3_CHNEG;                      // switch AIN- channel
         ADC_DATA = ADC0L;                      // read current data low byte
         TF2H = 0;                              // clear timer overflow flag
      }
      else
      {
         // wait for Timer2 overflow flag
         while (!TF2H);
         AMX0P = T1_CHPOS;                      // switch AIN+ channel
         AMX0N = T1_CHNEG;                      // switch AIN- channel
         ADC_DATA = ADC0L;                      // read current data low byte
         TF2H = 0;                              // clear timer overflow flag
      }

      // Perform exponential average
      runningsum1 = runningsum1 + ADC_DATA - rsum1_div;
      rsum1_div = runningsum1>>2;

      if (!ZERO_WAIT1)              // Test to see if still waiting for zeros
      {                             // If NOT.. collect data

         if (!LASTEDGE1)            // Test if last edge was negative
         {
            if (runningsum1 > next_peak1)          // Test against peak limit
            {
                                                   // Establish new local max
                                                   //    and compute min-max
                                                   //    peak timing
               localmax1 = runningsum1;
               cyclecount1 += mintime1 - maxtime1;
               next_peak1 = localmax1 - THRESHOLD1;

               if (cyclecount1 <= CP75pct1)        // 1/2 or Full cycle?
               {                                   // **1/2 cycle
                  BIT_RECORD1 = 1;
                  FIRST_ONE1 = 1;
               }
               else                                // **Full cycle
               {
                  cyclesum1 += cyclecount1;        // Update cycle sum
                  cycleindex1++;

                  if (FIRST_ONE1)                  // If first '1' is found
                  {
                     Temp_Byte1 = Temp_Byte1 << 1;
                     Temp1_b0 = BIT_RECORD1;       // Record a bit
                     bytecount1++;
                     BIT_RECORD1 = 0;              // Reset bit value to '0'
                  }
                  cyclecount1 = 0;                 // Reset cycle counter
                  CLEAR_TIMEOUT = 0;               // Keep from timing out
               }
               LASTEDGE1 = 1;                      // Positive edge
            }
            else if (runningsum1 < localmin1)      // Check against local min
            {
               localmin1 = runningsum1;            // Update local min
                                                   //   and next peak
               next_peak1 = localmin1 + THRESHOLD1;
               mintime1 = maincycle;               // Time stamp local min
            }
            else                                   // Perform some housekeeping
            {
               if (bytecount1 == 8)                // Store the current byte
               {
                  T1RAW[COLLECTED1] = Temp_Byte1;
                  bytecount1 = 0;
                  COLLECTED1++;
               }
               if (cycleindex1 >= 3)               // Calculate 75% Value
               {
                  CP75pct1 = cyclesum1 >> 2;
                  cyclesum1 = 0;
                  cycleindex1 = 0;
               }
            }
         }
         else                                      // Last edge was positive..
         {
            if (runningsum1 < next_peak1)          // Test against peak limit
            {
                                                   // Establish new local min
                                                   //    and compute max-min
                                                   //    peak timing
               localmin1 = runningsum1;
               cyclecount1 += maxtime1 - mintime1;
               next_peak1 = localmin1 + THRESHOLD1;

               if (cyclecount1 <= CP75pct1)        // 1/2 or Full cycle?
               {                                   // **1/2 cycle
                  BIT_RECORD1 = 1;
                  FIRST_ONE1 = 1;
               }
               else                                // **Full cycle
               {
                  cyclesum1 += cyclecount1;        // Update cycle sum
                  cycleindex1++;

                  if (FIRST_ONE1)                  // If first '1' is found
                  {
                     Temp_Byte1 = Temp_Byte1 << 1;
                     Temp1_b0 = BIT_RECORD1;       // Record a bit
                     bytecount1++;
                     BIT_RECORD1 = 0;              // Reset bit value to '0'
                  }
                  cyclecount1 = 0;                 // Reset cycle counter
                  CLEAR_TIMEOUT = 0;
               }
               LASTEDGE1 = 0;                      // Negative edge
            }
            else if (runningsum1 > localmax1)      // Check against local max
            {
               localmax1 = runningsum1;            // Update local max
                                                   //   and next peak
               next_peak1 = localmax1 - THRESHOLD1;
               maxtime1 = maincycle;               // Time stamp local max
            }
            else                                   // Perform some housekeeping
            {
               if (bytecount1 == 8)                // Store the current byte
               {
                  T1RAW[COLLECTED1] = Temp_Byte1;
                  bytecount1 = 0;
                  COLLECTED1++;
               }
               if (cycleindex1 >= 3)               // Calculate 75% Value
               {
                  CP75pct1 = cyclesum1 >> 2;
                  cyclesum1 = 0;
                  cycleindex1 = 0;
               }
            }
         }
      }     // End of data collection code (after Z_LIMIT zeros detected)

      else  // IF ZERO_WAIT1 == 1, still waiting for Z_LIMIT zeros
      {
         CLEAR_TIMEOUT = 0;

         if (!LASTEDGE1)            // Test if last edge was negative
         {
            if (runningsum1 > next_peak1)          // Test against peak limit
            {
                                                   // Establish new local max
                                                   //    and compute min-max
                                                   //    peak timing
               localmax1 = runningsum1;
               cyclecount1 += mintime1 - maxtime1;
               next_peak1 = localmax1 - THRESHOLD1;

               cyclesum1 += cyclecount1;           // Update cycle sum
               cycleindex1++;

               // Check for a value that looks periodic
               if ((cyclecount1 > CP75pct1)&&(cyclecount1 < CP150pct1))
               {
                  if (++zerocount1 == Z_LIMIT)     // Count up and check
                  {                                // for Z_LIMIT
                     ZERO_WAIT1 = 0;
                     TK1_RED_LED = 1;
                     TK1_GRN_LED = 1;
                  }
               }
               else                                // Outside of range
               {
                  zerocount1 = 0;                  // Reset zero count
               }

               cyclecount1 = 0;                    // Reset cycle counter
               LASTEDGE1 = 1;                      // Positive edge
            }
            else if (runningsum1 < localmin1)      // Check against local min
            {
               localmin1 = runningsum1;            // Update local min
                                                   //   and next peak
               next_peak1 = localmin1 + THRESHOLD1;
               mintime1 = maincycle;               // Time stamp local min
            }
            else                                   // Perform some housekeeping
            {
               if (cycleindex1 >= 3)               // Calculate 75% and 150%
               {
                  CP150pct1 = cyclesum1 >> 1;
                  CP75pct1 = CP150pct1 >> 1;
                  cyclesum1 = 0;
                  cycleindex1 = 0;
               }
            }
         }
         else                                      // Last edge was positive
         {
            if (runningsum1 < next_peak1)          // Test against peak limit
            {
                                                   // Establish new local min
                                                   //    and compute max-min
                                                   //    peak timing
               localmin1 = runningsum1;
               cyclecount1 += maxtime1 - mintime1;
               next_peak1 = localmin1 + THRESHOLD1;

               cyclesum1 += cyclecount1;           // Update cycle sum
               cycleindex1++;

               // Check for a value that looks periodic
               if ((cyclecount1 > CP75pct1)&&(cyclecount1 < CP150pct1))
               {
                  if (++zerocount1 == Z_LIMIT)     // Count up and check
                  {                                // for Z_LIMIT
                     ZERO_WAIT1 = 0;
                     TK1_RED_LED = 1;
                     TK1_GRN_LED = 1;
                  }
               }
               else                                // Outside of range
               {
                  zerocount1 = 0;                  // Reset zero count
               }
               cyclecount1 = 0;                    // Reset cycle counter
               LASTEDGE1 = 0;                      // Negative edge
            }
            else if (runningsum1 > localmax1)      // Check against local max
            {
               localmax1 = runningsum1;            // Update local max
                                                   //   and next peak
               next_peak1 = localmax1 - THRESHOLD1;
               maxtime1 = maincycle;               // Time stamp local max
            }
            else                                   // Perform some housekeeping
            {
               if (cycleindex1 >= 3)               // Calculate 75% and 150%
               {
                  CP150pct1 = cyclesum1 >> 1;
                  CP75pct1 = CP150pct1 >> 1;
                  cyclesum1 = 0;
                  cycleindex1 = 0;
               }
            }
         }
      }     // End of Waiting for Zeroes code (before Z_LIMIT reached)


      if (CH2_SWITCH)                              // Check if CH2 is sampled
      {
         // wait for Timer2 overflow flag
         while (!TF2H);
         AMX0P = T1_CHPOS;                         // switch AIN+ channel
         AMX0N = T1_CHNEG;                         // switch AIN- channel
         ADC_DATA = ADC0L;                         // read current data low byte
         TF2H = 0;                                 // clear timer overflow flag

         // Perform exponential average
         runningsum2 = runningsum2 + ADC_DATA - rsum2_div;
         rsum2_div = runningsum2>>2;

         maincycle++;

         if (!ZERO_WAIT2)           // Test to see if still waiting for zeros
         {                          // If NOT.. collect data

            if (!LASTEDGE2)         // Test if last edge was negative
            {
               if (runningsum2 > next_peak2)       // Test against peak limit
               {
                                                   // Establish new local max
                                                   //    and compute min-max
                                                   //    peak timing
                  localmax2 = runningsum2;
                  cyclecount2 += mintime2 - maxtime2;
                  next_peak2 = localmax2 - THRESHOLD2;

                  if (cyclecount2 <= CP75pct2)     // 1/2 or Full cycle?
                  {                                // **1/2 cycle
                     BIT_RECORD2 = 1;
                     FIRST_ONE2 = 1;
                  }
                  else                             // **Full cycle
                  {
                     cyclesum2 += cyclecount2;     // Update cycle sum
                     cycleindex2++;

                     if (FIRST_ONE2)               // If first '1' is found
                     {
                        Temp_Byte2 = Temp_Byte2 << 1;
                        Temp2_b0 = BIT_RECORD2;    // Record a bit
                        bytecount2++;
                        BIT_RECORD2 = 0;           // Reset bit value to '0'
                     }
                     cyclecount2 = 0;              // Reset cycle counter
                  }
                  LASTEDGE2 = 1;                   // Positive edge
               }
               else if (runningsum2 < localmin2)   // Check against local min
               {
                                                   // Update local min
                  localmin2 = runningsum2;         //   and next peak
                  next_peak2 = localmin2 + THRESHOLD2;
                  mintime2 = maincycle;            // Time stamp local min
               }
               else                                // Perform some housekeeping
               {
                  if (bytecount2 == 8)             // Store the current byte
                  {
                     T2RAW[COLLECTED2] = Temp_Byte2;
                     bytecount2 = 0;
                     COLLECTED2++;
                  }
                  if (cycleindex2 >= 3)            // Calculate 75% Value
                  {
                     CP75pct2 = cyclesum2 >> 2;
                     cyclesum2 = 0;
                     cycleindex2 = 0;
                  }
               }
            }
            else                                   // Last edge was positive..
            {
               if (runningsum2 < next_peak2)       // Test against peak limit
               {
                                                   // Establish new local min
                                                   //    and compute max-min
                                                   //    peak timing
                  localmin2 = runningsum2;
                  cyclecount2 += maxtime2 - mintime2;
                  next_peak2 = localmin2 + THRESHOLD2;

                  if (cyclecount2 <= CP75pct2)     // 1/2 or Full cycle?
                  {                                // **1/2 cycle
                     BIT_RECORD2 = 1;
                     FIRST_ONE2 = 1;
                  }
                  else                             // **Full cycle
                  {
                     cyclesum2 += cyclecount2;     // Update cycle sum
                     cycleindex2++;

                     if (FIRST_ONE2)               // If first '1' is found
                     {
                        Temp_Byte2 = Temp_Byte2 << 1;
                        Temp2_b0 = BIT_RECORD2;    // Record a bit
                        bytecount2++;
                        BIT_RECORD2 = 0;           // Reset bit value to '0'
                     }
                     cyclecount2 = 0;              // Reset cycle counter
                  }
                  LASTEDGE2 = 0;                   // Negative edge
               }
               else if (runningsum2 > localmax2)   // Check against local max
               {
                                                   // Update local max
                  localmax2 = runningsum2;         //   and next peak
                  next_peak2 = localmax2 - THRESHOLD2;
                  maxtime2 = maincycle;            // Time stamp local max

               }
               else                                // Perform some housekeeping
               {
                  if (bytecount2 == 8)             // Store the current byte
                  {
                     T2RAW[COLLECTED2] = Temp_Byte2;
                     bytecount2 = 0;
                     COLLECTED2++;
                  }
                  if (cycleindex2 >= 3)            // Calculate 75% Value
                  {
                     CP75pct2 = cyclesum2 >> 2;
                     cyclesum2 = 0;
                     cycleindex2 = 0;
                  }
               }
            }
         }     // End of data collection code (after Z_LIMIT zeros detected)

         else  // IF ZERO_WAIT2 == 1, still waiting for Z_LIMIT zeros
         {
            if (!LASTEDGE2)         // Test if last edge was negative
            {
               if (runningsum2 > next_peak2)       // Test against peak limit
               {
                                                   // Establish new local max
                                                   //    and compute min-max
                                                   //    peak timing
                  localmax2 = runningsum2;
                  cyclecount2 += mintime2 - maxtime2;
                  next_peak2 = localmax2 - THRESHOLD2;

                  cyclesum2 += cyclecount2;        // Update cycle sum
                  cycleindex2++;

                  // Check for a value that looks periodic
                  if ((cyclecount2 > CP75pct2)&&(cyclecount2 < CP150pct2))
                  {
                     if (++zerocount2 == Z_LIMIT)  // Count up and check
                     {                             // for Z_LIMIT
                        ZERO_WAIT2 = 0;
                        TK2_RED_LED = 1;
                        TK2_GRN_LED = 1;
                     }
                  }
                  else                             // Outside of range
                  {
                     zerocount2 = 0;               // Reset zero count
                  }
                  cyclecount2 = 0;                 // Reset cycle counter
                  LASTEDGE2 = 1;                   // Positive edge
               }
               else if (runningsum2 < localmin2)   // Check against local min
               {
                                                   // Update local min
                  localmin2 = runningsum2;         //   and next peak
                  next_peak2 = localmin2 + THRESHOLD2;
                  mintime2 = maincycle;            // Time stamp local min
               }
               else                                // Perform some housekeeping
               {
                  if (cycleindex2 >= 3)            // Calculate 75% and 150%
                  {
                     CP150pct2 = cyclesum2 >> 1;
                     CP75pct2 = CP150pct2 >> 1;
                     cyclesum2 = 0;
                     cycleindex2 = 0;
                  }
               }
            }
            else                                   // Last edge was positive
            {
               if (runningsum2 < next_peak2)       // Test against peak limit
               {
                                                   // Establish new local min
                                                   //    and compute max-min
                                                   //    peak timing
                  localmin2 = runningsum2;
                  cyclecount2 += maxtime2 - mintime2;
                  next_peak2 = localmin2 + THRESHOLD2;

                  cyclesum2 += cyclecount2;        // Update cycle sum
                  cycleindex2++;

                  // Check for a value that looks periodic
                  if ((cyclecount2 > CP75pct2)&&(cyclecount2 < CP150pct2))
                  {
                     if (++zerocount2 == Z_LIMIT)  // Count up and check
                     {                             // for Z_LIMIT
                        ZERO_WAIT2 = 0;
                        TK2_RED_LED = 1;
                        TK2_GRN_LED = 1;
                     }
                  }
                  else                             // Outside of range
                  {
                     zerocount2 = 0;               // Reset zero count
                  }
                  cyclecount2 = 0;                 // Reset cycle counter
                  LASTEDGE2 = 0;                   // Negative edge
               }
               else if (runningsum2 > localmax2)   // Check against local max
               {
                                                   // Update local max
                  localmax2 = runningsum2;         //   and next peak
                  next_peak2 = localmax2 - THRESHOLD2;
                  maxtime2 = maincycle;            // Time stamp local max
               }
               else                                // Perform some housekeeping
               {
                  if (cycleindex2 >= 3)            // Calculate 75% and 150%
                  {
                     CP150pct2 = cyclesum2 >> 1;
                     CP75pct2 = CP150pct2 >> 1;
                     cyclesum2 = 0;
                     cycleindex2 = 0;
                  }
               }
            }
         }     // End of Waiting for Zeroes code (before Z_LIMIT reached)

         // wait for Timer2 overflow flag
         while (!TF2H);
         AMX0P = T3_CHPOS;                         // switch AIN+ channel
         AMX0N = T3_CHNEG;                         // switch AIN- channel
         ADC_DATA = ADC0L;                         // read current data low byte
         TF2H = 0;                                 // clear timer overflow flag
      }  // End IF CH2_SWITCH

      else
      {
         // wait for Timer2 overflow flag
         while (!TF2H);
         AMX0P = T2_CHPOS;                         // switch AIN+ channel
         AMX0N = T2_CHNEG;                         // switch AIN- channel
         ADC_DATA = ADC0L;                         // read current data low byte
         TF2H = 0;                                 // clear timer overflow flag
      }

      // Perform exponential average
      runningsum3 = runningsum3 + ADC_DATA - rsum3_div;
      rsum3_div = runningsum3>>2;

      maincycle++;

      if (!ZERO_WAIT3)              // Test to see if still waiting for zeros
      {                             // If NOT.. collect data

         if (!LASTEDGE3)            // Test if last edge was negative
         {
            if (runningsum3 > next_peak3)          // Test against peak limit
            {
                                                   // Establish new local max
                                                   //    and compute min-max
                                                   //    peak timing
               localmax3 = runningsum3;
               cyclecount3 += mintime3 - maxtime3;
               next_peak3 = localmax3 - THRESHOLD3;

               if (cyclecount3 <= CP75pct3)        // 1/2 or Full cycle?
               {                                   // **1/2 cycle
                  BIT_RECORD3 = 1;
                  FIRST_ONE3 = 1;
               }
               else                                // **Full cycle
               {
                  cyclesum3 += cyclecount3;        // Update cycle sum
                  cycleindex3++;

                  if (FIRST_ONE3)                  // If first '1' is found
                  {
                     Temp_Byte3 = Temp_Byte3 << 1;
                     Temp3_b0 = BIT_RECORD3;       // Record a bit
                     bytecount3++;
                     BIT_RECORD3 = 0;              // Reset bit value to '0'

                  }
                  cyclecount3 = 0;                 // Reset cycle counter
               }
               LASTEDGE3 = 1;                      // Positive edge
            }
            else if (runningsum3 < localmin3)      // Check against local min
            {                                      // Update local min
               localmin3 = runningsum3;            //   and next peak
               next_peak3 = localmin3 + THRESHOLD3;
               mintime3 = maincycle;               // Time stamp local min
            }
            else                                   // Perform some housekeeping
            {
               if (bytecount3 == 8)                // Store the current byte
               {
                  T3RAW[COLLECTED3] = Temp_Byte3;
                  bytecount3 = 0;
                  COLLECTED3++;
               }
               if (cycleindex3 >= 3)               // Calculate 75% Value
               {
                  CP75pct3 = cyclesum3 >> 2;
                  cyclesum3 = 0;
                  cycleindex3 = 0;
               }
            }
         }
         else                                      // Last edge was positive..
         {
            if (runningsum3 < next_peak3)          // Test against peak limit
            {
                                                   // Establish new local min
                                                   //    and compute max-min
                                                   //    peak timing
               localmin3 = runningsum3;
               cyclecount3 += maxtime3 - mintime3;
               next_peak3 = localmin3 + THRESHOLD3;

               if (cyclecount3 <= CP75pct3)        // 1/2 or Full cycle?
               {                                   // **1/2 cycle
                  BIT_RECORD3 = 1;
                  FIRST_ONE3 = 1;
               }
               else                                // **Full cycle
               {
                  cyclesum3 += cyclecount3;        // Update cycle sum
                  cycleindex3++;

                  if (FIRST_ONE3)                  // If first '1' is found
                  {
                     Temp_Byte3 = Temp_Byte3 << 1;
                     Temp3_b0 = BIT_RECORD3;       // Record a bit
                     bytecount3++;
                     BIT_RECORD3 = 0;              // Reset bit value to '0'
                  }
                  cyclecount3 = 0;                 // Reset cycle counter
               }
               LASTEDGE3 = 0;                      // Negative edge
            }
            else if (runningsum3 > localmax3)      // Check against local max
            {                                      // Update local max
               localmax3 = runningsum3;            //   and next peak
               next_peak3 = localmax3 - THRESHOLD3;
               maxtime3 = maincycle;               // Time stamp local max
            }
            else                                   // Perform some housekeeping
            {
               if (bytecount3 == 8)                // Store the current byte
               {
                  T3RAW[COLLECTED3] = Temp_Byte3;
                  bytecount3 = 0;
                  COLLECTED3++;
               }
               if (cycleindex3 >= 3)               // Calculate 75% Value
               {
                  CP75pct3 = cyclesum3 >> 2;
                  cyclesum3 = 0;
                  cycleindex3 = 0;
               }
            }
         }
      }     // End of data collection code (after Z_LIMIT zeros detected)

      else  // IF ZERO_WAIT3 == 1, still waiting for Z_LIMIT zeros
      {
         if (!LASTEDGE3)            // Test if last edge was negative
         {
            if (runningsum3 > next_peak3)          // Test against peak limit
            {
                                                   // Establish new local max
                                                   //    and compute min-max
                                                   //    peak timing
               localmax3 = runningsum3;
               cyclecount3 += mintime3 - maxtime3;
               next_peak3 = localmax3 - THRESHOLD3;

               cyclesum3 += cyclecount3;           // Update cycle sum
               cycleindex3++;

               // Check for a value that looks periodic
               if ((cyclecount3 > CP75pct3)&&(cyclecount3 < CP150pct3))
               {
                  if (++zerocount3 == Z_LIMIT)     // Count up and check
                  {                                // for Z_LIMIT
                     ZERO_WAIT3 = 0;
                     TK3_RED_LED = 1;
                     TK3_GRN_LED = 1;
                  }
               }
               else                                // Outside of range
               {
                  zerocount3 = 0;                  // Reset zero count
               }
               cyclecount3 = 0;                    // Reset cycle counter
               LASTEDGE3 = 1;                      // Positive edge
            }
            else if (runningsum3 < localmin3)      // Check against local min
            {                                      // Update local min
               localmin3 = runningsum3;            //   and next peak
               next_peak3 = localmin3 + THRESHOLD3;
               mintime3 = maincycle;               // Time stamp local min
            }
            else                                   // Perform some housekeeping
            {
               if (cycleindex3 >= 3)               // Calculate 75% and 150%
               {
                  CP150pct3 = cyclesum3 >> 1;
                  CP75pct3 = CP150pct3 >> 1;
                  cyclesum3 = 0;
                  cycleindex3 = 0;
               }
            }
         }
         else                                      // Last edge was positive
         {
            if (runningsum3 < next_peak3)          // Test against peak limit
            {
                                                   // Establish new local min
                                                   //    and compute max-min
                                                   //    peak timing
               localmin3 = runningsum3;
               cyclecount3 += maxtime3 - mintime3;
               next_peak3 = localmin3 + THRESHOLD3;

               cyclesum3 += cyclecount3;           // Update cycle sum
               cycleindex3++;

               // Check for a value that looks periodic
               if ((cyclecount3 > CP75pct3)&&(cyclecount3 < CP150pct3))
               {
                  if (++zerocount3 == Z_LIMIT)     // Count up and check
                  {                                // for Z_LIMIT
                     ZERO_WAIT3 = 0;
                     TK3_RED_LED = 1;
                     TK3_GRN_LED = 1;
                  }
               }
               else                                // Outside of range
               {
                  zerocount3 = 0;                  // Reset zero count
               }

               cyclecount3 = 0;                    // Reset cycle counter
               LASTEDGE3 = 0;                      // Negative edge

            }
            else if (runningsum3 > localmax3)      // Check against local max
            {                                      // Update local max
               localmax3 = runningsum3;            //   and next peak
               next_peak3 = localmax3 - THRESHOLD3;
               maxtime3 = maincycle;               // Time stamp local max
            }
            else                                   // Perform some housekeeping
            {
               if (cycleindex3 >= 3)               // Calculate 75% and 150%
               {
                  CP150pct3 = cyclesum3 >> 1;
                  CP75pct3 = CP150pct3 >> 1;
                  cyclesum3 = 0;
                  cycleindex3 = 0;
               }
            }
         }
      }     // End of Waiting for Zeroes code (before Z_LIMIT reached)
   } // End While (!READ_TIMEOUT)

   // Finish off last bytes with zeros..
   while (bytecount1 < 8)
   {
      Temp_Byte1 = Temp_Byte1 << 1;
      Temp1_b0 = 0;                                // record a zero
      bytecount1++;
   }
   T1RAW[COLLECTED1] = Temp_Byte1;

   while (bytecount2 < 8)
   {
      Temp_Byte2 = Temp_Byte2 << 1;
      Temp2_b0 = 0;                                // record a zero
      bytecount2++;
   }
   T2RAW[COLLECTED2] = Temp_Byte2;

   while (bytecount3 < 8)
   {
      Temp_Byte3 = Temp_Byte3 << 1;
      Temp3_b0 = 0;                                // record a zero
      bytecount3++;
   }
   T3RAW[COLLECTED3] = Temp_Byte3;

   return (1);
}


//-----------------------------------------------------------------------------
// TrackErrorCheck
//-----------------------------------------------------------------------------
//
// This routine checks the decoded track data for Start Sentinel, End Sentinel,
// Parity, and LRC errors.
//

char TrackErrorCheck (unsigned char maxindex, unsigned char StartSen,
   unsigned char EndSen, unsigned char CharBits)
{
unsigned char idata ASCII_Index, ASCII_Mask;
unsigned char idata ASCII_Data, PC_count, Read_LRC = 0, Calc_LRC = 0;
char idata errorcode = 0;
bit ES_Found = 0,  ParityCheck = 0;

   ASCII_Mask = 0x7F >> (8 - CharBits);      // Mask used to separate data info

   if (ASCII_array[0] != StartSen)           // Check for SS at start of array
   {
      errorcode |= 0x81;                     // ERROR - SS is not 1st character
   }

   // Loop through ASCII_array and check each byte for errors
   for (ASCII_Index = 0; ASCII_Index <= maxindex; ASCII_Index++)
   {
      ASCII_Data = ASCII_array[ASCII_Index];
      if (!ES_Found)                         // If ES not found yet
      {
         // LRC Check - XOR's data from all bytes (except the LRC)
         Calc_LRC ^= (ASCII_Data & ASCII_Mask);

         if (ASCII_Data == EndSen)           // If this is the End Sentinel,
         {                                   // treat the next character as
                                             // the LRC, and signal that
                                             // the ES has been found
            Read_LRC = (ASCII_array[ASCII_Index+1] & ASCII_Mask);
            maxindex = ASCII_Index+1;
            ES_Found = 1;
         }
      }

      // Parity Check - checks #1's against Parity bit for ODD parity.
      ParityCheck = 0;                       // Reset parity check variable
      for (PC_count = 0; PC_count < CharBits; PC_count++)
      {
         ParityCheck ^= (ASCII_Data & 0x01);
         ASCII_Data = ASCII_Data >> 1;
      }
      if (ParityCheck == (ASCII_Data & 0x01))
      {
         ASCII_array[ASCII_Index] |= 0x80;   // Mark this byte for ID later
         errorcode |= 0x88;                  // ERROR - Parity error
      }
   }

   // Check that End Sentinel was found in captured data
   if (!ES_Found)
   {
      errorcode |=0x82; // ERROR - End Sentinel never found
   }
   // If ES was found...
   else if (Calc_LRC != (Read_LRC & ASCII_Mask))
   {
      errorcode |= 0x84; // LRC error

      // Parity Check for LRC - checks #1's against Parity bit for ODD parity.
      ParityCheck = 0;                       // Reset parity check variable
      for (PC_count = 0; PC_count < CharBits; PC_count++)
      {
         ParityCheck ^= (Read_LRC & 0x01);
         Read_LRC = Read_LRC >> 1;
      }
      if (ParityCheck == (Read_LRC & 0x01))
      {
         ASCII_array[maxindex] |= 0x80;      // Mark LRC byte for ID later
         errorcode |= 0x88;                  // ERROR - Parity error
      }
   }

   // If no errors were detected, return the number of bytes found.
   // Otherwise, return the error code.
   if (errorcode == 0)
   {
      return ASCII_Index;
   }
   else
   {
      return errorcode;
   }

}

//-----------------------------------------------------------------------------
// DecodeTrackForward
//-----------------------------------------------------------------------------
//
// This routine is used to decode a track into characters, assuming it was
// recorded in the forward direction into the array.
//

char DecodeTrackForward (unsigned char maxindex, unsigned char Byte_Offset,
   unsigned char Bit_Offset, unsigned char *TrackRAW, unsigned char CharBits)
{
unsigned char idata Track_Index = 0;
char idata ASCII_Index = 0, ASCII_Mask;
unsigned char idata Track_Data, ASCII_Data;
unsigned char idata Track_bit, ASCII_bit;

   // Reset temporary variables
   ASCII_bit = 0x01;
   ASCII_Data = 0x00;

   // Generate a bit comparison value for sorting through ASCII bytes
   ASCII_Mask = 0x01 << (CharBits-1);

   // Begin at the specified offset, and proceed until the end of the track
   for (Track_Index = Byte_Offset; Track_Index <= maxindex; Track_Index++)
   {
      // Grab a byte of raw data
      Track_Data = TrackRAW[Track_Index];

      // Unpack raw data byte into character(s)
      for (Track_bit = Bit_Offset; Track_bit != 0x00; Track_bit = Track_bit>>1)
      {
         if (Track_bit & Track_Data)
         {
            ASCII_Data |= ASCII_bit;
         }
         else
         {
            ASCII_Data &= ~ASCII_bit;
         }
         if (ASCII_bit != ASCII_Mask)
         {
            ASCII_bit = ASCII_bit << 1;
         }
         else
         {
            ASCII_bit = 0x01;
            ASCII_array[ASCII_Index] = ASCII_Data;

            if ((ASCII_Data == 0x00)||(ASCII_Index == 126))
            {
               Track_Index = maxindex;       // end translation
            }

            ASCII_Index++;
         }
      }
   }

   // Return the number of characters unpacked
   return (ASCII_Index);

}

//-----------------------------------------------------------------------------
// DecodeTrackBackward
//-----------------------------------------------------------------------------
//
// This routine is used to decode a track into characters, assuming it was
// recorded in the backward direction into the array.
//
char DecodeTrackBackward (unsigned char Byte_Offset, unsigned char Bit_Offset,
   unsigned char *TrackRAW, unsigned char CharBits)
{
unsigned char idata Track_Index;
char idata ASCII_Index = 0, ASCII_Mask;
unsigned char idata Track_Data, ASCII_Data;
unsigned char idata ASCII_bit;


   // Reset temporary variables
   ASCII_bit = 0x01;
   ASCII_Data = 0x00;

   // Generate a bit comparison value for sorting through ASCII bytes
   ASCII_Mask = 0x01 << (CharBits-1);

   // Begin at the specified offset, and proceed until the beginning
   for (Track_Index = Byte_Offset; Track_Index != 0x00; Track_Index--)
   {
      // Grab a byte of raw data
      Track_Data = TrackRAW[Track_Index];

      // Unpack raw data byte into character(s)
      while (Bit_Offset != 0x00)
      {
         if (Bit_Offset & Track_Data)
         {
            ASCII_Data |= ASCII_bit;
         }
         else
         {
            ASCII_Data &= ~ASCII_bit;
         }
         if (ASCII_bit != ASCII_Mask)
         {
            ASCII_bit = ASCII_bit << 1;
         }
         else
         {
            ASCII_bit = 0x01;
            ASCII_array[ASCII_Index] = ASCII_Data;
            ASCII_Data = 0;
            ASCII_Index++;
         }
         Bit_Offset = Bit_Offset << 1;
      }
      Bit_Offset = 0x01;
   }

   // Finish off last byte with trailing zeros
   ASCII_Mask = ASCII_Mask << 1;
   while (ASCII_bit != ASCII_Mask)
   {
      ASCII_Data &= ~ASCII_bit;
      ASCII_bit = ASCII_bit << 1;
   }
   ASCII_array[ASCII_Index] = ASCII_Data;

   // Return the number of characters unpacked
   return (ASCII_Index);

}


//-----------------------------------------------------------------------------
// GetDirection
//-----------------------------------------------------------------------------
//
// This routine determines which direction data was collected from the magnetic
// stripe and calls the appropriate decoding routine.
//

char GetDirection (unsigned char maxindex, unsigned char StartSen,
   unsigned char EndSen, unsigned char *TrackRAW, unsigned char CharBits)
{

unsigned char idata FW_Byte_Off, FW_Bit_Off, RV_Byte_Off, RV_Bit_Off;
unsigned char idata Read_Char, Bit_Count, Temp_Char, Temp_Bit, Temp_Mask;
char idata MAX_Decoded;
bit FW_StartSen, RV_StartSen, Direction_Found = 0, Abort_Direction = 0;

   // Initialize Index Pointers
   FW_Byte_Off = 1;
   FW_Bit_Off = 0x80;
   RV_Byte_Off = maxindex;
   RV_Bit_Off = 0x01;


   while ((Direction_Found == 0)&&(Abort_Direction == 0))
   {
      // Read a byte at FW pointer
      Read_Char = TrackRAW[FW_Byte_Off];

      // Find the next '1' Forward
      while ((FW_Byte_Off != RV_Byte_Off)&&((Read_Char & FW_Bit_Off) == 0))
      {
         FW_Bit_Off = FW_Bit_Off >> 1;
         if (FW_Bit_Off == 00)
         {
            FW_Bit_Off = 0x80;
            FW_Byte_Off++;
            Read_Char = TrackRAW[FW_Byte_Off];
         }
      }

      if (FW_Byte_Off == RV_Byte_Off)
      {
         Abort_Direction = 1;
      }

      Temp_Bit = 0x02;
      Temp_Char = 0x01;
      Temp_Mask = FW_Bit_Off;

      for (Bit_Count = 1; Bit_Count < CharBits; Bit_Count++)
      {
         Temp_Mask = Temp_Mask >> 1;
         if (Temp_Mask == 0x00)
         {
            Temp_Mask = 0x80;
            Read_Char = TrackRAW[FW_Byte_Off+1];
         }
         if (Read_Char & Temp_Mask)
         {
            Temp_Char |= Temp_Bit;
         }
         else
         {
            Temp_Char &= ~Temp_Bit;
         }
         Temp_Bit = Temp_Bit << 1;
      }

      // Check character against Start Sentinel
      if (Temp_Char == StartSen)
      {
         FW_StartSen = 1;
      }
      else
      {
         FW_StartSen = 0;
      }

      // Read a byte at RV pointer
      Read_Char = TrackRAW[RV_Byte_Off];

      // Find the next '1' Reverse
      while ((FW_Byte_Off != RV_Byte_Off)&&((Read_Char & RV_Bit_Off) == 0))
      {
         RV_Bit_Off = RV_Bit_Off << 1;
         if (RV_Bit_Off == 00)
         {
            RV_Bit_Off = 0x01;
            RV_Byte_Off--;
            Read_Char = TrackRAW[RV_Byte_Off];
         }
      }

      if (FW_Byte_Off == RV_Byte_Off)
      {
         Abort_Direction = 1;
      }

      Temp_Bit = 0x02;
      Temp_Char = 0x01;
      Temp_Mask = RV_Bit_Off;

      for (Bit_Count = 1; Bit_Count < CharBits; Bit_Count++)
      {
         Temp_Mask = Temp_Mask << 1;
         if (Temp_Mask == 0x00)
         {
            Temp_Mask = 0x01;
            Read_Char = TrackRAW[RV_Byte_Off-1];
         }
         if (Read_Char & Temp_Mask)
         {
            Temp_Char |= Temp_Bit;
         }
         else
         {
            Temp_Char &= ~Temp_Bit;
         }
         Temp_Bit = Temp_Bit << 1;
      }

      // Check character against Start Sentinel
      if (Temp_Char == StartSen)
      {
         RV_StartSen = 1;
      }
      else
      {
         RV_StartSen = 0;
      }

      if (FW_StartSen ^ RV_StartSen)
      {
         Direction_Found = 1;
      }
      else if (FW_StartSen && RV_StartSen)
      {
         //*** Check for ES Backwards in front
         Temp_Bit = 0x80;
         Temp_Char = 0x00;
         Temp_Mask = FW_Bit_Off;

         MAX_Decoded = FW_Byte_Off; // MAX_Decoded used as temporary storage
         if ((Temp_Mask >> CharBits) != 0x00)
         {
            Temp_Mask = Temp_Mask >> CharBits;
         }
         else
         {
            FW_Byte_Off++;
            Temp_Mask = Temp_Mask << (8 - CharBits);
         }

         Read_Char = TrackRAW[FW_Byte_Off];
         for (Bit_Count = 0; Bit_Count < CharBits; Bit_Count++)
         {
            if (Read_Char & Temp_Mask)
            {
               Temp_Char |= Temp_Bit;
            }
            else
            {
               Temp_Char &= ~Temp_Bit;
            }
            Temp_Bit = Temp_Bit >> 1;

            Temp_Mask = Temp_Mask >> 1;
            if (Temp_Mask == 0x00)
            {
               Temp_Mask = 0x80;
               Read_Char = TrackRAW[FW_Byte_Off+1];
            }
         }
         FW_Byte_Off = MAX_Decoded; // Restore FW_Byte_Off

         Temp_Char = Temp_Char >> (8 - CharBits);
         // Check character against End Sentinel
         // If found here, track is reverse.
         if (Temp_Char == EndSen)
         {
            FW_StartSen = 0;
         }
         //otherwise, it is forward
         else
         {
            RV_StartSen = 0;
         }

         Direction_Found = 1;
      }
      else if (!Abort_Direction)
      {
         FW_Bit_Off = FW_Bit_Off >> 1;
         if (FW_Bit_Off == 00)
         {
            FW_Bit_Off = 0x80;
            FW_Byte_Off++;
         }
         RV_Bit_Off = RV_Bit_Off << 1;
         if (RV_Bit_Off == 00)
         {
            RV_Bit_Off = 0x01;
            RV_Byte_Off--;
         }

         if (FW_Byte_Off >= RV_Byte_Off)
         {
            Abort_Direction = 1;
         }
      }

   }  // End while((Direction_Found == 0)&&(Abort_Direction == 0))

   if ((Direction_Found)&&(!Abort_Direction))
   {
      if (FW_StartSen)
      {
         MAX_Decoded = DecodeTrackForward(maxindex, FW_Byte_Off, FW_Bit_Off,
            TrackRAW, CharBits);
      }
      else if (RV_StartSen)
      {
         MAX_Decoded = DecodeTrackBackward(RV_Byte_Off, RV_Bit_Off,
            TrackRAW, CharBits);
      }
   }
   else
   {
      MAX_Decoded = 0x81;                    // Could not find Start Sentinel
   }

   return (MAX_Decoded);

}

//-----------------------------------------------------------------------------
// UART_CharOut
//-----------------------------------------------------------------------------
//
// This routine sends a single character to the UART.  It is used in lieu of
// printf() to reduce overall code size.
//

void UART_CharOut (unsigned char c)
{
   if (c == '\n')
   {
      while (!TI0);
      TI0 = 0;
      SBUF0 = 0x0d;                   /* output CR  */
   }
   while (!TI0);
   TI0 = 0;
   SBUF0 = c;
}

//-----------------------------------------------------------------------------
// UART_StringOut
//-----------------------------------------------------------------------------
//
// This routine calls the UART_CharOut repeatedly to send a string value to the
// UART.  It is used in lieu of printf() to reduce overall code size.
//

void UART_StringOut (unsigned char *c)
{
   while (*c != 0x00)
   {
      UART_CharOut(*c);
      c++;
   }
}


#if DEBUG
//-----------------------------------------------------------------------------
// UART_HexOut
//-----------------------------------------------------------------------------
//
// This routine sends the hexadecimal value of a character to the UART as ASCII
// text. Only used when DEBUG = 1.
//
void UART_HexOut (unsigned char c)
{
   while (!TI0);
   TI0 = 0;
   if ((c & 0xF0) < 0xA0)
      SBUF0 = ((c >> 4) & 0x0F) + 0x30;
   else
      SBUF0 = ((c >> 4) & 0x0F) + 0x37;

   while (!TI0);
   TI0 = 0;
   if ((c & 0x0F) < 0x0A)
      SBUF0 = (c & 0x0F) + 0x30;
   else
      SBUF0 = (c & 0x0F) + 0x37;

}
#endif   // END #if DEBUG