//-----------------------------------------------------------------------------
//
// Copyright 2004 Silicon Laboratories
// 
// Filename:      F350_Weigh_Scale.h
// Target Device: 8051F350
// Created:       15 JAN 2004
// Created By:    DKC
// Tool chain:    KEIL Eval C51
//
// This is a stand alone weith scale design. It output units in lbs via an LCD 
// or a PC via the UART.
//
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f350.h>
#include "F350_Weigh_Scale.h"          //Weigh Scale Hearder File
#include <stdio.h>
                   
//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------
void Config_F350(void) 
{  RSTSRC   = 0x06;                    // Enable VDD Monitor and missing clock   	
//-----------------------------------------------------------------------------
// PCA Configuration
//-----------------------------------------------------------------------------
   PCA0MD &= ~0x40;                    // WDTE = 0 (Disable watchdog timer)

//-----------------------------------------------------------------------------
// Port Configuration
//-----------------------------------------------------------------------------
   XBR0     = 0x01;                    // Enable UART to Pins P0.4, P0.5
   XBR1     = 0x40;                    // Enable Crossbar

   P0SKIP   = 0x00;                    // Skip No Port Pins
   P0MDOUT |= 0x10;                    // Enable UTX as push-pull output
   P0MDIN   = 0xFF;                    // Configure No Pins as Analog Inputs

//-----------------------------------------------------------------------------
// Oscilator Configuration
//-----------------------------------------------------------------------------
   OSCICN |= 0xC0;                     // Configure internal oscillator for
                                       // its default frequency

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//
   SCON0 = 0x10;                       // SCON0: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits
   if (SYSCLK/BAUDRATE/2/256 < 1) {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON |=  0x08;                  // T1M = 1; SCA1:0 = xx
   } else if (SYSCLK/BAUDRATE/2/256 < 4) {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 01                  
      CKCON |=  0x01;
   } else if (SYSCLK/BAUDRATE/2/256 < 12) {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 00
   } else {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 10
      CKCON |=  0x02;
   }

   TL1 = TH1;                          // init Timer1
   TMOD &= ~0xf0;                      // TMOD: timer 1 in 8-bit autoreload
   TMOD |=  0x20;                       
   TR1 = 1;                            // START Timer1
   TI0 = 1;                            // Indicate TX0 ready
}

//-----------------------------------------------------------------------------
// FLASH_PageUpdate
//-----------------------------------------------------------------------------
// 
// This routine updates <length> characters at <dest> with those pointed to
// by <src> using a page-based read-modify-write process.
//
// It first erases <SCRATCH_PAGE> and copies the page containing the <dest> 
// data to <SCRATCH_PAGE>, excluding <length> bytes starting at <dest>.  
// It then copies <SCRATCH_PAGE> back to the page containing <dest>.  Finally, 
// it copies <length> bytes from <src> to <dest>, completing the update 
// process.  
// Note: this algorithm does not take into account memory page boundaries...
//    	It assumes each update is within one page.
void FLASH_PageUpdate (signed char *src, signed char *dest, int length) reentrant
{ 
   int i;                              // byte counter
   unsigned char xdata *pwrite;        // FLASH write pointer
   unsigned char code *pread;          // FLASH read pointer
   unsigned char code *pstop;
   unsigned char code *pstart;

   pwrite = (unsigned char xdata *) SCRATCH_PAGE;

   EA = 0;                             // disable interrupts (precautionary)
   PSCTL = 0x03;                       // MOVX writes target FLASH memory;
                                       // FLASH Erase operations enabled
   FLKEY = 0xA5;                       // FLASH key sequence #1
   FLKEY = 0xF1;                       // FLASH key sequence #2

   *pwrite = 0x00;                     // initiate erase operation

   PSCTL = 0x00;                       // Disable FLASH Writes

   // initialize <pread> to beginning of FLASH page containing <dest>
   pread = (unsigned char code *) ((unsigned int) dest & 0xFE00);
   
   // <pstop> points to <dest>, which is the beginning of the area to exclude
   // from the copy process.
   pstop = (unsigned char code *) dest;

   // <pstart> points to the byte right after the area to exclude from the
   // copy process.
   pstart = (unsigned char code *) (pstop + length);

   // Now we copy the page containing <dest> to SCRATCH_PAGE, excluding
   // the bytes that are to be changed.
   for (i = 0; i < 512; i++) {
      if ((pread < pstop) || (pread >= pstart)) {
         if (*pread != 0xFF) {         // exclude copying 0xff's for efficiency
            
            PSCTL = 0x01;              // disable FLASH erase operations;
                                       // MOVX writes target FLASH
            FLKEY = 0xA5;              // FLASH key sequence #1
            FLKEY = 0xF1;              // FLASH key sequence #2
            *pwrite = *pread;          // copy bytes
         }  PSCTL = 0x00;              // Disable FLASH Writes
      }
      pwrite++;                        // advance pointers
      pread++;
   }

   // At this point, <SCRATCH_PAGE> has a copy of the entire page containing
   // <dest>, with the exclusion of the bytes to be replaced.  We now copy
   // <SCRATCH_PAGE> back to the page containing <dest>.

   pwrite = (unsigned char xdata *) ((unsigned int) dest & 0xFE00);
   pread = (unsigned char code *) SCRATCH_PAGE;
   FLASH_PageCopy (pread, pwrite);

   // At this point, the page containing <dest> has been restored; the bytes
   // to be changed are now 0xFF's; we can proceed with copying the bytes
   // from <src> into <dest> to complete the update.

   pwrite = (unsigned char xdata *) dest;

   EA = 0;                             // disable interrupts (precautionary)
   
   for (i = 0; i < length; i++) {
      if (*src != 0xFF) {              // exclude writing 0xff's for 
         PSCTL = 0x01;                 // MOVX writes target FLASH memory
                                       // efficiency
         FLKEY = 0xA5;                 // FLASH key sequence #1
         FLKEY = 0xF1;                 // FLASH key sequence #2
         *pwrite++ = *src++;           // copy bytes
      }  PSCTL = 0x00;                 // Disable FLASH Writes
   }
}

//-----------------------------------------------------------------------------
// FLASH_PageCopy
//-----------------------------------------------------------------------------
// 
// This routine copies the FLASH page starting at <src> to <dest>.  It erases
// the <dest> page before the copy process begins.
//
void FLASH_PageCopy (unsigned char code *src, unsigned char xdata *dest) reentrant
{  
   int i;                              // byte counter

   EA = 0;                             // disable interrupts (precautionary)
   PSCTL = 0x03;                       // MOVX writes target FLASH memory;
                                       // FLASH erase operations enabled

   FLKEY = 0xA5;                       // FLASH key sequence #1
   FLKEY = 0xF1;                       // FLASH key sequence #2
   *dest = 0;                          // initiate erasure of <dest> FLASH
                                       // page
   PSCTL = 0x00;                       // Disable FLASH Writes
   
   for (i = 0; i < 512; i++) {
      if (*src != 0xFF) {              // exclude writing 0xff's for efficiency

         PSCTL = 0x01;                 // disable FLASH erase operations;
                                       // MOVX writes target FLASH memory
         FLKEY = 0xA5;                 // FLASH key sequence #1
         FLKEY = 0xF1;                 // FLASH key sequence #2
         *dest = *src;                 // copy bytes
      }  PSCTL = 0x00;                 // Disable FLASH Writes
      dest++;                          // advance pointers
      src++;
   }
}

//-----------------------------------------------------------------------------
// CalibrateADCforMeasurement
//-----------------------------------------------------------------------------
// This routine assumes memory block 0x1A00 is erased
// Then this function calibrates the voltage channel and stores the calibration
// coefficients in the parameters xxx_slope and xxx_offset.
// This calibration routine uses the F350's internal calibration functions.  
//
void CalibrateADCforMeasurement(void)
{  unsigned char xdata *idata pwrite;  // FLASH write pointer
	
   EA = 0;                             // Disable All Interrupts

                                       // Perform Self Calibration 
   ADC0MD   = 0x84;                    // Enable ADC; Internal Offset Cal.
   while(!AD0CALC);                    // Wait for calibration to complete
     
   ADC0MD   = 0x85;                    // Enable ADC; Internal Gain Cal.
   while(!AD0CALC);                    // Wait for Calibration to complete

                                       // Memory's been erased at 0x1A00
                                       // Store Gain Coefficients to FLASH
   temp_LONG_1.l = 1234;               // Prepare storage parameter
   pwrite = (char xdata *)&(DATA_PAGE[scale_slope].l);
   FLASH_PageUpdate ((unsigned char *)&temp_LONG_1.b[0], pwrite, 4);

                                       // Memory's been erased at 0x1A00
                                       // Store the Offset Coeffs to FLASH
   temp_LONG_1.l = 1234;               // Prepare storage parameter
   pwrite = (char xdata *)&(DATA_PAGE[scale_offset].l);
   FLASH_PageUpdate ((signed char *)&temp_LONG_1.b[0], pwrite, 4);
}

//-----------------------------------------------------------------------------
// Calculate_One_Count
//-----------------------------------------------------------------------------
// This routine calculates the tare weight of the scale in digital counts
// Then this function assumes the user places 100 equal units to be weighed
// on scale.  From these two measurements the count of one unit is computed.
// From this, the number of units on the scales can be determined in future
// measurements
void Calculate_One_Count(void)
{  unsigned char xdata *idata pwrite;// FLASH write pointer
   char i;

   ADC0CN   = 0x00;                    // Unipolar; Gain = 1
   ADC0DECH = 0x04;                    // Set Output word rate at for ~23 Hz
   ADC0DECL = 0x00;                    // Set Output word rate at for ~23 Hz
   ADC0MUX  = VSCALE;                  // Select appropriate input for AMUX
   
   Tare_Count = 0;                     // Initialize to zero
   for(i=5;i;--i)                      // Average next 5 conversions
   {	
      temp_LONG_1.l = 0;
      AD0INT = 0;                      // Clear end-of-conversion indicator
      ADC0MD   = 0x82;                 // Enable ADC; Single conversions
      while(!AD0INT);                  // Wait for conversion to complete
      temp_LONG_1.l = ADC0H;
      temp_LONG_1.l = temp_LONG_1.l <<16;
	  temp_LONG_1.l += ADC0M <<8;
      temp_LONG_1.l += ADC0L ;
      Tare_Count += temp_LONG_1.l;
   }
   Tare_Count = Tare_Count/5;          // Store the Tare Digital Count Value

   Full_Counts = 0;                    // Initialize to zero
   for(i=5;i;--i)                      // Average next 5 conversions
   {	
      temp_LONG_1.l = 0;
      AD0INT = 0;                      // Clear end-of-conversion indicator
      ADC0MD   = 0x82;                 // Enable ADC; Single conversions
      while(!AD0INT);                  // Wait for conversion to complete
      temp_LONG_1.l = ADC0H;
      temp_LONG_1.l = temp_LONG_1.l <<16;
	  temp_LONG_1.l += ADC0M <<8;
      temp_LONG_1.l += ADC0L ;
      Full_Counts += temp_LONG_1.l;
   }

   Full_Counts = Full_Counts/ 5;       // Store the Calibration Count Value; 
                                       //     unually 100 units
   
   One_Count = Full_Counts - Tare_Count;
   One_Count = One_Count/100;          // Divide by 100 assumes there are 
                                       //     100 cal units on scale

   ADC0MD   = 0x00;                    // Turn off ADC Module

                                       // Memory's already been erased at 0x1A00
                                       // Store Gain Coefficients to FLASH
   temp_LONG_1.l = 1234;               // Prepare for FLASH Write
   pwrite = (char xdata *)&(DATA_PAGE[tare_count].l);
   FLASH_PageUpdate ((signed char *)&temp_LONG_1.b[0], pwrite, 4);

   temp_LONG_1.l = 1234;               // Prepare for FLASH Write
   pwrite = (char xdata *)&(DATA_PAGE[full_counts].l);
   FLASH_PageUpdate ((signed char *)&temp_LONG_1.b[0], pwrite, 4);

   temp_LONG_1.l = 1234;               // Prepare for FLASH Write
   pwrite = (char xdata *)&(DATA_PAGE[one_count].l);
   FLASH_PageUpdate ((signed char *)&temp_LONG_1.b[0], pwrite, 4);
}


//-----------------------------------------------------------------------------
// Monitor Weigh_Scale
//-----------------------------------------------------------------------------
// This routine configures the ADC's AMUX, acquires conversions and returns
// appropriate parameter (weight or temperature) via variable result.
// Weight is returned in Unit counts. Example 100 pennies is 100.
// Temperature is returned in degree Kelvin.  Ex 273 Kelvin returns a value 273
// 
int Monitor_Weigh_Scale(unsigned char value)
{  char i;
   unsigned long av =0,delay_count=0;
   long signed result;

   ADC0DECH = 0x04;                    // Set Output word rate at for 23 Hz
   ADC0DECL = 0x00;                    // Set Output word rate at for 23 Hz
 
   switch (value)
   {
      case TEMPERATURE:
         ADC0CN  = 0x00;               // Unipolar; Gain = 1
         ADC0MUX = TSCALE;             // Select appropriate input for AMUX	  
         break;
     
      case WEIGHT: 
         ADC0CN   = 0x00;              // Unipolar; Gain = 1
         ADC0MUX  = VSCALE;            // Select appropriate input for AMUX
         break;   
   }  
   //Compute average of next 5 A/D conversions
   av = 0;                             // Initialize to Zero
   for(i=5;i;--i)                      // Average next 5 conversions
   {	
      temp_LONG_1.l = 0;
      AD0INT = 0;                      // Clear end-of-conversion indicator
      ADC0MD   = 0x82;                 // Enable ADC; Single conversions
      while(!AD0INT);                  // Wait for conversion to complete
      temp_LONG_1.l = ADC0H;
      temp_LONG_1.l = temp_LONG_1.l <<16;
	  temp_LONG_1.l += ADC0M <<8;
      temp_LONG_1.l += ADC0L ;
      av += temp_LONG_1.l;
   }
   
   ADC0MD   = 0x00;                    // Turn off ADC Module

   av = av/5;                          // Compute the average
	
   switch (value)
   {	
      case TEMPERATURE:
         result =  (long) av /TEMP_SLOPE*1000; // Account for Temp. Slope
         result -= 100*TEMP_SENSOR_OFFSET;     // Account for Temp. Offset
         result = result + 27315;      // Convert to Degrees Kelvin 
         break;
      
      case WEIGHT:
        result = av - Tare_Count;
        result = result/One_Count;
        break;
   }
   return (signed int) result;  	
}

void Update_Display_Via_UART(void)
{
   // Send Information to Hyper Terminal at 9600 Baud,8,n,1
   //printf ("Temperature = %d hundredths degrees K\n", Temperature);
   printf ("Counts = %d  units\n", (int)Weight);
}
//-----------------------------------------------------------------------------
// Update_Display_Via_LCD
//-----------------------------------------------------------------------------
// This function provides starter algorithms for the HT1620,(Holtek) LCD driver.
// It provides algorithms to write to te HT1620 and Read from the HT1620.
// Specific algorithms to activate and deactivate the elements on the LCD that 
// correspond to specific letters and numerals will need to be develop.
//
void Update_Display_Via_LCD(void)
{  unsigned char BIT_count; // counter for SPI transaction

   // Here is an example of a write command to the HT1620
   Data_Word = 0x1400;                 // Prepare information for writing to LCD
   Data_Word = Data_Word << 3;         // Prepare information for writing to LCD
   CS   = 0;                           // Select Serial Port
   for (BIT_count = 13; BIT_count > 0;BIT_count--)// 13 bits 
   {
      DATA = Data_Word & 0x8000;       // put current outgoing bit on DATA
      Data_Word = Data_Word <<1;       // shift next bit into MSB
      
      WR = 0x01;                       //set sck high

      WR = 0x00;                       // set sck low
   } 

   CS   = 1;                           // Deselect LCD

   // Here is an example of a read command to the HT1620

   Data_Word = 0x1800;                 // Prepare information for writing to LCD
   Data_Word = Data_Word << 3;         // Prepare information for writing to LCD
   CS   = 0;                           // Select LCD
   for (BIT_count = 9; BIT_count > 0;BIT_count--)// 9 bits 
   {
      DATA = Data_Word & 0x8000;       // put current outgoing bit on DATA
      Data_Word = Data_Word <<1;       // shift next bit into MSB

      WR = 0x01;                       // set sck high

      Data_Word |= DATA;               // capture current bit on DATA
      
      WR = 0x00;                       // set sck low
   } 
   for (BIT_count = 4; BIT_count > 0;BIT_count--)// 4 bits 
   {
      DATA = Data_Word & 0x8000;       // put current outgoing bit on DATA
      Data_Word = Data_Word <<1;       // shift next bit into MSB
      
      RD = 0x01;                       // set sck high
      
      Data_Word |= DATA;               // capture current bit on DATA
      
      RD = 0x00;                       // set sck low
   } 

   CS   = 1;                           // Deselect LCD   
}

//-----------------------------------------------------------------------------
// Main Function
//-----------------------------------------------------------------------------
// - Main calls all the functions necessary to configure the C8051F350.
// 		It also calls routines to calibrate the electronic scale.  Once setup
//      and calibration are complete, Main() determines the Unit count on 
//      the scale and outputs via the UART. 
//
//      System Calibration is only performed the first time through the cycle.
//
void main(void)
{  
   Config_F350();                      // Config F350

   CalibrateADCforMeasurement();       // Calibrate ADC

   Calculate_One_Count();              // Calculate the size of one count
   
   // Update Temperature  and Bridge Monitoring Algorithms
   while(1)
   {                                   // Once pressed get temperature and weight
	  Temperature = Monitor_Weigh_Scale (TEMPERATURE);// Get Latest Temperature 
                                       // Acquire Pack's Voltages
      Weight = Monitor_Weigh_Scale (WEIGHT);
 
                                       // Update PC Via UART      
      Update_Display_Via_UART();
                                       // Update PC Via LCD     
      //Update_Display_Via_LCD();	  
   }
}
// END of File