//-----------------------------------------------------------------------------
// C8051F12x_FFT_Demo.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
// --------------------
//
// This program generates tones on the DAC, collects that data using ADC0 at
// <SAMPLE_RATE> Hz and performs an FFT on the data.  The Real and Imaginary
// parts of the results are then sent to the UART peripheral at <BAUDRATE> bps,
// where they can be displayed or captured using a terminal (ToolStick
// Terminal) program.
//
// This program uses the MAC to calculate both the Windowing and the FFT.
// Please note that if the MAC is used in any interrupts while the FFT is being
// calculated, steps must be taken to save the FFT data before the MAC is used.
//
// This program uses the internal oscillator multiplied by 4 for an effective
// SYSCLK of 98 Mhz. This program also initializes and uses UART0 at <BAUDRATE>
// bits per second.
//
// The FFT_Code_Tables_MAC.h file must be in the same folder as this file
// when compiling.
//
// Pinout:
//
// P0.0 - UART1 TX pin
// P0.1 - UART1 RX pin
//
// DAC0 - DAC0 output
//
// AIN0.0 - ADC0 input
//
// Note: DAC0 and AIN0.0 must be tied together.
//
// How To Test:
// ------------
//
// 1) Change the program parameters as necessary:
//       <SYSCLK> is the selected SYSCLK rate in Hz
//       <BAUDRATE> is the UART baud rate
//       <SAMPLE_RATE> is the ADC sample rate
//       <NUM_BITS> is the number of bits of data
//       <DATA_BEGIN> is the address in XRAM where the Real and Imag arrays
//          start
//       <RUN_ONCE> determines whether the FFT will be performed continuously
//
//       <NUM_FFT> in FFT_Code_Tables_MAC.h is the number of points in the FFT
//       <WINDOW_TYPE> in FFT_Code_Tables_MAC.h is the window type as described
//
//       <SAMPLE_RATE_DAC> is the output rate of the IDAC
//       tones can be modified as shown
// 2) Compile the code with FFT_Code_Tables_MAC.h in the folder.
// 3) Download the code to an C8051F12x device.  This code was specifically
//    written for the 'F12x Target Board. (Alternatively, download
//    the precompiled hex file C8051F12x_FFT_Demo.hex).
// 4) Connect DAC0 and AIN0.0 together (DAC0, pin 3 on J1, to
//    AIN0.0, pin 7 on J1, on the 'F12x Target Board).
// 5) Connect to the 'F12x using a Terminal program.  The output of
//    the UART may need to be saved to a file because of its length.
// 6) Use the FFT_graph.xls Excel file to graph the output of the FFT.
//
//
// Target:         C8051F12x ('F12x Target Board)
// Tool chain:     Keil C51 7.50 / SDCC 7.1
// Command Line:   None
//
// Release 1.1
//    -Modified to support both SDCC and Keil (TP)
//    -16 JUL 2007
//
// Release 1.0
//    -Initial Revision (BD / TP)
//    -31 MAY 2007
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "si_toolchain.h"             // Compiler-specific declarations
                                       // (Keil/SDCC)
#include "C8051F120_defs.h"            // SFR declarations
#include <stdio.h>
#include "FFT_Code_Tables_MAC.h"       // Code Tables for FFT routines

//-----------------------------------------------------------------------------
// Global Constants and Variable Type Definitions
//-----------------------------------------------------------------------------

// System Parameters
#define INTOSC       24500000          // Internal 24.5 MHz Oscillator
#define SYSCLK       98000000          // Output of PLL derived from
                                       // (INTOSC*4/1)
#define BAUDRATE     115200            // Baud Rate for UART0

#define SAMPLE_RATE  10000             // Sample frequency in Hz

#define NUM_BITS     16                // Number of Bits in Data

#define DATA_BEGIN   0x0000            // Beginning of XRAM Data

#define RUN_ONCE     1                 // Setting to a non-zero value will
                                       // cause the program to stop after one
                                       // data set.

// For frequency generation
#define SAMPLE_RATE_DAC   100000L      // Update rate of DAC in Hz

#define PHASE_PRECISION   65536L       // Range of phase accumulator

#define LOW_697           697L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_770           770L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_852           852L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_941           941L * PHASE_PRECISION / SAMPLE_RATE_DAC

#define HIGH_1209         1209L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1336         1336L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1477         1477L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1633         1633L * PHASE_PRECISION / SAMPLE_RATE_DAC

char code SINE_TABLE[256] = {
   0x00, 0x03, 0x06, 0x09, 0x0c, 0x0f, 0x12, 0x15,
   0x18, 0x1c, 0x1f, 0x22, 0x25, 0x28, 0x2b, 0x2e,
   0x30, 0x33, 0x36, 0x39, 0x3c, 0x3f, 0x41, 0x44,
   0x47, 0x49, 0x4c, 0x4e, 0x51, 0x53, 0x55, 0x58,
   0x5a, 0x5c, 0x5e, 0x60, 0x62, 0x64, 0x66, 0x68,
   0x6a, 0x6c, 0x6d, 0x6f, 0x70, 0x72, 0x73, 0x75,
   0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7c,
   0x7d, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
   0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7e, 0x7e,
   0x7d, 0x7c, 0x7c, 0x7b, 0x7a, 0x79, 0x78, 0x77,
   0x76, 0x75, 0x73, 0x72, 0x70, 0x6f, 0x6d, 0x6c,
   0x6a, 0x68, 0x66, 0x64, 0x62, 0x60, 0x5e, 0x5c,
   0x5a, 0x58, 0x55, 0x53, 0x51, 0x4e, 0x4c, 0x49,
   0x47, 0x44, 0x41, 0x3f, 0x3c, 0x39, 0x36, 0x33,
   0x30, 0x2e, 0x2b, 0x28, 0x25, 0x22, 0x1f, 0x1c,
   0x18, 0x15, 0x12, 0x0f, 0x0c, 0x09, 0x06, 0x03,
   0x00, 0xfd, 0xfa, 0xf7, 0xf4, 0xf1, 0xee, 0xeb,
   0xe8, 0xe4, 0xe1, 0xde, 0xdb, 0xd8, 0xd5, 0xd2,
   0xd0, 0xcd, 0xca, 0xc7, 0xc4, 0xc1, 0xbf, 0xbc,
   0xb9, 0xb7, 0xb4, 0xB2, 0xaf, 0xad, 0xab, 0xa8,
   0xa6, 0xa4, 0xa2, 0xa0, 0x9e, 0x9c, 0x9a, 0x98,
   0x96, 0x94, 0x93, 0x91, 0x90, 0x8e, 0x8d, 0x8b,
   0x8a, 0x89, 0x88, 0x87, 0x86, 0x85, 0x84, 0x84,
   0x83, 0x82, 0x82, 0x81, 0x81, 0x81, 0x81, 0x81,
   0x80, 0x81, 0x81, 0x81, 0x81, 0x81, 0x82, 0x82,
   0x83, 0x84, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
   0x8a, 0x8b, 0x8d, 0x8e, 0x90, 0x91, 0x93, 0x94,
   0x96, 0x98, 0x9a, 0x9c, 0x9e, 0xa0, 0xa2, 0xa4,
   0xa6, 0xa8, 0xab, 0xad, 0xaf, 0xB2, 0xb4, 0xb7,
   0xb9, 0xbc, 0xbf, 0xc1, 0xc4, 0xc7, 0xca, 0xcd,
   0xd0, 0xd2, 0xd5, 0xd8, 0xdb, 0xde, 0xe1, 0xe4,
   0xe8, 0xeb, 0xee, 0xf1, 0xf4, 0xf7, 0xfa, 0xfd
};

//-----------------------------------------------------------------------------
// Macros
//-----------------------------------------------------------------------------

// FFT_AplusB calculates:
//
// TempL.l = <A>.i + <B>.i;
// if ((TempL.l < 0)&&(0x01 & TempL.u8[3]))
// {
//    <result>.i = (TempL.l >> 1) + 1;
// }
// else
// {
//    <result>.i = TempL.l >> 1;
// }
//
// by calculating: -TempL.l = -A.i - TempReB.i;
// and using the MAC Rounding registers
//
// In C using the MAC:
//
// MAC0A = 0x8000;                     // Load the MAC0A register with "-1"
// MAC0BH = A.u8[MSB];                 // Multiply <A> * -1
// MAC0BL = A.u8[LSB];
//
// MAC0BH = B.u8[MSB];                 // Multiply <B> * -1
// MAC0BL = B.u8[LSB];
//
// NOP();                              // Wait for the multiplications and
// NOP();                              // accumulation to finish
//
// MAC0CF |= 0x01;                     // Switch the MAC to multiplication only
//
// MAC0BH = MAC0ACC3;                  // Multiply the result by "-1"
// MAC0BL = MAC0ACC2;
//
// NOP();                              // Wait for the multiplication to
// NOP();                              // finish
//
// MAC0CF |= 0x20;                     // Right shift the contents of the ACC
//
// NOP();
// NOP();
//
// <result> = MAC0RND;                 // Store the value of the Round register
//
// MAC0CF &= ~0x01;                    // Switch back to mult + acc mode
// MAC0CF |= 0x08;                     // Clear the ACC
//
// NOTE:
//   <A>, <B> must be of the SI_UU16_t type
//   <result> must be an int or long
#define FFT_AplusB(A, B, result) \
   MAC0A = 0x8000; \
   MAC0BH = A.u8[MSB]; \
   MAC0BL = A.u8[LSB]; \
   MAC0BH = B.u8[MSB]; \
   MAC0BL = B.u8[LSB]; \
   NOP(); \
   NOP(); \
   MAC0CF |= 0x01; \
   MAC0BH = MAC0ACC3; \
   MAC0BL = MAC0ACC2; \
   NOP(); \
   NOP(); \
   MAC0CF |= 0x20; \
   NOP(); \
   NOP(); \
   result = MAC0RND; \
   MAC0CF &= ~0x01; \
   MAC0CF |= 0x08;

// FFT_AminusB calculates:
//
// TempL.l = <A>.i - <B>.i;
// if ((TempL.l < 0)&&(0x01 & TempL.u8[3]))
// {
//    <result>.i = (TempL.l >> 1) + 1;
// }
// else
// {
//    <result>.i = TempL.l >> 1;
// }
//
// by calculating: -TempL.l = -TempReA.i - (-TempReB.i);
// and using the MAC Rounding registers
//
// In C using the MAC:
//
// MAC0A = 0x8000;                     // Load the MAC0A register with "-1"
// MAC0BH = B.u8[MSB];                 // Multiply <B> * -1
// MAC0BL = B.u8[LSB];
//
// NOP();                              // Wait for the multiplication to
// NOP();                              // finish
//
// temp.u8[MSB] = MAC0ACC3;            // Store -<B> in <temp>
// temp.u8[LSB] = MAC0ACC2;
//
// MAC0CF |= 0x08;                     // Clear the ACC
//
// MAC0BH = A.u8[MSB];                 // Multiply <A> * -1
// MAC0BL = A.u8[LSB];
//
// MAC0BH = temp.u8[MSB];              // Multiply <temp> (-<B>) * -1
// MAC0BL = temp.u8[LSB];
//
// NOP();                              // Wait for the multiplications and
// NOP();                              // accumulation to finish
//
// MAC0CF |= 0x01;                     // Switch the MAC to multiplication only
//
// MAC0BH = MAC0ACC3;                  // Multiply the result by "-1"
// MAC0BL = MAC0ACC2;
//
// NOP();                              // Wait for the multiplication to
// NOP();                              // finish
//
// MAC0CF |= 0x20;                     // Right shift the contents of the ACC
//
// NOP();
// NOP();
//
// result = MAC0RND;                   // Store the value of the Round register
//
// MAC0CF &= ~0x01;                    // Switch back to mult + acc mode
// MAC0CF |= 0x08;                     // Clear the ACC
//
// NOTE:
//   <A>, <B> must be of the SI_UU16_t type
//   <result> must be an int or long
//   <temp> must be of the SI_UU16_t type and declared in the calling function
#define FFT_AminusB(A, B, result) \
   MAC0A = 0x8000; \
   MAC0BH = B.u8[MSB]; \
   MAC0BL = B.u8[LSB]; \
   NOP(); \
   NOP(); \
   temp.u8[MSB] = MAC0ACC3; \
   temp.u8[LSB] = MAC0ACC2; \
   MAC0CF |= 0x08; \
   MAC0BH = A.u8[MSB]; \
   MAC0BL = A.u8[LSB]; \
   MAC0BH = temp.u8[MSB]; \
   MAC0BL = temp.u8[LSB]; \
   NOP(); \
   NOP(); \
   MAC0CF |= 0x01; \
   MAC0BH = MAC0ACC3; \
   MAC0BL = MAC0ACC2; \
   NOP(); \
   NOP(); \
   MAC0CF |= 0x20; \
   NOP(); \
   NOP(); \
   result = MAC0RND; \
   MAC0CF &= ~0x01; \
   MAC0CF |= 0x08;

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------

// XRAM storage of FFT: requires NUM_FFT*4 Bytes after DATA_BEGIN address
// 16 bits of data (so 2 bytes per data point) and Real and Imaginary arrays
// = 4*NUM_FFT
//SI_UU16_t xdata Real[NUM_FFT] _at_ DATA_BEGIN;
//SI_UU16_t xdata Imag[NUM_FFT] _at_ (DATA_BEGIN + (NUM_FFT * 2));
SI_LOCATED_VARIABLE_NO_INIT(Real[NUM_FFT], SI_UU16_t, xdata, DATA_BEGIN);
SI_LOCATED_VARIABLE_NO_INIT(Imag[NUM_FFT], SI_UU16_t, xdata, (DATA_BEGIN + (NUM_FFT * 2)));


// NUM_FFT is defined in the "FFT_Code_Tables_MAC.h" header file
#if (NUM_FFT < 256)
unsigned char index, ADC_Index;
#endif

// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.
#if (NUM_FFT >= 256)
unsigned int index, ADC_Index;
#endif

unsigned int BinNum;

unsigned int Input_Offset;

bit Conversion_Set_Complete;           // This indicates when the data has been
                                       // stored and is ready to be processed
                                       // using the FFT routines

// For frequency tone...
unsigned int phase_add1;               // Holds low-tone phase adder
unsigned int phase_add2;               // Holds low-tone phase adder

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);
void Port_Init (void);
void UART1_Init (void);
void ADC0_Init (void);
void DAC0_Init(void);
void MAC0_Init (void);
void Timer3_Init (int counts);
void Timer4_Init(int counts);

void WindowCalc(SI_UU16_t Win_Array[], bit Remove_DC);
void Bit_Reverse(SI_UU16_t BR_Array[]);
void Int_FFT(SI_UU16_t ReArray[], SI_UU16_t ImArray[]);

SI_INTERRUPT_PROTO_USING(Timer4_ISR, INTERRUPT_TIMER4, 3);
SI_INTERRUPT_PROTO_USING(ADC0_ISR, INTERRUPT_ADC0_EOC, 2);

// Define the UART printing functions
#if defined __C51__
char putchar (char c);                 // Define putchar for Keil
#elif defined SDCC
void putchar (char c);
#endif // defined SDCC

// Similar to STARTUP.A51 for Keil, this function stub for SDCC allows us to
// disable the WDT before memory is initialized.
#if defined SDCC
void _sdcc_external_startup (void);

void _sdcc_external_startup (void)
{
   WDTCN = 0xDE;                       // Disable watchdog timer
   WDTCN = 0xAD;
}
#endif // defined SDCC

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main()
{
   WDTCN = 0xDE;                       // Disable watchdog timer
   WDTCN = 0xAD;

   SYSCLK_Init();                      // Initialize external clock and PLL
   Port_Init ();                       // Set up Port I/O
   UART1_Init ();                      // Initialize UART1
   MAC0_Init ();                       // Initialize the MAC
   Timer3_Init (SYSCLK/SAMPLE_RATE);   // Initialize Timer3 to overflow at
                                       // <SAMPLE_RATE>
   Timer4_Init (SYSCLK/SAMPLE_RATE_DAC); // Initialize Timer4 to overflow at
                                       // <SAMPLE_RATE_DAC>
   ADC0_Init ();                       // Initialize ADC0
   DAC0_Init ();

   phase_add1 = LOW_770;
   phase_add2 = HIGH_1336;

   EA = 1;                             // Globally enable interrupts

   SFRPAGE = LEGACY_PAGE;

   while (1)
   {
      ADC_Index = 0;
      Conversion_Set_Complete = 0;

      EIE2 |= 0x06;                    // Enable ADC interrupts
                                       // Timer4 interrupt enable
                                       // ADC and DAC interrupts are disabled
                                       // in the ADC ISR after N samples are
                                       // collected

      while(!Conversion_Set_Complete);

      WindowCalc(Real, 1);             // Window Real Data, and remove
                                       // the DC information in the FFT

      Bit_Reverse(Real);               // Sort Real (Input) Data in bit-reverse
                                       // order

      Int_FFT(Real, Imag);             // Perform FFT on data


      // Print the output of the FFT
      printf("Bin\tReal\tImag\n");
      for (BinNum = 0; BinNum < NUM_FFT/2; BinNum++)
      {
         // Print Data in the format: Bin <tab> Real <tab> Imaginary
         printf("%d\t%d\t%d\n", BinNum, Real[BinNum].s16, Imag[BinNum].s16);
      }

      // If RUN_ONCE is set, loop infinitely
      if (RUN_ONCE)
      {
         while(1);
      }
   }
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// This routine initializes the system clock to use the internal 24.5MHz*4
// oscillator as its clock source.
//
//-----------------------------------------------------------------------------
void SYSCLK_Init (void)
{
   char i;
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   OSCICN = 0x83;

   // Step 2. Set the PLLSRC bit (PLL0CN.2) to select the desired
   // clock source for the PLL.
   PLL0CN &= ~0x04;                    // Internal oscillator

   // Step 3. Program the Flash read timing bits, FLRT (FLSCL.5-4) to the
   // appropriate value for the new clock rate (see Section 15. Flash Memory
   // on page 199).
   SFRPAGE = LEGACY_PAGE;
   FLSCL |= 0x30;                      // >= 100 MHz
   SFRPAGE = CONFIG_PAGE;

   // Step 4. Enable power to the PLL by setting PLLPWR (PLL0CN.0) to 1.
   PLL0CN |= 0x01;

   // Step 5. Program the PLL0DIV register to produce the divided reference
   // frequency to the PLL.
   PLL0DIV = 0x01;

   // Step 6. Program the PLLLP3-0 bits (PLL0FLT.3-0) to the appropriate
   // range for the divided reference frequency.
   PLL0FLT |= 0x01;

   // Step 7. Program the PLLICO1-0 bits (PLL0FLT.5-4) to the appropriate
   // range for the PLL output frequency.
   PLL0FLT &= ~0x30;

   // Step 8. Program the PLL0MUL register to the desired clock multiplication
   // factor.
   PLL0MUL = 0x04;

   // Step 9. Wait at least 5 s, to provide a fast frequency lock.
   for (i = 100; i > 0; i--);

   // Step 10. Enable the PLL by setting PLLEN (PLL0CN.1) to 1.
   PLL0CN |= 0x02;

   // Step 11. Poll PLLLCK (PLL0CN.4) until it changes from 0 to 1.
   while ((PLL0CN & 0x10) != 0x10);

   // Step 12. Switch the System Clock source to the PLL using the CLKSEL
   // register.
   CLKSEL = 0x02;

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Port_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the Crossbar and GPIO ports.
//
// Pinout:
//
// P0.0 - UART TX1 (push-pull)
// P0.1 - UART RX1
//
// DAC0 - DAC0 output
//
// AIN0.0 - ADC0 analog input
//
// Note: DAC0 and AIN0.0 must be tied together.
//
//-----------------------------------------------------------------------------
void Port_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   XBR0     = 0x00;
   XBR1     = 0x00;
   XBR2     = 0x44;                    // Enable crossbar and weak pull-up
                                       // Enable UART1

   P0MDOUT |= 0x01;                    // Set TX1 pin to push-pull

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// UART1_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the UART1 using Timer1, for <BAUDRATE> and 8-N-1.
//
//-----------------------------------------------------------------------------
void UART1_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = UART1_PAGE;               // Switch to the necessary SFRPAGE

   SCON1 = 0x10;                       // SCON1: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits

   SFRPAGE = TIMER01_PAGE;

   TMOD   &= ~0xF0;                    // Timer 1, Mode 2, 8-bit reload
   TMOD   |=  0x20;

   if (SYSCLK/BAUDRATE/2/256 < 1)
   {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON |=  0x08;                  // T1M = 1; SCA1:0 = xx
   }
   else if (SYSCLK/BAUDRATE/2/256 < 4)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 01
      CKCON |=  0x01;
   }
   else if (SYSCLK/BAUDRATE/2/256 < 12)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 00
   }
   else if (SYSCLK/BAUDRATE/2/256 < 48)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 10
      CKCON |=  0x02;
   }
   else
   {
      while (1);                       // Error.  Unsupported baud rate
   }

   TL1 = TH1;                          // Init Timer1
   TR1 = 1;                            // START Timer1

   SFRPAGE = UART1_PAGE;

   TI1 = 1;                            // Indicate TX1 ready

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure ADC0 to use Timer2 overflows as conversion source, to
// generate an interrupt on conversion complete, and to use left-justified
// output mode.  Enables ADC0.
//
//-----------------------------------------------------------------------------
void ADC0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = ADC0_PAGE;

   ADC0CN = 0x05;                      // ADC0 disabled; normal tracking
                                       // mode; ADC0 conversions are initiated
                                       // on overflow of Timer3; ADC0 data is
                                       // left-justified

   REF0CN = 0x03;                      // Enable on-chip VREF and VREF output
                                       // buffer

   AMX0SL = 0x00;                      // Select AIN0.0 as ADC mux input

   ADC0CF = (SYSCLK/2500000) << 3;     // ADC conversion clock = 2.5MHz

   AD0EN = 1;                          // Enable ADC0

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// DAC0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure DAC0 to update on Timer4 overflows.  VREF is already enabled by
// the ADC initialization code.
//
//-----------------------------------------------------------------------------
void DAC0_Init(void){

   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = DAC0_PAGE;

   DAC0CN = 0x94;                      // Enable DAC0 in left-justified mode
                                       // managed by Timer4 overflows

   DAC0L = 0x00;                       // Initialize IDA0L to '0' for "8-bit"
                                       // IDAC mode

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// MAC0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Initialize the MAC for use in the Windowing and FFT calculations.
//
//-----------------------------------------------------------------------------
void MAC0_Init(void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = MAC0_PAGE;                // Switch to the necessary SFRPAGE

   MAC0CF = 0x00;                      // Integer mode;
                                       // Multiply and Accumulate

   MAC0CF |= 0x08;                     // Clear the accumulator

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:
//   1) counts - the number of timer clocks to count before a timer interrupt
//           should occur
//
// Configure Timer3 to auto-reload at interval specified by <counts> (no
// interrupt generated) using SYSCLK as its time base.
//
// Timer 3 overflow automatically triggers ADC0 conversion.
//
//-----------------------------------------------------------------------------
void Timer3_Init (int counts)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = TMR3_PAGE;

   TMR3CN = 0x00;                      // Stop Timer3; Clear TF3;
   TMR3CF = 0x08;                      // use SYSCLK as timebase

   RCAP3   = -counts;                  // Init reload values
   TMR3    = RCAP3;                    // set to reload immediately
   EIE2   &= ~0x01;                    // Disable Timer3 interrupts
   TR3 = 1;                            // Start Timer3

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// Timer4_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:
//   1) counts - the number of timer clocks to count before a timer interrupt
//           should occur
//
// Configure Timer4 to auto-reload mode and to generate interrupts
// at intervals specified in <counts> using SYSCLK as its time base.
//
// Timer 4 overflow controls the DAC update rate.
//
//-----------------------------------------------------------------------------
void Timer4_Init (int counts)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = TMR4_PAGE;

   TMR4CN = 0x00;                   // Stop Timer4; Clear overflow flag (TF4);
                                    // Set to Auto-Reload Mode

   TMR4CF = 0x08;                   // Configure Timer4 to increment;
                                    // Timer4 counts SYSCLKs

   RCAP4 = -counts;                 // Set reload value
   TMR4 = RCAP4;                    // Initialzie Timer4 to reload value

   TR4 = 1;                         // Start Timer4

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// ADC0_ISR
//-----------------------------------------------------------------------------
//
// The ADC sample is stored in memory, and an index variable is incremented.
// If enough samples have been taken to process the FFT, then a flag is set,
// and ADC interrupts are disabled until the next set is requested.
//
//-----------------------------------------------------------------------------
//void ADC0_ISR (void) interrupt 15
SI_INTERRUPT_USING(ADC0_ISR, INTERRUPT_ADC0_EOC, 2)
{
   static unsigned int ADC_Max_Input = 0x0000;
   static unsigned int ADC_Min_Input = 0xFFFF;

   AD0INT = 0;                         // Clear ADC conversion complete
                                       // flag

   Real[ADC_Index].u16 = ADC0;         // Store ADC value

   // Determine the minimum and maximum inputs values so that the Windowing
   // routine can properly remove the DC offset.
   if (Real[ADC_Index].u16 > ADC_Max_Input)
   {
      ADC_Max_Input = Real[ADC_Index].u16;
   }
   else if (Real[ADC_Index].u16 < ADC_Min_Input)
   {
      ADC_Min_Input = Real[ADC_Index].u16;
   }

   ADC_Index++;                        // Increment the index into memory

   if (ADC_Index >= NUM_FFT)           // If enough samples have been collected
   {
      Input_Offset = ((ADC_Max_Input - ADC_Min_Input)/2) + ADC_Min_Input;

      Conversion_Set_Complete = 1;     // Tell the Main Routine and...
      EIE2 &= ~0x06;                   // Disable ADC interrupts
                                       // Disable IDAC interrupts
   }
}

//-----------------------------------------------------------------------------
// Timer4_ISR
//-----------------------------------------------------------------------------
//
// This ISR is called on Timer4 overflows.  Timer4 is set to auto-reload mode
// and is used to schedule the DAC output sample rate in this example.
// Note that the value that is written to the DAC during this ISR call is
// actually transferred to the DAC at the next Timer4 overflow.
//
// Note: DTMF "twist" is not generated in this routine.  This feature can be
// added by scaling the high tone or low tone before outputting the value on
// the DAC.
//
//-----------------------------------------------------------------------------
//void Timer4_ISR (void) interrupt 16
SI_INTERRUPT_USING(Timer4_ISR, INTERRUPT_TIMER4, 3)
{
   static SI_UU16_t phase_acc1 = {0};       // Holds low-tone phase accumulator
   static SI_UU16_t phase_acc2 = {0};       // Holds high-tone phase accumulator

   unsigned int temp1;                 // Temp values for table results
   unsigned int temp2;

   TMR3CN &= ~0x80;                    // Clear Timer3 overflow flag

   phase_acc1.u16 += phase_add1;       // Update phase acc1 (low tone)
   temp1 = SINE_TABLE[phase_acc1.u8[MSB]];

   phase_acc2.u16 += phase_add2;       // Update phase acc2 (high tone)

   // Read the table value
   temp2 = SINE_TABLE[phase_acc2.u8[MSB]];

   SFRPAGE = DAC0_PAGE;

   // Now update the DAC value.  Note: the XOR with 0x80 translates
   // the bipolar table look-up into a unipolar quantity.
   DAC0H = 0x80 ^ ((temp1 >> 1) + (temp2 >> 1));
}

//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// WindowCalc
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:
//    1) SI_UU16_t Win_Array Real[] - array of the input data from the ADC
//    2) unsigned char Remove_DC - if Remove_DC is 1, the input data will be
//             centered around 0 to remove the DC component.
//
// Uses the values in WindowFunc[] to window the stored data
// (in FFT_Code_Tables_MAC.h).
//
// The WindowFunc[] Array contains window coefficients between samples
// 0 and (NUM_FFT/2)-1, and samples from NUM_FFT/2 to NUM_FFT-1 are the mirror
// image of the other side.
//
// Window values are interpreted as a fraction of 1 (WindowFunc[x]/65536).
// The value at NUM_FFT/2 is assumed to be 1.0 (65536).
//
// If Remove_DC = 1, the input data is move to center around 0 to cancel the DC
// offset.  To do this, the ADC ISR records the maximum and minimum ADC data,
// calculates the middle point of the input waveform.  This value is then
// subtracted from the single-ended ADC inputs.
//
//-----------------------------------------------------------------------------
void WindowCalc(SI_UU16_t Win_Array[], bit Remove_DC)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

#if (WINDOW_TYPE != 0)                 // Use this section if a window has been
                                       // specified

   SFRPAGE = MAC0_PAGE;                // Switch to the necessary SFRPAGE

   for (index = 0; index < NUM_FFT/2; index++)
   {
      // Array positions 1 to (NUM_FFT/2 - 1)
      if (Remove_DC)
      {
         Win_Array[index].u16 -= Input_Offset;
      }
      else
      {
         Win_Array[index].u16 = (unsigned)Win_Array[index].u16 >> 1;
      }

      MAC0A = WindowFunc[index];
      MAC0BH = Win_Array[index].u8[MSB];
      MAC0BL = Win_Array[index].u8[LSB];

      NOP();
      NOP();                           // ACC results available here
      NOP();

      if (!Remove_DC)
      {
         MAC0CF |= 0x20;
      }

      Win_Array[index].u16 = MAC0RND;

      MAC0CF |= 0x08;

      // Array positions (NUM_FFT/2 + 1) to (NUM_FFT - 1)
      if (Remove_DC)
      {
         Win_Array[NUM_FFT-index-1].u16 -= Input_Offset;
      }
      else
      {
         Win_Array[NUM_FFT-index-1].u16 = (unsigned)Win_Array[NUM_FFT-index-1].u16
                                      >> 1;
      }

      MAC0BH = Win_Array[NUM_FFT-index-1].u8[MSB];
      MAC0BL = Win_Array[NUM_FFT-index-1].u8[LSB];

      NOP();
      NOP();                           // ACC results available here
      NOP();

      if (!Remove_DC)
      {
         MAC0CF |= 0x20;
      }

      Win_Array[NUM_FFT-index-1].u16 = MAC0RND;

      MAC0CF |= 0x08;
   }

#endif

#if (WINDOW_TYPE == 0)                 // Compile this if no window has been
                                       // specified
   if (Remove_DC)
   {
      for (index = 0; index < NUM_FFT; index++)
      {
         Win_Array[index].u16 -= Input_Offset;
      }
   }
#endif

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// Bit_Reverse
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:
//    1) SI_UU16_t BR_Array[] - array to be bit-reversed in order.
//
// Sorts data in Bit Reversed Address order
//
// The BRTable[] array is used to find which values must be swapped.  Only
// half of this array is stored, to save code space.  The second half is
// assumed to be a mirror image of the first half.
//
//-----------------------------------------------------------------------------
void Bit_Reverse(SI_UU16_t BR_Array[])
{

// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.
#if (NUM_FFT >= 512)
   unsigned int swapA, swapB, sw_cnt;  // Swap Indices
#endif

#if (NUM_FFT <= 256)
   unsigned char swapA, swapB, sw_cnt; // Swap Indices
#endif

   int TempStore;

   // Loop through locations to swap
   for (sw_cnt = 1; sw_cnt < NUM_FFT/2; sw_cnt++)
   {
      swapA = sw_cnt;                  // Store current location
      swapB = BRTable[sw_cnt] * 2;     // Retrieve bit-reversed index
      if (swapB > swapA)               // If the bit-reversed index is
      {                                // larger than the current index,
         TempStore = BR_Array[swapA].u16; // the two data locations are
         BR_Array[swapA].u16 = BR_Array[swapB].u16; // swapped. Using this
         BR_Array[swapB].u16 = TempStore;  // comparison ensures that locations
      }                                // are only swapped once, and never with
                                       // themselves

      swapA += NUM_FFT/2;              // Now perform the same operations
      swapB++;                         // on the second half of the data
      if (swapB > swapA)
      {
         TempStore = BR_Array[swapA].u16;
         BR_Array[swapA].u16 = BR_Array[swapB].u16;
         BR_Array[swapB].u16 = TempStore;
      }
   }
}

//-----------------------------------------------------------------------------
// Int_FFT
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:
//    1) SI_UU16_t ReArray[] - array of Real data values
//    2) SI_UU16_t ImArray[] - array of Imaginary data values
//
// Performs a Radix-2 Decimation-In-Time FFT on the input array ReArray[]
//
// During each stage of the FFT, the values are calculated using a set of
// "Butterfly" equations, as listed below:
//
// ReA = ReA + (Cos(x)*ReB + Sin(x)*ImB)
// ReB = ReA - (Cos(x)*ReB + Sin(x)*ImB)
// ImA = ImA + (Cos(x)*ImB - Sin(x)*ReB)
// ImB = ImA - (Cos(x)*ImB - Sin(x)*ReB)
//
// The routine implements this calculation using the MAC and the following
// values:
//
// ReA = ReArray[indexA], ReB = ReArray[indexB]
// ImA = ImArray[indexA], ImB = ImArray[indexB]
// x = the angle: 2*pi*(sin_index/NUM_FFT), in radians.  The necessary values
//    are stored in code space in the SinTable[] array.
//
//
// Key Points for using this FFT routine:
//
// 1) It expects REAL data (in ReArray[]), in 2's complement, 16-bit binary
//    format and assumes an initial value of 0 for all imaginary locations
//    (in ImArray[]).
//
// 2) It expects the REAL input data to be sorted in bit-reversed index order.
//
// 3) SIN and COS values are retrieved and calculated from a table consisting
//    of 1/4 of a period of a SIN function.
//
// 4) It is optimized to use integer math only (no floating-point operations),
//    and for storage space.  The input, all intermediate stages, and the
//    output of the FFT are stored as 16-bit INTEGER values. This limits the
//    precision of the routine.  When using input data of less than 16-bits,
//    the best results are produced by left-justifying the data prior to
//    windowing and performing the FFT.
//
// 5) The algorithm is a Radix-2 type, meaning that the number of samples must
//    be 2^N, where N is an integer.  The minimum number of samples to process
//    is 4.  The constant NUM_FFT contains the number of samples to process.
//
//-----------------------------------------------------------------------------
void Int_FFT(SI_UU16_t ReArray[], SI_UU16_t ImArray[])
{
// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.
#if (NUM_FFT >= 512)
   unsigned int sin_index, g_cnt, s_cnt; // Keeps track of the proper index
   unsigned int indexA, indexB;          // locations for each calculation
#endif

#if (NUM_FFT <= 256)
   unsigned char sin_index, g_cnt, s_cnt; // Keeps track of the proper index
   unsigned char indexA, indexB;          // locations for each calculation
#endif

   SI_UU16_t temp;                          // This must be declared here for use
                                       // with the FFT_AminusB macro

   unsigned int group = NUM_FFT/4, stage = 2;
   int CosVal, SinVal;
   SI_UU32_t ReTwid, ImTwid;
   SI_UU16_t TempReA, TempReB, TempImA, TempImB;
   SI_UU16_t TempReA2, TempReB2, TempImA2, TempImB2;

   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = MAC0_PAGE;

// ----------------------------------------------------------------------------
// FIRST STAGE - optimized for REAL input data only.  This will set all
// Imaginary locations to zero.
//
// The first stage is all of the 2-bit butterflies (so the groups are in
// pairs).
//
// Shortcuts have been taken to remove unnecessary multiplications during this
// stage. The angle "x" is 0 radians for all calculations at this point, so
// the SIN value is equal to 0.0 and the COS value is equal to 1.0.
// Additionally, all Imaginary locations are assumed to be '0' in this stage of
// the algorithm, and are set to '0'.
// ----------------------------------------------------------------------------

   // Initialize the MAC
   MAC0CF |= 0x10;                     // Right-shift mode
   MAC0CF |= 0x02;                     // Put the MAC in fractional mode
   MAC0CF |= 0x08;                     // Clear the accumulator

   indexA = 0;
   for (g_cnt = 0; g_cnt < NUM_FFT/2; g_cnt++)
   {
      indexB = indexA + 1;             // With bit-reversed input, A and B
                                       // are next to each other in the array

      TempReA.u16 = ReArray[indexA].u16; // Store the values in temporary
      TempReB.u16 = ReArray[indexB].u16; // variables to reduce clock cycles

      // Calculate new value for ReArray[indexA]
      // ReA2 = ReA.i + ReB.i;
      FFT_AplusB(TempReA, TempReB, ReArray[indexA].u16);

      // Calculate new value for ReArray[indexB]
      // ReA2 = ReA.i - ReB.i;
      FFT_AminusB(TempReA, TempReB, ReArray[indexB].u16)

      ImArray[indexA].u16 = 0;         // Set Imaginary locations to '0'
      ImArray[indexB].u16 = 0;

      indexA = indexB + 1;             // Move to the next butterfly pair
   }

// END OF FIRST STAGE

// ----------------------------------------------------------------------------
// 2nd - Nth stage - Calculate the appropriate butterflies.
//
// The 2nd stage is the 4-bit butterflies, 3rd stage is 8-bit butterflies, and
// so forth.  The <group> variable distinguishes between each n-bit butterfly,
// and the <stage> variable is used to determine which A and B in the array
// are combined to create the data for the next stage.
//
// Shortcuts have been taken to remove unnecessary multiplications during these
// stages. Whenever the angle "x" is 0 radians or pi/2 radians, the
// multiplications have been removed.
// ----------------------------------------------------------------------------

   while (stage <= NUM_FFT/2)
   {
      indexA = 0;
      sin_index = 0;

      for (g_cnt = 0; g_cnt < group; g_cnt++)
      {
         for (s_cnt = 0; s_cnt < stage; s_cnt++)
         {
            indexB = indexA + stage;

            TempReA.u16 = ReArray[indexA].u16;
            TempReB.u16 = ReArray[indexB].u16;
            TempImA.u16 = ImArray[indexA].u16;
            TempImB.u16 = ImArray[indexB].u16;

            // The following first checks for the special cases when the angle
            // "x" is equal to either 0 or pi/2 radians.  In these cases,
            // unnecessary multiplications have been removed to improve the
            // processing speed.
            //
            // Real_Twiddle = ReB*CosVal + ImB*SinVal
            // Imaginary_Twiddle = ImB*CosVal - ReB*SinVal
            //
            // ReA = ReA + Real_Twiddle
            // ReB = ReA - Real_Twiddle
            // ImA = ImA + Imaginary_Twiddle
            // ImB = ImA - Imaginary_Twiddle
            //
            // If the CosVal is 1 and SinVal is 0 (like when "x" = 0):
            //
            // ReA = ReA + (ReB*1 + ImB*0) = ReA + ReB
            // ReB = ReA - (ReB*1 + ImB*0) = ReA - ReB
            // ImA = ImA + (ImB*1 - ReB*0) = ImA + ImB
            // ImB = ImA - (ImB*1 - ReB*0) = ImA - ImB
            //
            // If the CosVal is 0 and SinVal is 1 (like when "x" = pi/2):
            //
            // ReA = ReA + (ReB*0 + ImB*1) = ReA + ImB
            // ReB = ReA - (ReB*0 + ImB*1) = ReA - ImB
            // ImA = ImA + (ImB*0 - ReB*1) = ImA - ReB
            // ImB = ImA - (ImB*0 - ReB*1) = ImA + ReB
            //
            if (sin_index == 0)  // corresponds to "x" = 0 radians
            {

               // Calculate new value for ReArray[indexA]
               // ReA = ReA + (ReB*1 + ImB*0) = ReA + ReB
               FFT_AplusB(TempReA, TempReB, TempReA2.u16);

               // Calculate new value for ReArray[indexB]
               // ReB = ReA - (ReB*1 + ImB*0) = ReA - ReB
               FFT_AminusB(TempReA, TempReB, TempReB2.u16);

               // Calculate new value for ImArray[indexA]
               // ImA = ImA + (ImB*1 - ReB*0) = ImA + ImB
               FFT_AplusB(TempImA, TempImB, TempImA2.u16);

               // Calculate new value for ImArray[indexB]
               // ImB = ImA - (ImB*1 - ReB*0) = ImA - ImB
               FFT_AminusB(TempImA, TempImB, TempImB2.u16);
            }
            else if (sin_index == NUM_FFT/4) // corresponds to "x" = pi/2
            {                                // radians

               // Calculate new value for ReArray[indexA]
               // ReA = ReA + (ReB*0 + ImB*1) = ReA + ImB
               FFT_AplusB(TempReA, TempImB, TempReA2.u16);

               // Calculate new value for ReArray[indexB]
               // ReB = ReA - (ReB*0 + ImB*1) = ReA - ImB
               FFT_AminusB(TempReA, TempImB, TempReB2.u16);

               // Calculate new value for ImArray[indexA]
               // ImA = ImA + (ImB*0 - ReB*1) = ImA - ReB
               FFT_AminusB(TempImA, TempReB, TempImA2.u16);

               // Calculate new value for ImArray[indexB]
               // ImB = ImA - (ImB*0 - ReB*1) = ImA + ReB
               FFT_AplusB(TempImA, TempReB, TempImB2.u16);
            }
            else
            {
               // If no multiplication shortcuts can be taken, the SIN and COS
               // values for the Butterfly calculation are fetched from the
               // SinTable[] array.
               //
               // Real_Twiddle = ReB*CosVal + ImB*SinVal
               // Imaginary_Twiddle = ImB*CosVal - ReB*SinVal
               //
               // ReA = ReA + Real_Twiddle
               // ReB = ReA - Real_Twiddle
               // ImA = ImA + Imaginary_Twiddle
               // ImB = ImA - Imaginary_Twiddle

               if (sin_index > NUM_FFT/4)
               {
                  SinVal = SinTable[(NUM_FFT/2) - sin_index];
                  CosVal = -SinTable[sin_index - (NUM_FFT/4)];
               }
               else
               {
                  SinVal = SinTable[sin_index];
                  CosVal = SinTable[(NUM_FFT/4) - sin_index];
               }

               // The SIN and COS values are used here to calculate part of the
               // Butterfly equation

               MAC0CF &= ~0x02;        // Put the MAC in integer mode

               // ReTwid.l = ((long)TempReB.u16 * CosVal) +
               //            ((long)TempImB.u16 * SinVal);
               MAC0A = CosVal;         // CosVal * ReB
               MAC0BH = TempReB.u8[MSB];
               MAC0BL = TempReB.u8[LSB];

               MAC0A = SinVal;         // SinVal * ImB
               MAC0BH = TempImB.u8[MSB];
               MAC0BL = TempImB.u8[LSB];

               NOP();
               NOP();

               ReTwid.u8[B3] = MAC0ACC3;
               ReTwid.u8[B2] = MAC0ACC2;
               ReTwid.u8[B1] = MAC0ACC1;
               ReTwid.u8[B0] = MAC0ACC0;

               MAC0CF |= 0x08;         // Clear the ACC

               // ImTwid.l = ((long)TempImB.u16 * CosVal) -
               //            ((long)TempReB.u16 * SinVal);
               MAC0A = SinVal;         // Multiply SinVal by -1
               MAC0BH = 0xFF;
               MAC0BL = 0xFF;

               NOP();
               NOP();

               MAC0AH = MAC0ACC1;
               MAC0AL = MAC0ACC0;

               MAC0CF |= 0x08;         // Clear the ACC

               MAC0BH = TempReB.u8[MSB]; // -SinVal * ReB
               MAC0BL = TempReB.u8[LSB];

               MAC0A = CosVal;         // CosVal * ImB
               MAC0BH = TempImB.u8[MSB];
               MAC0BL = TempImB.u8[LSB];

               NOP();
               NOP();

               ImTwid.u8[B3] = MAC0ACC3;
               ImTwid.u8[B2] = MAC0ACC2;
               ImTwid.u8[B1] = MAC0ACC1;
               ImTwid.u8[B0] = MAC0ACC0;

               MAC0CF |= 0x08;         // Clear the ACC

               MAC0CF |= 0x02;         // Put the MAC in fractional mode


               // Using the values calculated above, the new variables
               // are computed

               // Calculate new value for ReArray[indexA]
               // ReA = ReA + Real_Twiddle
               // by calculating:
               // ReA = -(-ReA) + Real_Twiddle
               MAC0A = TempReA.u16;      // ReA * -1
               MAC0BH = 0x80;
               MAC0BL = 0x00;

               NOP();
               NOP();

               MAC0CF |= 0x20;         // Right shift to guarantee no overflow

               NOP();
               NOP();

               MAC0AH = MAC0ACC3;      // Load -ReA in the MAC0A registers
               MAC0AL = MAC0ACC2;

               MAC0ACC3 = ReTwid.u8[B3]; // Load the ACC with ReTwid
               MAC0ACC2 = ReTwid.u8[B2];
               MAC0ACC1 = ReTwid.u8[B1];
               MAC0ACC0 = ReTwid.u8[B0];

               MAC0BH = 0x80;          // -ReA * -1
               MAC0BL = 0x00;

               NOP();
               NOP();

               TempReA2.u16 = MAC0RND;

               MAC0CF |= 0x08;         // Clear the ACC

               // Calculate new value for ReArray[indexB]
               // ReB = ReA - Real_Twiddle
               // by calculating:
               // -ReB = -ReA + Real_Twiddle
               MAC0ACC3 = ReTwid.u8[B3]; // Load the ACC with ReTwid
               MAC0ACC2 = ReTwid.u8[B2];
               MAC0ACC1 = ReTwid.u8[B1];
               MAC0ACC0 = ReTwid.u8[B0];

               MAC0A = 0x8000;         // ReA * -1
               MAC0BH = TempReA.u8[MSB];
               MAC0BL = TempReA.u8[LSB];

               NOP();
               NOP();
               NOP();

               MAC0CF |= 0x01;         // Multiply only mode

               MAC0A = MAC0RND;        // -ReB * -1
               MAC0BH = 0x80;
               MAC0BL = 0x00;

               NOP();
               NOP();

               TempReB2.u8[MSB] = MAC0ACC3;
               TempReB2.u8[LSB] = MAC0ACC2;

               MAC0CF &= ~0x01;        // Multiply and accumulate

               MAC0CF |= 0x08;         // Clear the ACC

               // Calculate new value for ImArray[indexA]
               // ImA = ImA + Imaginary_Twiddle
               // by calculating:
               // ImA = -(-ImA) + Imaginary_Twiddle
               MAC0A = TempImA.u16;      // ImA * -1
               MAC0BH = 0x80;
               MAC0BL = 0x00;

               NOP();
               NOP();

               MAC0CF |= 0x20;         // Right shift to guarantee no overflow

               NOP();
               NOP();

               MAC0AH = MAC0ACC3;      // Load -ImA in the MAC0A registers
               MAC0AL = MAC0ACC2;

               MAC0ACC3 = ImTwid.u8[B3]; // Load the ACC with ImTwid
               MAC0ACC2 = ImTwid.u8[B2];
               MAC0ACC1 = ImTwid.u8[B1];
               MAC0ACC0 = ImTwid.u8[B0];

               MAC0BH = 0x80;          // -ImA * -1
               MAC0BL = 0x00;

               NOP();
               NOP();

               TempImA2.u16 = MAC0RND;

               MAC0CF |= 0x08;         // Clear the ACC

               // Calculate new value for ImArray[indexB]
               // ImB = ImA - Imaginary_Twiddle
               // by calculating
               // -ImB = -ImA + Imaginary_Twiddle
               MAC0ACC3 = ImTwid.u8[B3]; // Load the ACC with ImTwid
               MAC0ACC2 = ImTwid.u8[B2];
               MAC0ACC1 = ImTwid.u8[B1];
               MAC0ACC0 = ImTwid.u8[B0];

               MAC0A = 0x8000;         // ImA * -1
               MAC0BH = TempImA.u8[MSB];
               MAC0BL = TempImA.u8[LSB];

               NOP();
               NOP();
               NOP();

               MAC0CF |= 0x01;         // Multiply only mode

               MAC0A = MAC0RND;        // -ImB * -1
               MAC0BH = 0x80;
               MAC0BL = 0x00;

               NOP();
               NOP();

               TempImB2.u8[MSB] = MAC0ACC3;
               TempImB2.u8[LSB] = MAC0ACC2;

               MAC0CF &= ~0x01;        // Multiply and accumulate

               MAC0CF |= 0x08;         // Clear the ACC

            }

            ReArray[indexA].u16 = TempReA2.u16;
            ReArray[indexB].u16 = TempReB2.u16;
            ImArray[indexA].u16 = TempImA2.u16;
            ImArray[indexB].u16 = TempImB2.u16;

            indexA++;
            sin_index += group;
         }                             // END of stage FOR loop (s_cnt)

         indexA = indexB + 1;
         sin_index = 0;
      }                                // END of group FOR loop (g_cnt)

      group /= 2;
      stage *= 2;
   }

   SFRPAGE = SFRPAGE_save;
}


//-----------------------------------------------------------------------------
// putchar
//-----------------------------------------------------------------------------
//
// Return Value:
//   1) char c - returns the char c that was passed as a parameter
// Parameters:
//   1) char c - the character to be printed
//
// Print the character <c> using UART1 at <BAUDRATE>.
//
//-----------------------------------------------------------------------------
#if defined __C51__
char putchar (char c)
#elif defined SDCC
void putchar (char c)
#endif
{
   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = UART1_PAGE;

   if (c == '\n')                      // Print a carriage return
   {
      while (!TI1);
      TI1 = 0;
      SBUF1 = 0x0d;
   }
   while (!TI1);
   TI1 = 0;
   SBUF1 = c;

   SFRPAGE = SFRPAGE_SAVE;

#if defined __C51__
   return c;                           // Print the character
#endif
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------
