//-----------------------------------------------------------------------------
// FFT_Code_Tables_MAC.h
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// File Description:
// --------------------
//
// This header file is used to provide Sine, Bit Reversal, and Window tables
// for calculating an FFT.  The tables are stored in FLASH memory (code space).
// All of the tables are conditionally used at compile time based on the values
// of NUM_FFT and WINDOW_TYPE, so only one table of each type is used in the
// software.
//
// Pinout:
//
// P0.4 - UART0 TX pin
// P0.5 - UART0 RX pin
//
// P1.1 - ADC0 input
//
// Target:         C8051F362 (TOOLSTICK360DC) / C8051F120 ('F120 TB)
// Tool chain:     Keil C51 7.50 / Keil EVAL C51 / SDCC
// Command Line:   None
//
// Release 1.0
//    -Initial Revision (BD / TP)
//    -10 MAR 2007
//

#define NUM_FFT 256     // Length of FFT to process
                        // Must be 2^N, where N is an integer >= 2

#define WINDOW_TYPE 4   // WINDOW_TYPE specifies the window to use on the data
                        // The available window functions are:
                        // 0 = No Window
                        // 1 = Triangle Window
                        // 2 = Hanning Window
                        // 3 = Hamming Window
                        // 4 = Blackman Window

// SinTable[] - SIN Tables are first 1/4 of a SIN wave - used to perform
// complex math functions.  These are encoded such that a value of 1.0
// corresponds to 32768, and a value of -1.0 corresponds to -32768.

// BRTable[] - Bit Reversal tables are used to bit-reverse sort the data and
// perform other indexing functions.  The Bit Reversal tables are stored
// as 1/2 of their actual value, and the real value is computed at
// runtime.

// WindowFunc[] - Tables used to window data.  These are encoded such that
// 1.0 corresponds to 32768, and 0.0 corresponds to 0.  The tables do not use
// the MSB because the MSB of the MAC is the sign bit.

// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 1024
//-----------------------------------------------------------------------------
// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.
#if (NUM_FFT == 1024)
int code SinTable[256] =
{
0x0000, 0x00C9, 0x0192, 0x025B, 0x0324, 0x03ED, 0x04B6, 0x057F,
0x0647, 0x0710, 0x07D9, 0x08A2, 0x096A, 0x0A33, 0x0AFB, 0x0BC3,
0x0C8B, 0x0D53, 0x0E1B, 0x0EE3, 0x0FAB, 0x1072, 0x1139, 0x1201,
0x12C8, 0x138E, 0x1455, 0x151B, 0x15E2, 0x16A8, 0x176D, 0x1833,
0x18F8, 0x19BD, 0x1A82, 0x1B47, 0x1C0B, 0x1CCF, 0x1D93, 0x1E56,
0x1F19, 0x1FDC, 0x209F, 0x2161, 0x2223, 0x22E5, 0x23A6, 0x2467,
0x2528, 0x25E8, 0x26A8, 0x2767, 0x2826, 0x28E5, 0x29A3, 0x2A61,
0x2B1F, 0x2BDC, 0x2C98, 0x2D55, 0x2E11, 0x2ECC, 0x2F87, 0x3041,
0x30FB, 0x31B5, 0x326E, 0x3326, 0x33DE, 0x3496, 0x354D, 0x3604,
0x36BA, 0x376F, 0x3824, 0x38D8, 0x398C, 0x3A40, 0x3AF2, 0x3BA5,
0x3C56, 0x3D07, 0x3DB8, 0x3E68, 0x3F17, 0x3FC5, 0x4073, 0x4121,
0x41CE, 0x427A, 0x4325, 0x43D0, 0x447A, 0x4524, 0x45CD, 0x4675,
0x471C, 0x47C3, 0x4869, 0x490F, 0x49B4, 0x4A58, 0x4AFB, 0x4B9E,
0x4C3F, 0x4CE1, 0x4D81, 0x4E21, 0x4EBF, 0x4F5E, 0x4FFB, 0x5097,
0x5133, 0x51CE, 0x5269, 0x5302, 0x539B, 0x5433, 0x54CA, 0x5560,
0x55F5, 0x568A, 0x571D, 0x57B0, 0x5842, 0x58D4, 0x5964, 0x59F3,
0x5A82, 0x5B10, 0x5B9D, 0x5C29, 0x5CB4, 0x5D3E, 0x5DC7, 0x5E50,
0x5ED7, 0x5F5E, 0x5FE3, 0x6068, 0x60EC, 0x616F, 0x61F1, 0x6271,
0x62F2, 0x6371, 0x63EF, 0x646C, 0x64E8, 0x6563, 0x65DD, 0x6657,
0x66CF, 0x6746, 0x67BD, 0x6832, 0x68A6, 0x6919, 0x698C, 0x69FD,
0x6A6D, 0x6ADC, 0x6B4A, 0x6BB8, 0x6C24, 0x6C8F, 0x6CF9, 0x6D62,
0x6DCA, 0x6E30, 0x6E96, 0x6EFB, 0x6F5F, 0x6FC1, 0x7023, 0x7083,
0x70E2, 0x7141, 0x719E, 0x71FA, 0x7255, 0x72AF, 0x7307, 0x735F,
0x73B5, 0x740B, 0x745F, 0x74B2, 0x7504, 0x7555, 0x75A5, 0x75F4,
0x7641, 0x768E, 0x76D9, 0x7723, 0x776C, 0x77B4, 0x77FA, 0x7840,
0x7884, 0x78C7, 0x7909, 0x794A, 0x798A, 0x79C8, 0x7A05, 0x7A42,
0x7A7D, 0x7AB6, 0x7AEF, 0x7B26, 0x7B5D, 0x7B92, 0x7BC5, 0x7BF8,
0x7C29, 0x7C5A, 0x7C89, 0x7CB7, 0x7CE3, 0x7D0F, 0x7D39, 0x7D62,
0x7D8A, 0x7DB0, 0x7DD6, 0x7DFA, 0x7E1D, 0x7E3F, 0x7E5F, 0x7E7F,
0x7E9D, 0x7EBA, 0x7ED5, 0x7EF0, 0x7F09, 0x7F21, 0x7F38, 0x7F4D,
0x7F62, 0x7F75, 0x7F87, 0x7F97, 0x7FA7, 0x7FB5, 0x7FC2, 0x7FCE,
0x7FD8, 0x7FE1, 0x7FE9, 0x7FF0, 0x7FF6, 0x7FFA, 0x7FFD, 0x7FFF
};

unsigned int code BRTable[512] =
{
0, 256, 128, 384, 64, 320, 192, 448,
32, 288, 160, 416, 96, 352, 224, 480,
16, 272, 144, 400, 80, 336, 208, 464,
48, 304, 176, 432, 112, 368, 240, 496,
8, 264, 136, 392, 72, 328, 200, 456,
40, 296, 168, 424, 104, 360, 232, 488,
24, 280, 152, 408, 88, 344, 216, 472,
56, 312, 184, 440, 120, 376, 248, 504,
4, 260, 132, 388, 68, 324, 196, 452,
36, 292, 164, 420, 100, 356, 228, 484,
20, 276, 148, 404, 84, 340, 212, 468,
52, 308, 180, 436, 116, 372, 244, 500,
12, 268, 140, 396, 76, 332, 204, 460,
44, 300, 172, 428, 108, 364, 236, 492,
28, 284, 156, 412, 92, 348, 220, 476,
60, 316, 188, 444, 124, 380, 252, 508,
2, 258, 130, 386, 66, 322, 194, 450,
34, 290, 162, 418, 98, 354, 226, 482,
18, 274, 146, 402, 82, 338, 210, 466,
50, 306, 178, 434, 114, 370, 242, 498,
10, 266, 138, 394, 74, 330, 202, 458,
42, 298, 170, 426, 106, 362, 234, 490,
26, 282, 154, 410, 90, 346, 218, 474,
58, 314, 186, 442, 122, 378, 250, 506,
6, 262, 134, 390, 70, 326, 198, 454,
38, 294, 166, 422, 102, 358, 230, 486,
22, 278, 150, 406, 86, 342, 214, 470,
54, 310, 182, 438, 118, 374, 246, 502,
14, 270, 142, 398, 78, 334, 206, 462,
46, 302, 174, 430, 110, 366, 238, 494,
30, 286, 158, 414, 94, 350, 222, 478,
62, 318, 190, 446, 126, 382, 254, 510,
1, 257, 129, 385, 65, 321, 193, 449,
33, 289, 161, 417, 97, 353, 225, 481,
17, 273, 145, 401, 81, 337, 209, 465,
49, 305, 177, 433, 113, 369, 241, 497,
9, 265, 137, 393, 73, 329, 201, 457,
41, 297, 169, 425, 105, 361, 233, 489,
25, 281, 153, 409, 89, 345, 217, 473,
57, 313, 185, 441, 121, 377, 249, 505,
5, 261, 133, 389, 69, 325, 197, 453,
37, 293, 165, 421, 101, 357, 229, 485,
21, 277, 149, 405, 85, 341, 213, 469,
53, 309, 181, 437, 117, 373, 245, 501,
13, 269, 141, 397, 77, 333, 205, 461,
45, 301, 173, 429, 109, 365, 237, 493,
29, 285, 157, 413, 93, 349, 221, 477,
61, 317, 189, 445, 125, 381, 253, 509,
3, 259, 131, 387, 67, 323, 195, 451,
35, 291, 163, 419, 99, 355, 227, 483,
19, 275, 147, 403, 83, 339, 211, 467,
51, 307, 179, 435, 115, 371, 243, 499,
11, 267, 139, 395, 75, 331, 203, 459,
43, 299, 171, 427, 107, 363, 235, 491,
27, 283, 155, 411, 91, 347, 219, 475,
59, 315, 187, 443, 123, 379, 251, 507,
7, 263, 135, 391, 71, 327, 199, 455,
39, 295, 167, 423, 103, 359, 231, 487,
23, 279, 151, 407, 87, 343, 215, 471,
55, 311, 183, 439, 119, 375, 247, 503,
15, 271, 143, 399, 79, 335, 207, 463,
47, 303, 175, 431, 111, 367, 239, 495,
31, 287, 159, 415, 95, 351, 223, 479,
63, 319, 191, 447, 127, 383, 255, 511
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 512
//-----------------------------------------------------------------------------
// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.
#if (NUM_FFT == 512)
int code SinTable[128] =
{
0x0000, 0x0192, 0x0324, 0x04B6, 0x0647, 0x07D9, 0x096A, 0x0AFB,
0x0C8B, 0x0E1B, 0x0FAB, 0x1139, 0x12C8, 0x1455, 0x15E2, 0x176D,
0x18F8, 0x1A82, 0x1C0B, 0x1D93, 0x1F19, 0x209F, 0x2223, 0x23A6,
0x2528, 0x26A8, 0x2826, 0x29A3, 0x2B1F, 0x2C98, 0x2E11, 0x2F87,
0x30FB, 0x326E, 0x33DE, 0x354D, 0x36BA, 0x3824, 0x398C, 0x3AF2,
0x3C56, 0x3DB8, 0x3F17, 0x4073, 0x41CE, 0x4325, 0x447A, 0x45CD,
0x471C, 0x4869, 0x49B4, 0x4AFB, 0x4C3F, 0x4D81, 0x4EBF, 0x4FFB,
0x5133, 0x5269, 0x539B, 0x54CA, 0x55F5, 0x571D, 0x5842, 0x5964,
0x5A82, 0x5B9D, 0x5CB4, 0x5DC7, 0x5ED7, 0x5FE3, 0x60EC, 0x61F1,
0x62F2, 0x63EF, 0x64E8, 0x65DD, 0x66CF, 0x67BD, 0x68A6, 0x698C,
0x6A6D, 0x6B4A, 0x6C24, 0x6CF9, 0x6DCA, 0x6E96, 0x6F5F, 0x7023,
0x70E2, 0x719E, 0x7255, 0x7307, 0x73B5, 0x745F, 0x7504, 0x75A5,
0x7641, 0x76D9, 0x776C, 0x77FA, 0x7884, 0x7909, 0x798A, 0x7A05,
0x7A7D, 0x7AEF, 0x7B5D, 0x7BC5, 0x7C29, 0x7C89, 0x7CE3, 0x7D39,
0x7D8A, 0x7DD6, 0x7E1D, 0x7E5F, 0x7E9D, 0x7ED5, 0x7F09, 0x7F38,
0x7F62, 0x7F87, 0x7FA7, 0x7FC2, 0x7FD8, 0x7FE9, 0x7FF6, 0x7FFD
};

unsigned char code BRTable[256] =
{
0, 128, 64, 192, 32, 160, 96, 224,
16, 144, 80, 208, 48, 176, 112, 240,
8, 136, 72, 200, 40, 168, 104, 232,
24, 152, 88, 216, 56, 184, 120, 248,
4, 132, 68, 196, 36, 164, 100, 228,
20, 148, 84, 212, 52, 180, 116, 244,
12, 140, 76, 204, 44, 172, 108, 236,
28, 156, 92, 220, 60, 188, 124, 252,
2, 130, 66, 194, 34, 162, 98, 226,
18, 146, 82, 210, 50, 178, 114, 242,
10, 138, 74, 202, 42, 170, 106, 234,
26, 154, 90, 218, 58, 186, 122, 250,
6, 134, 70, 198, 38, 166, 102, 230,
22, 150, 86, 214, 54, 182, 118, 246,
14, 142, 78, 206, 46, 174, 110, 238,
30, 158, 94, 222, 62, 190, 126, 254,
1, 129, 65, 193, 33, 161, 97, 225,
17, 145, 81, 209, 49, 177, 113, 241,
9, 137, 73, 201, 41, 169, 105, 233,
25, 153, 89, 217, 57, 185, 121, 249,
5, 133, 69, 197, 37, 165, 101, 229,
21, 149, 85, 213, 53, 181, 117, 245,
13, 141, 77, 205, 45, 173, 109, 237,
29, 157, 93, 221, 61, 189, 125, 253,
3, 131, 67, 195, 35, 163, 99, 227,
19, 147, 83, 211, 51, 179, 115, 243,
11, 139, 75, 203, 43, 171, 107, 235,
27, 155, 91, 219, 59, 187, 123, 251,
7, 135, 71, 199, 39, 167, 103, 231,
23, 151, 87, 215, 55, 183, 119, 247,
15, 143, 79, 207, 47, 175, 111, 239,
31, 159, 95, 223, 63, 191, 127, 255
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 256
//-----------------------------------------------------------------------------
#if (NUM_FFT == 256)
int code SinTable[64] =
{
0x0000, 0x0324, 0x0647, 0x096A, 0x0C8B, 0x0FAB, 0x12C8, 0x15E2,
0x18F8, 0x1C0B, 0x1F19, 0x2223, 0x2528, 0x2826, 0x2B1F, 0x2E11,
0x30FB, 0x33DE, 0x36BA, 0x398C, 0x3C56, 0x3F17, 0x41CE, 0x447A,
0x471C, 0x49B4, 0x4C3F, 0x4EBF, 0x5133, 0x539B, 0x55F5, 0x5842,
0x5A82, 0x5CB4, 0x5ED7, 0x60EC, 0x62F2, 0x64E8, 0x66CF, 0x68A6,
0x6A6D, 0x6C24, 0x6DCA, 0x6F5F, 0x70E2, 0x7255, 0x73B5, 0x7504,
0x7641, 0x776C, 0x7884, 0x798A, 0x7A7D, 0x7B5D, 0x7C29, 0x7CE3,
0x7D8A, 0x7E1D, 0x7E9D, 0x7F09, 0x7F62, 0x7FA7, 0x7FD8, 0x7FF6
};

unsigned char code BRTable[128] =
{
0, 64, 32, 96, 16, 80, 48, 112,
8, 72, 40, 104, 24, 88, 56, 120,
4, 68, 36, 100, 20, 84, 52, 116,
12, 76, 44, 108, 28, 92, 60, 124,
2, 66, 34, 98, 18, 82, 50, 114,
10, 74, 42, 106, 26, 90, 58, 122,
6, 70, 38, 102, 22, 86, 54, 118,
14, 78, 46, 110, 30, 94, 62, 126,
1, 65, 33, 97, 17, 81, 49, 113,
9, 73, 41, 105, 25, 89, 57, 121,
5, 69, 37, 101, 21, 85, 53, 117,
13, 77, 45, 109, 29, 93, 61, 125,
3, 67, 35, 99, 19, 83, 51, 115,
11, 75, 43, 107, 27, 91, 59, 123,
7, 71, 39, 103, 23, 87, 55, 119,
15, 79, 47, 111, 31, 95, 63, 127
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 128
//-----------------------------------------------------------------------------
#if (NUM_FFT == 128)
int code SinTable[32] =
{
0x0000, 0x0647, 0x0C8B, 0x12C8, 0x18F8, 0x1F19, 0x2528, 0x2B1F,
0x30FB, 0x36BA, 0x3C56, 0x41CE, 0x471C, 0x4C3F, 0x5133, 0x55F5,
0x5A82, 0x5ED7, 0x62F2, 0x66CF, 0x6A6D, 0x6DCA, 0x70E2, 0x73B5,
0x7641, 0x7884, 0x7A7D, 0x7C29, 0x7D8A, 0x7E9D, 0x7F62, 0x7FD8
};

unsigned char code BRTable[64] =
{
0, 32, 16, 48, 8, 40, 24, 56,
4, 36, 20, 52, 12, 44, 28, 60,
2, 34, 18, 50, 10, 42, 26, 58,
6, 38, 22, 54, 14, 46, 30, 62,
1, 33, 17, 49, 9, 41, 25, 57,
5, 37, 21, 53, 13, 45, 29, 61,
3, 35, 19, 51, 11, 43, 27, 59,
7, 39, 23, 55, 15, 47, 31, 63
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 64
//-----------------------------------------------------------------------------
#if (NUM_FFT == 64)
int code SinTable[16] =
{
0x0000, 0x0C8B, 0x18F8, 0x2528, 0x30FB, 0x3C56, 0x471C, 0x5133,
0x5A82, 0x62F2, 0x6A6D, 0x70E2, 0x7641, 0x7A7D, 0x7D8A, 0x7F62
};

unsigned char code BRTable[32] =
{
0, 16, 8, 24, 4, 20, 12, 28,
2, 18, 10, 26, 6, 22, 14, 30,
1, 17, 9, 25, 5, 21, 13, 29,
3, 19, 11, 27, 7, 23, 15, 31
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 32
//-----------------------------------------------------------------------------
#if (NUM_FFT == 32)
int code SinTable[8] =
{
0x0000, 0x18F8, 0x30FB, 0x471C, 0x5A82, 0x6A6D, 0x7641, 0x7D8A
};

unsigned char code BRTable[16] =
{
0, 8, 4, 12, 2, 10, 6, 14,
1, 9, 5, 13, 3, 11, 7, 15
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 16
//-----------------------------------------------------------------------------
#if (NUM_FFT == 16)
int code SinTable[4] =
{
0x0000, 0x30FB, 0x5A82, 0x7641
};

unsigned char code BRTable[8] =
{
0, 4, 2, 6, 1, 5, 3, 7
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Tables for NUM_FFT = 8
//-----------------------------------------------------------------------------
#if (NUM_FFT == 8)
int code SinTable[2] =
{
0x0000, 0x5A82
};

unsigned char code BRTable[4] =
{
0, 2, 1, 3
};
#endif

//-----------------------------------------------------------------------------
// SIN and BR Table for NUM_FFT = 4
//-----------------------------------------------------------------------------
#if (NUM_FFT == 4)

int code SinTable[1] =
{
0x0000
};

unsigned char code BRTable[2] =
{
0, 1
};
#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 1024
//-----------------------------------------------------------------------------
// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.
#if (NUM_FFT == 1024)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[512] =
{
 0x0000, 0x0040, 0x0080, 0x00c0, 0x0100, 0x0140, 0x0180, 0x01c0,
 0x0200, 0x0240, 0x0280, 0x02c0, 0x0300, 0x0340, 0x0380, 0x03c0,
 0x0400, 0x0440, 0x0480, 0x04c0, 0x0500, 0x0540, 0x0580, 0x05c0,
 0x0600, 0x0640, 0x0680, 0x06c0, 0x0700, 0x0740, 0x0780, 0x07c0,
 0x0800, 0x0840, 0x0880, 0x08c0, 0x0900, 0x0940, 0x0980, 0x09c0,
 0x0a00, 0x0a40, 0x0a80, 0x0ac0, 0x0b00, 0x0b40, 0x0b80, 0x0bc0,
 0x0c00, 0x0c40, 0x0c80, 0x0cc0, 0x0d00, 0x0d40, 0x0d80, 0x0dc0,
 0x0e00, 0x0e40, 0x0e80, 0x0ec0, 0x0f00, 0x0f40, 0x0f80, 0x0fc0,
 0x1000, 0x1040, 0x1080, 0x10c0, 0x1100, 0x1140, 0x1180, 0x11c0,
 0x1200, 0x1240, 0x1280, 0x12c0, 0x1300, 0x1340, 0x1380, 0x13c0,
 0x1400, 0x1440, 0x1480, 0x14c0, 0x1500, 0x1540, 0x1580, 0x15c0,
 0x1600, 0x1640, 0x1680, 0x16c0, 0x1700, 0x1740, 0x1780, 0x17c0,
 0x1800, 0x1840, 0x1880, 0x18c0, 0x1900, 0x1940, 0x1980, 0x19c0,
 0x1a00, 0x1a40, 0x1a80, 0x1ac0, 0x1b00, 0x1b40, 0x1b80, 0x1bc0,
 0x1c00, 0x1c40, 0x1c80, 0x1cc0, 0x1d00, 0x1d40, 0x1d80, 0x1dc0,
 0x1e00, 0x1e40, 0x1e80, 0x1ec0, 0x1f00, 0x1f40, 0x1f80, 0x1fc0,
 0x2000, 0x2040, 0x2080, 0x20c0, 0x2100, 0x2140, 0x2180, 0x21c0,
 0x2200, 0x2240, 0x2280, 0x22c0, 0x2300, 0x2340, 0x2380, 0x23c0,
 0x2400, 0x2440, 0x2480, 0x24c0, 0x2500, 0x2540, 0x2580, 0x25c0,
 0x2600, 0x2640, 0x2680, 0x26c0, 0x2700, 0x2740, 0x2780, 0x27c0,
 0x2800, 0x2840, 0x2880, 0x28c0, 0x2900, 0x2940, 0x2980, 0x29c0,
 0x2a00, 0x2a40, 0x2a80, 0x2ac0, 0x2b00, 0x2b40, 0x2b80, 0x2bc0,
 0x2c00, 0x2c40, 0x2c80, 0x2cc0, 0x2d00, 0x2d40, 0x2d80, 0x2dc0,
 0x2e00, 0x2e40, 0x2e80, 0x2ec0, 0x2f00, 0x2f40, 0x2f80, 0x2fc0,
 0x3000, 0x3040, 0x3080, 0x30c0, 0x3100, 0x3140, 0x3180, 0x31c0,
 0x3200, 0x3240, 0x3280, 0x32c0, 0x3300, 0x3340, 0x3380, 0x33c0,
 0x3400, 0x3440, 0x3480, 0x34c0, 0x3500, 0x3540, 0x3580, 0x35c0,
 0x3600, 0x3640, 0x3680, 0x36c0, 0x3700, 0x3740, 0x3780, 0x37c0,
 0x3800, 0x3840, 0x3880, 0x38c0, 0x3900, 0x3940, 0x3980, 0x39c0,
 0x3a00, 0x3a40, 0x3a80, 0x3ac0, 0x3b00, 0x3b40, 0x3b80, 0x3bc0,
 0x3c00, 0x3c40, 0x3c80, 0x3cc0, 0x3d00, 0x3d40, 0x3d80, 0x3dc0,
 0x3e00, 0x3e40, 0x3e80, 0x3ec0, 0x3f00, 0x3f40, 0x3f80, 0x3fc0,
 0x4000, 0x4040, 0x4080, 0x40c0, 0x4100, 0x4140, 0x4180, 0x41c0,
 0x4200, 0x4240, 0x4280, 0x42c0, 0x4300, 0x4340, 0x4380, 0x43c0,
 0x4400, 0x4440, 0x4480, 0x44c0, 0x4500, 0x4540, 0x4580, 0x45c0,
 0x4600, 0x4640, 0x4680, 0x46c0, 0x4700, 0x4740, 0x4780, 0x47c0,
 0x4800, 0x4840, 0x4880, 0x48c0, 0x4900, 0x4940, 0x4980, 0x49c0,
 0x4a00, 0x4a40, 0x4a80, 0x4ac0, 0x4b00, 0x4b40, 0x4b80, 0x4bc0,
 0x4c00, 0x4c40, 0x4c80, 0x4cc0, 0x4d00, 0x4d40, 0x4d80, 0x4dc0,
 0x4e00, 0x4e40, 0x4e80, 0x4ec0, 0x4f00, 0x4f40, 0x4f80, 0x4fc0,
 0x5000, 0x5040, 0x5080, 0x50c0, 0x5100, 0x5140, 0x5180, 0x51c0,
 0x5200, 0x5240, 0x5280, 0x52c0, 0x5300, 0x5340, 0x5380, 0x53c0,
 0x5400, 0x5440, 0x5480, 0x54c0, 0x5500, 0x5540, 0x5580, 0x55c0,
 0x5600, 0x5640, 0x5680, 0x56c0, 0x5700, 0x5740, 0x5780, 0x57c0,
 0x5800, 0x5840, 0x5880, 0x58c0, 0x5900, 0x5940, 0x5980, 0x59c0,
 0x5a00, 0x5a40, 0x5a80, 0x5ac0, 0x5b00, 0x5b40, 0x5b80, 0x5bc0,
 0x5c00, 0x5c40, 0x5c80, 0x5cc0, 0x5d00, 0x5d40, 0x5d80, 0x5dc0,
 0x5e00, 0x5e40, 0x5e80, 0x5ec0, 0x5f00, 0x5f40, 0x5f80, 0x5fc0,
 0x6000, 0x6040, 0x6080, 0x60c0, 0x6100, 0x6140, 0x6180, 0x61c0,
 0x6200, 0x6240, 0x6280, 0x62c0, 0x6300, 0x6340, 0x6380, 0x63c0,
 0x6400, 0x6440, 0x6480, 0x64c0, 0x6500, 0x6540, 0x6580, 0x65c0,
 0x6600, 0x6640, 0x6680, 0x66c0, 0x6700, 0x6740, 0x6780, 0x67c0,
 0x6800, 0x6840, 0x6880, 0x68c0, 0x6900, 0x6940, 0x6980, 0x69c0,
 0x6a00, 0x6a40, 0x6a80, 0x6ac0, 0x6b00, 0x6b40, 0x6b80, 0x6bc0,
 0x6c00, 0x6c40, 0x6c80, 0x6cc0, 0x6d00, 0x6d40, 0x6d80, 0x6dc0,
 0x6e00, 0x6e40, 0x6e80, 0x6ec0, 0x6f00, 0x6f40, 0x6f80, 0x6fc0,
 0x7000, 0x7040, 0x7080, 0x70c0, 0x7100, 0x7140, 0x7180, 0x71c0,
 0x7200, 0x7240, 0x7280, 0x72c0, 0x7300, 0x7340, 0x7380, 0x73c0,
 0x7400, 0x7440, 0x7480, 0x74c0, 0x7500, 0x7540, 0x7580, 0x75c0,
 0x7600, 0x7640, 0x7680, 0x76c0, 0x7700, 0x7740, 0x7780, 0x77c0,
 0x7800, 0x7840, 0x7880, 0x78c0, 0x7900, 0x7940, 0x7980, 0x79c0,
 0x7a00, 0x7a40, 0x7a80, 0x7ac0, 0x7b00, 0x7b40, 0x7b80, 0x7bc0,
 0x7c00, 0x7c40, 0x7c80, 0x7cc0, 0x7d00, 0x7d40, 0x7d80, 0x7dc0,
 0x7e00, 0x7e40, 0x7e80, 0x7ec0, 0x7f00, 0x7f40, 0x7f80, 0x7fc0
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[512] =
{
 0x0000, 0x0000, 0x0001, 0x0002, 0x0004, 0x0007, 0x000b, 0x000f,
 0x0013, 0x0018, 0x001e, 0x0025, 0x002c, 0x0034, 0x003c, 0x0045,
 0x004e, 0x0059, 0x0063, 0x006f, 0x007b, 0x0087, 0x0095, 0x00a2,
 0x00b1, 0x00c0, 0x00d0, 0x00e0, 0x00f1, 0x0102, 0x0114, 0x0127,
 0x013a, 0x014e, 0x0163, 0x0178, 0x018e, 0x01a4, 0x01bb, 0x01d2,
 0x01eb, 0x0203, 0x021d, 0x0236, 0x0251, 0x026c, 0x0288, 0x02a4,
 0x02c1, 0x02de, 0x02fd, 0x031b, 0x033a, 0x035a, 0x037b, 0x039c,
 0x03bd, 0x03df, 0x0402, 0x0425, 0x0449, 0x046e, 0x0493, 0x04b8,
 0x04df, 0x0505, 0x052d, 0x0555, 0x057d, 0x05a6, 0x05d0, 0x05fa,
 0x0625, 0x0650, 0x067c, 0x06a8, 0x06d5, 0x0702, 0x0730, 0x075f,
 0x078e, 0x07be, 0x07ee, 0x081f, 0x0850, 0x0882, 0x08b4, 0x08e7,
 0x091a, 0x094e, 0x0983, 0x09b8, 0x09ed, 0x0a23, 0x0a5a, 0x0a91,
 0x0ac9, 0x0b01, 0x0b39, 0x0b73, 0x0bac, 0x0be6, 0x0c21, 0x0c5c,
 0x0c98, 0x0cd4, 0x0d11, 0x0d4e, 0x0d8b, 0x0dc9, 0x0e08, 0x0e47,
 0x0e86, 0x0ec7, 0x0f07, 0x0f48, 0x0f89, 0x0fcb, 0x100e, 0x1050,
 0x1094, 0x10d7, 0x111c, 0x1160, 0x11a5, 0x11eb, 0x1231, 0x1277,
 0x12be, 0x1306, 0x134d, 0x1395, 0x13de, 0x1427, 0x1471, 0x14ba,
 0x1505, 0x154f, 0x159a, 0x15e6, 0x1632, 0x167e, 0x16cb, 0x1718,
 0x1766, 0x17b4, 0x1802, 0x1850, 0x18a0, 0x18ef, 0x193f, 0x198f,
 0x19e0, 0x1a30, 0x1a82, 0x1ad3, 0x1b25, 0x1b78, 0x1bcb, 0x1c1e,
 0x1c71, 0x1cc5, 0x1d19, 0x1d6d, 0x1dc2, 0x1e17, 0x1e6d, 0x1ec2,
 0x1f18, 0x1f6f, 0x1fc6, 0x201d, 0x2074, 0x20cb, 0x2123, 0x217c,
 0x21d4, 0x222d, 0x2286, 0x22df, 0x2339, 0x2393, 0x23ed, 0x2448,
 0x24a2, 0x24fd, 0x2559, 0x25b4, 0x2610, 0x266c, 0x26c8, 0x2725,
 0x2782, 0x27df, 0x283c, 0x2899, 0x28f7, 0x2955, 0x29b3, 0x2a11,
 0x2a70, 0x2acf, 0x2b2e, 0x2b8d, 0x2bec, 0x2c4c, 0x2cab, 0x2d0b,
 0x2d6b, 0x2dcc, 0x2e2c, 0x2e8d, 0x2eee, 0x2f4f, 0x2fb0, 0x3011,
 0x3073, 0x30d4, 0x3136, 0x3198, 0x31fa, 0x325c, 0x32be, 0x3321,
 0x3383, 0x33e6, 0x3449, 0x34ab, 0x350e, 0x3572, 0x35d5, 0x3638,
 0x369b, 0x36ff, 0x3763, 0x37c6, 0x382a, 0x388e, 0x38f2, 0x3956,
 0x39ba, 0x3a1e, 0x3a82, 0x3ae6, 0x3b4a, 0x3bae, 0x3c13, 0x3c77,
 0x3cdc, 0x3d40, 0x3da4, 0x3e09, 0x3e6d, 0x3ed2, 0x3f36, 0x3f9b,
 0x3fff, 0x4064, 0x40c9, 0x412d, 0x4192, 0x41f6, 0x425b, 0x42bf,
 0x4323, 0x4388, 0x43ec, 0x4451, 0x44b5, 0x4519, 0x457d, 0x45e1,
 0x4645, 0x46a9, 0x470d, 0x4771, 0x47d5, 0x4839, 0x489c, 0x4900,
 0x4964, 0x49c7, 0x4a2a, 0x4a8d, 0x4af1, 0x4b54, 0x4bb6, 0x4c19,
 0x4c7c, 0x4cde, 0x4d41, 0x4da3, 0x4e05, 0x4e67, 0x4ec9, 0x4f2b,
 0x4f8c, 0x4fee, 0x504f, 0x50b0, 0x5111, 0x5172, 0x51d3, 0x5233,
 0x5294, 0x52f4, 0x5354, 0x53b3, 0x5413, 0x5472, 0x54d1, 0x5530,
 0x558f, 0x55ee, 0x564c, 0x56aa, 0x5708, 0x5766, 0x57c3, 0x5820,
 0x587d, 0x58da, 0x5937, 0x5993, 0x59ef, 0x5a4b, 0x5aa6, 0x5b02,
 0x5b5d, 0x5bb7, 0x5c12, 0x5c6c, 0x5cc6, 0x5d20, 0x5d79, 0x5dd2,
 0x5e2b, 0x5e83, 0x5edc, 0x5f34, 0x5f8b, 0x5fe2, 0x6039, 0x6090,
 0x60e7, 0x613d, 0x6192, 0x61e8, 0x623d, 0x6292, 0x62e6, 0x633a,
 0x638e, 0x63e1, 0x6434, 0x6487, 0x64da, 0x652c, 0x657d, 0x65cf,
 0x661f, 0x6670, 0x66c0, 0x6710, 0x675f, 0x67af, 0x67fd, 0x684b,
 0x6899, 0x68e7, 0x6934, 0x6981, 0x69cd, 0x6a19, 0x6a65, 0x6ab0,
 0x6afa, 0x6b45, 0x6b8e, 0x6bd8, 0x6c21, 0x6c6a, 0x6cb2, 0x6cf9,
 0x6d41, 0x6d88, 0x6dce, 0x6e14, 0x6e5a, 0x6e9f, 0x6ee3, 0x6f28,
 0x6f6b, 0x6faf, 0x6ff1, 0x7034, 0x7076, 0x70b7, 0x70f8, 0x7138,
 0x7179, 0x71b8, 0x71f7, 0x7236, 0x7274, 0x72b1, 0x72ee, 0x732b,
 0x7367, 0x73a3, 0x73de, 0x7419, 0x7453, 0x748c, 0x74c6, 0x74fe,
 0x7536, 0x756e, 0x75a5, 0x75dc, 0x7612, 0x7647, 0x767c, 0x76b1,
 0x76e5, 0x7718, 0x774b, 0x777d, 0x77af, 0x77e0, 0x7811, 0x7841,
 0x7871, 0x78a0, 0x78cf, 0x78fd, 0x792a, 0x7957, 0x7983, 0x79af,
 0x79da, 0x7a05, 0x7a2f, 0x7a59, 0x7a82, 0x7aaa, 0x7ad2, 0x7afa,
 0x7b20, 0x7b47, 0x7b6c, 0x7b91, 0x7bb6, 0x7bda, 0x7bfd, 0x7c20,
 0x7c42, 0x7c63, 0x7c84, 0x7ca5, 0x7cc5, 0x7ce4, 0x7d02, 0x7d21,
 0x7d3e, 0x7d5b, 0x7d77, 0x7d93, 0x7dae, 0x7dc9, 0x7de2, 0x7dfc,
 0x7e14, 0x7e2d, 0x7e44, 0x7e5b, 0x7e71, 0x7e87, 0x7e9c, 0x7eb1,
 0x7ec5, 0x7ed8, 0x7eeb, 0x7efd, 0x7f0e, 0x7f1f, 0x7f2f, 0x7f3f,
 0x7f4e, 0x7f5d, 0x7f6a, 0x7f78, 0x7f84, 0x7f90, 0x7f9c, 0x7fa6,
 0x7fb1, 0x7fba, 0x7fc3, 0x7fcb, 0x7fd3, 0x7fda, 0x7fe1, 0x7fe7,
 0x7fec, 0x7ff0, 0x7ff4, 0x7ff8, 0x7ffb, 0x7ffd, 0x7ffe, 0x7fff
};

#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[512] =
{
 0x0a3d, 0x0a3d, 0x0a3e, 0x0a3f, 0x0a41, 0x0a44, 0x0a47, 0x0a4b,
 0x0a4f, 0x0a54, 0x0a59, 0x0a5f, 0x0a66, 0x0a6d, 0x0a75, 0x0a7d,
 0x0a86, 0x0a8f, 0x0a99, 0x0aa3, 0x0aae, 0x0aba, 0x0ac6, 0x0ad3,
 0x0ae0, 0x0aee, 0x0afc, 0x0b0b, 0x0b1b, 0x0b2b, 0x0b3c, 0x0b4d,
 0x0b5f, 0x0b71, 0x0b84, 0x0b97, 0x0bab, 0x0bc0, 0x0bd5, 0x0bea,
 0x0c01, 0x0c17, 0x0c2f, 0x0c47, 0x0c5f, 0x0c78, 0x0c91, 0x0cab,
 0x0cc6, 0x0ce1, 0x0cfd, 0x0d19, 0x0d36, 0x0d53, 0x0d71, 0x0d8f,
 0x0dae, 0x0dcd, 0x0ded, 0x0e0e, 0x0e2f, 0x0e51, 0x0e73, 0x0e95,
 0x0eb8, 0x0edc, 0x0f00, 0x0f25, 0x0f4a, 0x0f70, 0x0f96, 0x0fbd,
 0x0fe4, 0x100c, 0x1034, 0x105d, 0x1086, 0x10b0, 0x10db, 0x1105,
 0x1131, 0x115d, 0x1189, 0x11b6, 0x11e3, 0x1211, 0x123f, 0x126e,
 0x129d, 0x12cd, 0x12fe, 0x132e, 0x1360, 0x1391, 0x13c3, 0x13f6,
 0x1429, 0x145d, 0x1491, 0x14c6, 0x14fb, 0x1530, 0x1566, 0x159c,
 0x15d3, 0x160b, 0x1642, 0x167b, 0x16b3, 0x16ec, 0x1726, 0x1760,
 0x179a, 0x17d5, 0x1811, 0x184c, 0x1889, 0x18c5, 0x1902, 0x1940,
 0x197e, 0x19bc, 0x19fb, 0x1a3a, 0x1a79, 0x1ab9, 0x1afa, 0x1b3b,
 0x1b7c, 0x1bbd, 0x1bff, 0x1c42, 0x1c85, 0x1cc8, 0x1d0b, 0x1d4f,
 0x1d94, 0x1dd8, 0x1e1d, 0x1e63, 0x1ea9, 0x1eef, 0x1f36, 0x1f7d,
 0x1fc4, 0x200c, 0x2054, 0x209c, 0x20e5, 0x212e, 0x2177, 0x21c1,
 0x220b, 0x2256, 0x22a0, 0x22eb, 0x2337, 0x2383, 0x23cf, 0x241b,
 0x2468, 0x24b5, 0x2502, 0x2550, 0x259e, 0x25ec, 0x263b, 0x268a,
 0x26d9, 0x2728, 0x2778, 0x27c8, 0x2819, 0x2869, 0x28ba, 0x290b,
 0x295d, 0x29ae, 0x2a00, 0x2a53, 0x2aa5, 0x2af8, 0x2b4b, 0x2b9e,
 0x2bf2, 0x2c45, 0x2c99, 0x2ced, 0x2d42, 0x2d97, 0x2deb, 0x2e41,
 0x2e96, 0x2eeb, 0x2f41, 0x2f97, 0x2fed, 0x3044, 0x309a, 0x30f1,
 0x3148, 0x319f, 0x31f7, 0x324e, 0x32a6, 0x32fe, 0x3356, 0x33ae,
 0x3407, 0x345f, 0x34b8, 0x3511, 0x356a, 0x35c3, 0x361d, 0x3676,
 0x36d0, 0x3729, 0x3783, 0x37dd, 0x3838, 0x3892, 0x38ec, 0x3947,
 0x39a2, 0x39fc, 0x3a57, 0x3ab2, 0x3b0d, 0x3b68, 0x3bc4, 0x3c1f,
 0x3c7b, 0x3cd6, 0x3d32, 0x3d8d, 0x3de9, 0x3e45, 0x3ea1, 0x3efd,
 0x3f59, 0x3fb5, 0x4011, 0x406d, 0x40c9, 0x4126, 0x4182, 0x41de,
 0x423b, 0x4297, 0x42f3, 0x4350, 0x43ac, 0x4409, 0x4465, 0x44c2,
 0x451e, 0x457b, 0x45d7, 0x4634, 0x4690, 0x46ed, 0x4749, 0x47a5,
 0x4802, 0x485e, 0x48bb, 0x4917, 0x4973, 0x49cf, 0x4a2b, 0x4a88,
 0x4ae4, 0x4b40, 0x4b9c, 0x4bf8, 0x4c53, 0x4caf, 0x4d0b, 0x4d66,
 0x4dc2, 0x4e1d, 0x4e79, 0x4ed4, 0x4f2f, 0x4f8a, 0x4fe5, 0x5040,
 0x509b, 0x50f6, 0x5150, 0x51aa, 0x5205, 0x525f, 0x52b9, 0x5313,
 0x536d, 0x53c6, 0x5420, 0x5479, 0x54d2, 0x552c, 0x5584, 0x55dd,
 0x5636, 0x568e, 0x56e6, 0x573f, 0x5796, 0x57ee, 0x5846, 0x589d,
 0x58f4, 0x594b, 0x59a2, 0x59f9, 0x5a4f, 0x5aa5, 0x5afb, 0x5b51,
 0x5ba7, 0x5bfc, 0x5c51, 0x5ca6, 0x5cfb, 0x5d4f, 0x5da3, 0x5df7,
 0x5e4b, 0x5e9e, 0x5ef2, 0x5f45, 0x5f97, 0x5fea, 0x603c, 0x608e,
 0x60e0, 0x6131, 0x6182, 0x61d3, 0x6224, 0x6274, 0x62c4, 0x6314,
 0x6363, 0x63b3, 0x6402, 0x6450, 0x649e, 0x64ec, 0x653a, 0x6587,
 0x65d4, 0x6621, 0x666e, 0x66ba, 0x6706, 0x6751, 0x679c, 0x67e7,
 0x6831, 0x687b, 0x68c5, 0x690f, 0x6958, 0x69a1, 0x69e9, 0x6a31,
 0x6a79, 0x6ac0, 0x6b07, 0x6b4d, 0x6b94, 0x6bda, 0x6c1f, 0x6c64,
 0x6ca9, 0x6ced, 0x6d31, 0x6d75, 0x6db8, 0x6dfb, 0x6e3d, 0x6e7f,
 0x6ec1, 0x6f02, 0x6f43, 0x6f83, 0x6fc3, 0x7003, 0x7042, 0x7080,
 0x70bf, 0x70fd, 0x713a, 0x7177, 0x71b4, 0x71f0, 0x722c, 0x7267,
 0x72a2, 0x72dc, 0x7316, 0x7350, 0x7389, 0x73c2, 0x73fa, 0x7432,
 0x7469, 0x74a0, 0x74d6, 0x750c, 0x7542, 0x7577, 0x75ab, 0x75e0,
 0x7613, 0x7646, 0x7679, 0x76ab, 0x76dd, 0x770e, 0x773f, 0x776f,
 0x779f, 0x77ce, 0x77fd, 0x782b, 0x7859, 0x7887, 0x78b3, 0x78e0,
 0x790c, 0x7937, 0x7962, 0x798c, 0x79b6, 0x79df, 0x7a08, 0x7a31,
 0x7a58, 0x7a80, 0x7aa6, 0x7acd, 0x7af2, 0x7b18, 0x7b3c, 0x7b60,
 0x7b84, 0x7ba7, 0x7bca, 0x7bec, 0x7c0d, 0x7c2f, 0x7c4f, 0x7c6f,
 0x7c8e, 0x7cad, 0x7ccc, 0x7ce9, 0x7d07, 0x7d23, 0x7d40, 0x7d5b,
 0x7d76, 0x7d91, 0x7dab, 0x7dc5, 0x7ddd, 0x7df6, 0x7e0e, 0x7e25,
 0x7e3c, 0x7e52, 0x7e68, 0x7e7d, 0x7e91, 0x7ea5, 0x7eb9, 0x7ecc,
 0x7ede, 0x7ef0, 0x7f01, 0x7f11, 0x7f22, 0x7f31, 0x7f40, 0x7f4f,
 0x7f5c, 0x7f6a, 0x7f76, 0x7f83, 0x7f8e, 0x7f99, 0x7fa4, 0x7fae,
 0x7fb7, 0x7fc0, 0x7fc8, 0x7fd0, 0x7fd7, 0x7fdd, 0x7fe3, 0x7fe9,
 0x7fed, 0x7ff2, 0x7ff5, 0x7ff8, 0x7ffb, 0x7ffd, 0x7ffe, 0x7fff
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[512] =
{
 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0002, 0x0003, 0x0005,
 0x0007, 0x0009, 0x000b, 0x000d, 0x0010, 0x0012, 0x0015, 0x0019,
 0x001c, 0x0020, 0x0024, 0x0028, 0x002c, 0x0031, 0x0036, 0x003b,
 0x0040, 0x0045, 0x004b, 0x0051, 0x0057, 0x005e, 0x0065, 0x006c,
 0x0073, 0x007a, 0x0082, 0x008a, 0x0092, 0x009a, 0x00a3, 0x00ac,
 0x00b5, 0x00be, 0x00c8, 0x00d2, 0x00dc, 0x00e6, 0x00f1, 0x00fc,
 0x0107, 0x0113, 0x011e, 0x012a, 0x0137, 0x0143, 0x0150, 0x015d,
 0x016a, 0x0178, 0x0186, 0x0194, 0x01a2, 0x01b1, 0x01c0, 0x01cf,
 0x01df, 0x01ef, 0x01ff, 0x020f, 0x0220, 0x0231, 0x0243, 0x0254,
 0x0266, 0x0278, 0x028b, 0x029e, 0x02b1, 0x02c5, 0x02d8, 0x02ed,
 0x0301, 0x0316, 0x032b, 0x0340, 0x0356, 0x036c, 0x0383, 0x039a,
 0x03b1, 0x03c8, 0x03e0, 0x03f8, 0x0411, 0x042a, 0x0443, 0x045c,
 0x0476, 0x0491, 0x04ab, 0x04c6, 0x04e2, 0x04fe, 0x051a, 0x0536,
 0x0553, 0x0571, 0x058e, 0x05ac, 0x05cb, 0x05ea, 0x0609, 0x0628,
 0x0648, 0x0669, 0x068a, 0x06ab, 0x06cd, 0x06ef, 0x0711, 0x0734,
 0x0757, 0x077b, 0x079f, 0x07c4, 0x07e9, 0x080e, 0x0834, 0x085a,
 0x0881, 0x08a8, 0x08d0, 0x08f8, 0x0920, 0x0949, 0x0972, 0x099c,
 0x09c6, 0x09f1, 0x0a1c, 0x0a48, 0x0a74, 0x0aa0, 0x0acd, 0x0afb,
 0x0b29, 0x0b57, 0x0b86, 0x0bb5, 0x0be5, 0x0c15, 0x0c46, 0x0c77,
 0x0ca9, 0x0cdb, 0x0d0e, 0x0d41, 0x0d75, 0x0da9, 0x0dde, 0x0e13,
 0x0e48, 0x0e7f, 0x0eb5, 0x0eec, 0x0f24, 0x0f5c, 0x0f95, 0x0fce,
 0x1007, 0x1041, 0x107c, 0x10b7, 0x10f3, 0x112f, 0x116b, 0x11a9,
 0x11e6, 0x1224, 0x1263, 0x12a2, 0x12e2, 0x1322, 0x1363, 0x13a4,
 0x13e6, 0x1428, 0x146b, 0x14ae, 0x14f2, 0x1536, 0x157b, 0x15c1,
 0x1607, 0x164d, 0x1694, 0x16db, 0x1723, 0x176c, 0x17b5, 0x17fe,
 0x1848, 0x1893, 0x18de, 0x1929, 0x1975, 0x19c2, 0x1a0f, 0x1a5c,
 0x1aaa, 0x1af9, 0x1b48, 0x1b98, 0x1be8, 0x1c38, 0x1c89, 0x1cdb,
 0x1d2d, 0x1d80, 0x1dd3, 0x1e26, 0x1e7b, 0x1ecf, 0x1f24, 0x1f7a,
 0x1fd0, 0x2026, 0x207d, 0x20d5, 0x212d, 0x2185, 0x21de, 0x2238,
 0x2291, 0x22ec, 0x2347, 0x23a2, 0x23fe, 0x245a, 0x24b6, 0x2514,
 0x2571, 0x25cf, 0x262e, 0x268c, 0x26ec, 0x274b, 0x27ac, 0x280c,
 0x286d, 0x28cf, 0x2931, 0x2993, 0x29f6, 0x2a59, 0x2abc, 0x2b20,
 0x2b85, 0x2be9, 0x2c4e, 0x2cb4, 0x2d1a, 0x2d80, 0x2de7, 0x2e4e,
 0x2eb5, 0x2f1d, 0x2f85, 0x2fed, 0x3056, 0x30bf, 0x3129, 0x3193,
 0x31fd, 0x3267, 0x32d2, 0x333d, 0x33a9, 0x3415, 0x3481, 0x34ed,
 0x355a, 0x35c6, 0x3634, 0x36a1, 0x370f, 0x377d, 0x37eb, 0x385a,
 0x38c9, 0x3938, 0x39a7, 0x3a16, 0x3a86, 0x3af6, 0x3b66, 0x3bd7,
 0x3c47, 0x3cb8, 0x3d29, 0x3d9a, 0x3e0b, 0x3e7d, 0x3eef, 0x3f60,
 0x3fd2, 0x4045, 0x40b7, 0x4129, 0x419c, 0x420f, 0x4281, 0x42f4,
 0x4367, 0x43da, 0x444e, 0x44c1, 0x4534, 0x45a8, 0x461b, 0x468f,
 0x4702, 0x4776, 0x47ea, 0x485d, 0x48d1, 0x4945, 0x49b9, 0x4a2c,
 0x4aa0, 0x4b14, 0x4b88, 0x4bfb, 0x4c6f, 0x4ce3, 0x4d56, 0x4dca,
 0x4e3d, 0x4eb0, 0x4f24, 0x4f97, 0x500a, 0x507d, 0x50f0, 0x5163,
 0x51d5, 0x5248, 0x52ba, 0x532d, 0x539f, 0x5411, 0x5482, 0x54f4,
 0x5565, 0x55d7, 0x5648, 0x56b8, 0x5729, 0x5799, 0x5809, 0x5879,
 0x58e9, 0x5958, 0x59c8, 0x5a36, 0x5aa5, 0x5b13, 0x5b81, 0x5bef,
 0x5c5d, 0x5cca, 0x5d36, 0x5da3, 0x5e0f, 0x5e7b, 0x5ee6, 0x5f51,
 0x5fbc, 0x6026, 0x6090, 0x60fa, 0x6163, 0x61cc, 0x6234, 0x629c,
 0x6303, 0x636a, 0x63d1, 0x6437, 0x649d, 0x6502, 0x6567, 0x65cb,
 0x662f, 0x6692, 0x66f5, 0x6757, 0x67b9, 0x681a, 0x687b, 0x68db,
 0x693a, 0x699a, 0x69f8, 0x6a56, 0x6ab3, 0x6b10, 0x6b6c, 0x6bc8,
 0x6c23, 0x6c7d, 0x6cd7, 0x6d30, 0x6d89, 0x6de0, 0x6e38, 0x6e8e,
 0x6ee4, 0x6f39, 0x6f8e, 0x6fe2, 0x7035, 0x7087, 0x70d9, 0x712a,
 0x717b, 0x71cb, 0x7219, 0x7268, 0x72b5, 0x7302, 0x734e, 0x7399,
 0x73e4, 0x742e, 0x7477, 0x74bf, 0x7506, 0x754d, 0x7593, 0x75d8,
 0x761c, 0x765f, 0x76a2, 0x76e4, 0x7725, 0x7765, 0x77a4, 0x77e3,
 0x7821, 0x785d, 0x7899, 0x78d4, 0x790f, 0x7948, 0x7980, 0x79b8,
 0x79ef, 0x7a25, 0x7a59, 0x7a8e, 0x7ac1, 0x7af3, 0x7b24, 0x7b55,
 0x7b84, 0x7bb3, 0x7be0, 0x7c0d, 0x7c39, 0x7c64, 0x7c8e, 0x7cb7,
 0x7cdf, 0x7d06, 0x7d2c, 0x7d51, 0x7d76, 0x7d99, 0x7dbb, 0x7ddd,
 0x7dfd, 0x7e1d, 0x7e3b, 0x7e59, 0x7e75, 0x7e91, 0x7eab, 0x7ec5,
 0x7edd, 0x7ef5, 0x7f0b, 0x7f21, 0x7f36, 0x7f49, 0x7f5c, 0x7f6e,
 0x7f7e, 0x7f8e, 0x7f9c, 0x7faa, 0x7fb7, 0x7fc2, 0x7fcd, 0x7fd7,
 0x7fdf, 0x7fe7, 0x7fed, 0x7ff3, 0x7ff7, 0x7ffb, 0x7ffd, 0x7fff
};
#endif

#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 512
//-----------------------------------------------------------------------------
// NOTE: NUM_FFT sizes larger than 256 can only be used in conjuction with
// an external XRAM through the EMIF.
#if (NUM_FFT == 512)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[256] =
{
 0x0000, 0x0080, 0x0100, 0x0180, 0x0200, 0x0280, 0x0300, 0x0380,
 0x0400, 0x0480, 0x0500, 0x0580, 0x0600, 0x0680, 0x0700, 0x0780,
 0x0800, 0x0880, 0x0900, 0x0980, 0x0a00, 0x0a80, 0x0b00, 0x0b80,
 0x0c00, 0x0c80, 0x0d00, 0x0d80, 0x0e00, 0x0e80, 0x0f00, 0x0f80,
 0x1000, 0x1080, 0x1100, 0x1180, 0x1200, 0x1280, 0x1300, 0x1380,
 0x1400, 0x1480, 0x1500, 0x1580, 0x1600, 0x1680, 0x1700, 0x1780,
 0x1800, 0x1880, 0x1900, 0x1980, 0x1a00, 0x1a80, 0x1b00, 0x1b80,
 0x1c00, 0x1c80, 0x1d00, 0x1d80, 0x1e00, 0x1e80, 0x1f00, 0x1f80,
 0x2000, 0x2080, 0x2100, 0x2180, 0x2200, 0x2280, 0x2300, 0x2380,
 0x2400, 0x2480, 0x2500, 0x2580, 0x2600, 0x2680, 0x2700, 0x2780,
 0x2800, 0x2880, 0x2900, 0x2980, 0x2a00, 0x2a80, 0x2b00, 0x2b80,
 0x2c00, 0x2c80, 0x2d00, 0x2d80, 0x2e00, 0x2e80, 0x2f00, 0x2f80,
 0x3000, 0x3080, 0x3100, 0x3180, 0x3200, 0x3280, 0x3300, 0x3380,
 0x3400, 0x3480, 0x3500, 0x3580, 0x3600, 0x3680, 0x3700, 0x3780,
 0x3800, 0x3880, 0x3900, 0x3980, 0x3a00, 0x3a80, 0x3b00, 0x3b80,
 0x3c00, 0x3c80, 0x3d00, 0x3d80, 0x3e00, 0x3e80, 0x3f00, 0x3f80,
 0x4000, 0x4080, 0x4100, 0x4180, 0x4200, 0x4280, 0x4300, 0x4380,
 0x4400, 0x4480, 0x4500, 0x4580, 0x4600, 0x4680, 0x4700, 0x4780,
 0x4800, 0x4880, 0x4900, 0x4980, 0x4a00, 0x4a80, 0x4b00, 0x4b80,
 0x4c00, 0x4c80, 0x4d00, 0x4d80, 0x4e00, 0x4e80, 0x4f00, 0x4f80,
 0x5000, 0x5080, 0x5100, 0x5180, 0x5200, 0x5280, 0x5300, 0x5380,
 0x5400, 0x5480, 0x5500, 0x5580, 0x5600, 0x5680, 0x5700, 0x5780,
 0x5800, 0x5880, 0x5900, 0x5980, 0x5a00, 0x5a80, 0x5b00, 0x5b80,
 0x5c00, 0x5c80, 0x5d00, 0x5d80, 0x5e00, 0x5e80, 0x5f00, 0x5f80,
 0x6000, 0x6080, 0x6100, 0x6180, 0x6200, 0x6280, 0x6300, 0x6380,
 0x6400, 0x6480, 0x6500, 0x6580, 0x6600, 0x6680, 0x6700, 0x6780,
 0x6800, 0x6880, 0x6900, 0x6980, 0x6a00, 0x6a80, 0x6b00, 0x6b80,
 0x6c00, 0x6c80, 0x6d00, 0x6d80, 0x6e00, 0x6e80, 0x6f00, 0x6f80,
 0x7000, 0x7080, 0x7100, 0x7180, 0x7200, 0x7280, 0x7300, 0x7380,
 0x7400, 0x7480, 0x7500, 0x7580, 0x7600, 0x7680, 0x7700, 0x7780,
 0x7800, 0x7880, 0x7900, 0x7980, 0x7a00, 0x7a80, 0x7b00, 0x7b80,
 0x7c00, 0x7c80, 0x7d00, 0x7d80, 0x7e00, 0x7e80, 0x7f00, 0x7f80
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[256] =
{
 0x0000, 0x0001, 0x0004, 0x000b, 0x0013, 0x001e, 0x002c, 0x003c,
 0x004e, 0x0063, 0x007b, 0x0095, 0x00b1, 0x00d0, 0x00f1, 0x0114,
 0x013a, 0x0163, 0x018e, 0x01bb, 0x01eb, 0x021d, 0x0251, 0x0288,
 0x02c1, 0x02fd, 0x033a, 0x037b, 0x03bd, 0x0402, 0x0449, 0x0493,
 0x04df, 0x052d, 0x057d, 0x05d0, 0x0625, 0x067c, 0x06d5, 0x0730,
 0x078e, 0x07ee, 0x0850, 0x08b4, 0x091a, 0x0983, 0x09ed, 0x0a5a,
 0x0ac9, 0x0b39, 0x0bac, 0x0c21, 0x0c98, 0x0d11, 0x0d8b, 0x0e08,
 0x0e86, 0x0f07, 0x0f89, 0x100e, 0x1094, 0x111c, 0x11a5, 0x1231,
 0x12be, 0x134d, 0x13de, 0x1471, 0x1505, 0x159a, 0x1632, 0x16cb,
 0x1766, 0x1802, 0x18a0, 0x193f, 0x19e0, 0x1a82, 0x1b25, 0x1bcb,
 0x1c71, 0x1d19, 0x1dc2, 0x1e6d, 0x1f18, 0x1fc6, 0x2074, 0x2123,
 0x21d4, 0x2286, 0x2339, 0x23ed, 0x24a2, 0x2559, 0x2610, 0x26c8,
 0x2782, 0x283c, 0x28f7, 0x29b3, 0x2a70, 0x2b2e, 0x2bec, 0x2cab,
 0x2d6b, 0x2e2c, 0x2eee, 0x2fb0, 0x3073, 0x3136, 0x31fa, 0x32be,
 0x3383, 0x3449, 0x350e, 0x35d5, 0x369b, 0x3763, 0x382a, 0x38f2,
 0x39ba, 0x3a82, 0x3b4a, 0x3c13, 0x3cdc, 0x3da4, 0x3e6d, 0x3f36,
 0x3fff, 0x40c9, 0x4192, 0x425b, 0x4323, 0x43ec, 0x44b5, 0x457d,
 0x4645, 0x470d, 0x47d5, 0x489c, 0x4964, 0x4a2a, 0x4af1, 0x4bb6,
 0x4c7c, 0x4d41, 0x4e05, 0x4ec9, 0x4f8c, 0x504f, 0x5111, 0x51d3,
 0x5294, 0x5354, 0x5413, 0x54d1, 0x558f, 0x564c, 0x5708, 0x57c3,
 0x587d, 0x5937, 0x59ef, 0x5aa6, 0x5b5d, 0x5c12, 0x5cc6, 0x5d79,
 0x5e2b, 0x5edc, 0x5f8b, 0x6039, 0x60e7, 0x6192, 0x623d, 0x62e6,
 0x638e, 0x6434, 0x64da, 0x657d, 0x661f, 0x66c0, 0x675f, 0x67fd,
 0x6899, 0x6934, 0x69cd, 0x6a65, 0x6afa, 0x6b8e, 0x6c21, 0x6cb2,
 0x6d41, 0x6dce, 0x6e5a, 0x6ee3, 0x6f6b, 0x6ff1, 0x7076, 0x70f8,
 0x7179, 0x71f7, 0x7274, 0x72ee, 0x7367, 0x73de, 0x7453, 0x74c6,
 0x7536, 0x75a5, 0x7612, 0x767c, 0x76e5, 0x774b, 0x77af, 0x7811,
 0x7871, 0x78cf, 0x792a, 0x7983, 0x79da, 0x7a2f, 0x7a82, 0x7ad2,
 0x7b20, 0x7b6c, 0x7bb6, 0x7bfd, 0x7c42, 0x7c84, 0x7cc5, 0x7d02,
 0x7d3e, 0x7d77, 0x7dae, 0x7de2, 0x7e14, 0x7e44, 0x7e71, 0x7e9c,
 0x7ec5, 0x7eeb, 0x7f0e, 0x7f2f, 0x7f4e, 0x7f6a, 0x7f84, 0x7f9c,
 0x7fb1, 0x7fc3, 0x7fd3, 0x7fe1, 0x7fec, 0x7ff4, 0x7ffb, 0x7ffe
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[256] =
{
 0x0a3d, 0x0a3e, 0x0a41, 0x0a47, 0x0a4f, 0x0a59, 0x0a66, 0x0a75,
 0x0a86, 0x0a99, 0x0aae, 0x0ac6, 0x0ae0, 0x0afc, 0x0b1b, 0x0b3c,
 0x0b5f, 0x0b84, 0x0bab, 0x0bd5, 0x0c01, 0x0c2f, 0x0c5f, 0x0c91,
 0x0cc6, 0x0cfd, 0x0d36, 0x0d71, 0x0dae, 0x0ded, 0x0e2f, 0x0e73,
 0x0eb8, 0x0f00, 0x0f4a, 0x0f96, 0x0fe4, 0x1034, 0x1086, 0x10db,
 0x1131, 0x1189, 0x11e3, 0x123f, 0x129d, 0x12fe, 0x1360, 0x13c3,
 0x1429, 0x1491, 0x14fb, 0x1566, 0x15d3, 0x1642, 0x16b3, 0x1726,
 0x179a, 0x1811, 0x1889, 0x1902, 0x197e, 0x19fb, 0x1a79, 0x1afa,
 0x1b7c, 0x1bff, 0x1c85, 0x1d0b, 0x1d94, 0x1e1d, 0x1ea9, 0x1f36,
 0x1fc4, 0x2054, 0x20e5, 0x2177, 0x220b, 0x22a0, 0x2337, 0x23cf,
 0x2468, 0x2502, 0x259e, 0x263b, 0x26d9, 0x2778, 0x2819, 0x28ba,
 0x295d, 0x2a00, 0x2aa5, 0x2b4b, 0x2bf2, 0x2c99, 0x2d42, 0x2deb,
 0x2e96, 0x2f41, 0x2fed, 0x309a, 0x3148, 0x31f7, 0x32a6, 0x3356,
 0x3407, 0x34b8, 0x356a, 0x361d, 0x36d0, 0x3783, 0x3838, 0x38ec,
 0x39a2, 0x3a57, 0x3b0d, 0x3bc4, 0x3c7b, 0x3d32, 0x3de9, 0x3ea1,
 0x3f59, 0x4011, 0x40c9, 0x4182, 0x423b, 0x42f3, 0x43ac, 0x4465,
 0x451e, 0x45d7, 0x4690, 0x4749, 0x4802, 0x48bb, 0x4973, 0x4a2b,
 0x4ae4, 0x4b9c, 0x4c53, 0x4d0b, 0x4dc2, 0x4e79, 0x4f2f, 0x4fe5,
 0x509b, 0x5150, 0x5205, 0x52b9, 0x536d, 0x5420, 0x54d2, 0x5584,
 0x5636, 0x56e6, 0x5796, 0x5846, 0x58f4, 0x59a2, 0x5a4f, 0x5afb,
 0x5ba7, 0x5c51, 0x5cfb, 0x5da3, 0x5e4b, 0x5ef2, 0x5f97, 0x603c,
 0x60e0, 0x6182, 0x6224, 0x62c4, 0x6363, 0x6402, 0x649e, 0x653a,
 0x65d4, 0x666e, 0x6706, 0x679c, 0x6831, 0x68c5, 0x6958, 0x69e9,
 0x6a79, 0x6b07, 0x6b94, 0x6c1f, 0x6ca9, 0x6d31, 0x6db8, 0x6e3d,
 0x6ec1, 0x6f43, 0x6fc3, 0x7042, 0x70bf, 0x713a, 0x71b4, 0x722c,
 0x72a2, 0x7316, 0x7389, 0x73fa, 0x7469, 0x74d6, 0x7542, 0x75ab,
 0x7613, 0x7679, 0x76dd, 0x773f, 0x779f, 0x77fd, 0x7859, 0x78b3,
 0x790c, 0x7962, 0x79b6, 0x7a08, 0x7a58, 0x7aa6, 0x7af2, 0x7b3c,
 0x7b84, 0x7bca, 0x7c0d, 0x7c4f, 0x7c8e, 0x7ccc, 0x7d07, 0x7d40,
 0x7d76, 0x7dab, 0x7ddd, 0x7e0e, 0x7e3c, 0x7e68, 0x7e91, 0x7eb9,
 0x7ede, 0x7f01, 0x7f22, 0x7f40, 0x7f5c, 0x7f76, 0x7f8e, 0x7fa4,
 0x7fb7, 0x7fc8, 0x7fd7, 0x7fe3, 0x7fed, 0x7ff5, 0x7ffb, 0x7ffe
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[256] =
{
 0x0000, 0x0000, 0x0001, 0x0003, 0x0007, 0x000b, 0x0010, 0x0015,
 0x001c, 0x0024, 0x002c, 0x0036, 0x0040, 0x004b, 0x0057, 0x0065,
 0x0073, 0x0082, 0x0092, 0x00a3, 0x00b5, 0x00c8, 0x00dc, 0x00f1,
 0x0107, 0x011e, 0x0137, 0x0150, 0x016a, 0x0186, 0x01a2, 0x01c0,
 0x01df, 0x01ff, 0x0220, 0x0243, 0x0266, 0x028b, 0x02b1, 0x02d8,
 0x0301, 0x032b, 0x0356, 0x0383, 0x03b1, 0x03e0, 0x0411, 0x0443,
 0x0476, 0x04ab, 0x04e2, 0x051a, 0x0553, 0x058e, 0x05cb, 0x0609,
 0x0648, 0x068a, 0x06cd, 0x0711, 0x0757, 0x079f, 0x07e9, 0x0834,
 0x0881, 0x08d0, 0x0920, 0x0972, 0x09c6, 0x0a1c, 0x0a74, 0x0acd,
 0x0b29, 0x0b86, 0x0be5, 0x0c46, 0x0ca9, 0x0d0e, 0x0d75, 0x0dde,
 0x0e48, 0x0eb5, 0x0f24, 0x0f95, 0x1007, 0x107c, 0x10f3, 0x116b,
 0x11e6, 0x1263, 0x12e2, 0x1363, 0x13e6, 0x146b, 0x14f2, 0x157b,
 0x1607, 0x1694, 0x1723, 0x17b5, 0x1848, 0x18de, 0x1975, 0x1a0f,
 0x1aaa, 0x1b48, 0x1be8, 0x1c89, 0x1d2d, 0x1dd3, 0x1e7b, 0x1f24,
 0x1fd0, 0x207d, 0x212d, 0x21de, 0x2291, 0x2347, 0x23fe, 0x24b6,
 0x2571, 0x262e, 0x26ec, 0x27ac, 0x286d, 0x2931, 0x29f6, 0x2abc,
 0x2b85, 0x2c4e, 0x2d1a, 0x2de7, 0x2eb5, 0x2f85, 0x3056, 0x3129,
 0x31fd, 0x32d2, 0x33a9, 0x3481, 0x355a, 0x3634, 0x370f, 0x37eb,
 0x38c9, 0x39a7, 0x3a86, 0x3b66, 0x3c47, 0x3d29, 0x3e0b, 0x3eef,
 0x3fd2, 0x40b7, 0x419c, 0x4281, 0x4367, 0x444e, 0x4534, 0x461b,
 0x4702, 0x47ea, 0x48d1, 0x49b9, 0x4aa0, 0x4b88, 0x4c6f, 0x4d56,
 0x4e3d, 0x4f24, 0x500a, 0x50f0, 0x51d5, 0x52ba, 0x539f, 0x5482,
 0x5565, 0x5648, 0x5729, 0x5809, 0x58e9, 0x59c8, 0x5aa5, 0x5b81,
 0x5c5d, 0x5d36, 0x5e0f, 0x5ee6, 0x5fbc, 0x6090, 0x6163, 0x6234,
 0x6303, 0x63d1, 0x649d, 0x6567, 0x662f, 0x66f5, 0x67b9, 0x687b,
 0x693a, 0x69f8, 0x6ab3, 0x6b6c, 0x6c23, 0x6cd7, 0x6d89, 0x6e38,
 0x6ee4, 0x6f8e, 0x7035, 0x70d9, 0x717b, 0x7219, 0x72b5, 0x734e,
 0x73e4, 0x7477, 0x7506, 0x7593, 0x761c, 0x76a2, 0x7725, 0x77a4,
 0x7821, 0x7899, 0x790f, 0x7980, 0x79ef, 0x7a59, 0x7ac1, 0x7b24,
 0x7b84, 0x7be0, 0x7c39, 0x7c8e, 0x7cdf, 0x7d2c, 0x7d76, 0x7dbb,
 0x7dfd, 0x7e3b, 0x7e75, 0x7eab, 0x7edd, 0x7f0b, 0x7f36, 0x7f5c,
 0x7f7e, 0x7f9c, 0x7fb7, 0x7fcd, 0x7fdf, 0x7fed, 0x7ff7, 0x7ffd
};
#endif

#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 256
//-----------------------------------------------------------------------------
#if (NUM_FFT == 256)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[128] =
{
 0x0000, 0x0100, 0x0200, 0x0300, 0x0400, 0x0500, 0x0600, 0x0700,
 0x0800, 0x0900, 0x0a00, 0x0b00, 0x0c00, 0x0d00, 0x0e00, 0x0f00,
 0x1000, 0x1100, 0x1200, 0x1300, 0x1400, 0x1500, 0x1600, 0x1700,
 0x1800, 0x1900, 0x1a00, 0x1b00, 0x1c00, 0x1d00, 0x1e00, 0x1f00,
 0x2000, 0x2100, 0x2200, 0x2300, 0x2400, 0x2500, 0x2600, 0x2700,
 0x2800, 0x2900, 0x2a00, 0x2b00, 0x2c00, 0x2d00, 0x2e00, 0x2f00,
 0x3000, 0x3100, 0x3200, 0x3300, 0x3400, 0x3500, 0x3600, 0x3700,
 0x3800, 0x3900, 0x3a00, 0x3b00, 0x3c00, 0x3d00, 0x3e00, 0x3f00,
 0x4000, 0x4100, 0x4200, 0x4300, 0x4400, 0x4500, 0x4600, 0x4700,
 0x4800, 0x4900, 0x4a00, 0x4b00, 0x4c00, 0x4d00, 0x4e00, 0x4f00,
 0x5000, 0x5100, 0x5200, 0x5300, 0x5400, 0x5500, 0x5600, 0x5700,
 0x5800, 0x5900, 0x5a00, 0x5b00, 0x5c00, 0x5d00, 0x5e00, 0x5f00,
 0x6000, 0x6100, 0x6200, 0x6300, 0x6400, 0x6500, 0x6600, 0x6700,
 0x6800, 0x6900, 0x6a00, 0x6b00, 0x6c00, 0x6d00, 0x6e00, 0x6f00,
 0x7000, 0x7100, 0x7200, 0x7300, 0x7400, 0x7500, 0x7600, 0x7700,
 0x7800, 0x7900, 0x7a00, 0x7b00, 0x7c00, 0x7d00, 0x7e00, 0x7f00
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[128] =
{
 0x0000, 0x0004, 0x0013, 0x002c, 0x004e, 0x007b, 0x00b1, 0x00f1,
 0x013a, 0x018e, 0x01eb, 0x0251, 0x02c1, 0x033a, 0x03bd, 0x0449,
 0x04df, 0x057d, 0x0625, 0x06d5, 0x078e, 0x0850, 0x091a, 0x09ed,
 0x0ac9, 0x0bac, 0x0c98, 0x0d8b, 0x0e86, 0x0f89, 0x1094, 0x11a5,
 0x12be, 0x13de, 0x1505, 0x1632, 0x1766, 0x18a0, 0x19e0, 0x1b25,
 0x1c71, 0x1dc2, 0x1f18, 0x2074, 0x21d4, 0x2339, 0x24a2, 0x2610,
 0x2782, 0x28f7, 0x2a70, 0x2bec, 0x2d6b, 0x2eee, 0x3073, 0x31fa,
 0x3383, 0x350e, 0x369b, 0x382a, 0x39ba, 0x3b4a, 0x3cdc, 0x3e6d,
 0x3fff, 0x4192, 0x4323, 0x44b5, 0x4645, 0x47d5, 0x4964, 0x4af1,
 0x4c7c, 0x4e05, 0x4f8c, 0x5111, 0x5294, 0x5413, 0x558f, 0x5708,
 0x587d, 0x59ef, 0x5b5d, 0x5cc6, 0x5e2b, 0x5f8b, 0x60e7, 0x623d,
 0x638e, 0x64da, 0x661f, 0x675f, 0x6899, 0x69cd, 0x6afa, 0x6c21,
 0x6d41, 0x6e5a, 0x6f6b, 0x7076, 0x7179, 0x7274, 0x7367, 0x7453,
 0x7536, 0x7612, 0x76e5, 0x77af, 0x7871, 0x792a, 0x79da, 0x7a82,
 0x7b20, 0x7bb6, 0x7c42, 0x7cc5, 0x7d3e, 0x7dae, 0x7e14, 0x7e71,
 0x7ec5, 0x7f0e, 0x7f4e, 0x7f84, 0x7fb1, 0x7fd3, 0x7fec, 0x7ffb
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[128] =
{
 0x0a3d, 0x0a41, 0x0a4f, 0x0a66, 0x0a86, 0x0aae, 0x0ae0, 0x0b1b,
 0x0b5f, 0x0bab, 0x0c01, 0x0c5f, 0x0cc6, 0x0d36, 0x0dae, 0x0e2f,
 0x0eb8, 0x0f4a, 0x0fe4, 0x1086, 0x1131, 0x11e3, 0x129d, 0x1360,
 0x1429, 0x14fb, 0x15d3, 0x16b3, 0x179a, 0x1889, 0x197e, 0x1a79,
 0x1b7c, 0x1c85, 0x1d94, 0x1ea9, 0x1fc4, 0x20e5, 0x220b, 0x2337,
 0x2468, 0x259e, 0x26d9, 0x2819, 0x295d, 0x2aa5, 0x2bf2, 0x2d42,
 0x2e96, 0x2fed, 0x3148, 0x32a6, 0x3407, 0x356a, 0x36d0, 0x3838,
 0x39a2, 0x3b0d, 0x3c7b, 0x3de9, 0x3f59, 0x40c9, 0x423b, 0x43ac,
 0x451e, 0x4690, 0x4802, 0x4973, 0x4ae4, 0x4c53, 0x4dc2, 0x4f2f,
 0x509b, 0x5205, 0x536d, 0x54d2, 0x5636, 0x5796, 0x58f4, 0x5a4f,
 0x5ba7, 0x5cfb, 0x5e4b, 0x5f97, 0x60e0, 0x6224, 0x6363, 0x649e,
 0x65d4, 0x6706, 0x6831, 0x6958, 0x6a79, 0x6b94, 0x6ca9, 0x6db8,
 0x6ec1, 0x6fc3, 0x70bf, 0x71b4, 0x72a2, 0x7389, 0x7469, 0x7542,
 0x7613, 0x76dd, 0x779f, 0x7859, 0x790c, 0x79b6, 0x7a58, 0x7af2,
 0x7b84, 0x7c0d, 0x7c8e, 0x7d07, 0x7d76, 0x7ddd, 0x7e3c, 0x7e91,
 0x7ede, 0x7f22, 0x7f5c, 0x7f8e, 0x7fb7, 0x7fd7, 0x7fed, 0x7ffb
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[128] =
{
 0x0000, 0x0001, 0x0007, 0x0010, 0x001c, 0x002c, 0x0040, 0x0057,
 0x0073, 0x0092, 0x00b5, 0x00dc, 0x0107, 0x0137, 0x016a, 0x01a2,
 0x01df, 0x0220, 0x0266, 0x02b1, 0x0301, 0x0356, 0x03b1, 0x0411,
 0x0476, 0x04e2, 0x0553, 0x05cb, 0x0648, 0x06cd, 0x0757, 0x07e9,
 0x0881, 0x0920, 0x09c6, 0x0a74, 0x0b29, 0x0be5, 0x0ca9, 0x0d75,
 0x0e48, 0x0f24, 0x1007, 0x10f3, 0x11e6, 0x12e2, 0x13e6, 0x14f2,
 0x1607, 0x1723, 0x1848, 0x1975, 0x1aaa, 0x1be8, 0x1d2d, 0x1e7b,
 0x1fd0, 0x212d, 0x2291, 0x23fe, 0x2571, 0x26ec, 0x286d, 0x29f6,
 0x2b85, 0x2d1a, 0x2eb5, 0x3056, 0x31fd, 0x33a9, 0x355a, 0x370f,
 0x38c9, 0x3a86, 0x3c47, 0x3e0b, 0x3fd2, 0x419c, 0x4367, 0x4534,
 0x4702, 0x48d1, 0x4aa0, 0x4c6f, 0x4e3d, 0x500a, 0x51d5, 0x539f,
 0x5565, 0x5729, 0x58e9, 0x5aa5, 0x5c5d, 0x5e0f, 0x5fbc, 0x6163,
 0x6303, 0x649d, 0x662f, 0x67b9, 0x693a, 0x6ab3, 0x6c23, 0x6d89,
 0x6ee4, 0x7035, 0x717b, 0x72b5, 0x73e4, 0x7506, 0x761c, 0x7725,
 0x7821, 0x790f, 0x79ef, 0x7ac1, 0x7b84, 0x7c39, 0x7cdf, 0x7d76,
 0x7dfd, 0x7e75, 0x7edd, 0x7f36, 0x7f7e, 0x7fb7, 0x7fdf, 0x7ff7
};
#endif

#endif

//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 128
//-----------------------------------------------------------------------------
#if (NUM_FFT == 128)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[64] =
{
 0x0000, 0x0200, 0x0400, 0x0600, 0x0800, 0x0a00, 0x0c00, 0x0e00,
 0x1000, 0x1200, 0x1400, 0x1600, 0x1800, 0x1a00, 0x1c00, 0x1e00,
 0x2000, 0x2200, 0x2400, 0x2600, 0x2800, 0x2a00, 0x2c00, 0x2e00,
 0x3000, 0x3200, 0x3400, 0x3600, 0x3800, 0x3a00, 0x3c00, 0x3e00,
 0x4000, 0x4200, 0x4400, 0x4600, 0x4800, 0x4a00, 0x4c00, 0x4e00,
 0x5000, 0x5200, 0x5400, 0x5600, 0x5800, 0x5a00, 0x5c00, 0x5e00,
 0x6000, 0x6200, 0x6400, 0x6600, 0x6800, 0x6a00, 0x6c00, 0x6e00,
 0x7000, 0x7200, 0x7400, 0x7600, 0x7800, 0x7a00, 0x7c00, 0x7e00
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[64] =
{
 0x0000, 0x0013, 0x004e, 0x00b1, 0x013a, 0x01eb, 0x02c1, 0x03bd,
 0x04df, 0x0625, 0x078e, 0x091a, 0x0ac9, 0x0c98, 0x0e86, 0x1094,
 0x12be, 0x1505, 0x1766, 0x19e0, 0x1c71, 0x1f18, 0x21d4, 0x24a2,
 0x2782, 0x2a70, 0x2d6b, 0x3073, 0x3383, 0x369b, 0x39ba, 0x3cdc,
 0x3fff, 0x4323, 0x4645, 0x4964, 0x4c7c, 0x4f8c, 0x5294, 0x558f,
 0x587d, 0x5b5d, 0x5e2b, 0x60e7, 0x638e, 0x661f, 0x6899, 0x6afa,
 0x6d41, 0x6f6b, 0x7179, 0x7367, 0x7536, 0x76e5, 0x7871, 0x79da,
 0x7b20, 0x7c42, 0x7d3e, 0x7e14, 0x7ec5, 0x7f4e, 0x7fb1, 0x7fec
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[64] =
{
 0x0a3d, 0x0a4f, 0x0a86, 0x0ae0, 0x0b5f, 0x0c01, 0x0cc6, 0x0dae,
 0x0eb8, 0x0fe4, 0x1131, 0x129d, 0x1429, 0x15d3, 0x179a, 0x197e,
 0x1b7c, 0x1d94, 0x1fc4, 0x220b, 0x2468, 0x26d9, 0x295d, 0x2bf2,
 0x2e96, 0x3148, 0x3407, 0x36d0, 0x39a2, 0x3c7b, 0x3f59, 0x423b,
 0x451e, 0x4802, 0x4ae4, 0x4dc2, 0x509b, 0x536d, 0x5636, 0x58f4,
 0x5ba7, 0x5e4b, 0x60e0, 0x6363, 0x65d4, 0x6831, 0x6a79, 0x6ca9,
 0x6ec1, 0x70bf, 0x72a2, 0x7469, 0x7613, 0x779f, 0x790c, 0x7a58,
 0x7b84, 0x7c8e, 0x7d76, 0x7e3c, 0x7ede, 0x7f5c, 0x7fb7, 0x7fed
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[64] =
{
 0x0000, 0x0007, 0x001c, 0x0040, 0x0073, 0x00b5, 0x0107, 0x016a,
 0x01df, 0x0266, 0x0301, 0x03b1, 0x0476, 0x0553, 0x0648, 0x0757,
 0x0881, 0x09c6, 0x0b29, 0x0ca9, 0x0e48, 0x1007, 0x11e6, 0x13e6,
 0x1607, 0x1848, 0x1aaa, 0x1d2d, 0x1fd0, 0x2291, 0x2571, 0x286d,
 0x2b85, 0x2eb5, 0x31fd, 0x355a, 0x38c9, 0x3c47, 0x3fd2, 0x4367,
 0x4702, 0x4aa0, 0x4e3d, 0x51d5, 0x5565, 0x58e9, 0x5c5d, 0x5fbc,
 0x6303, 0x662f, 0x693a, 0x6c23, 0x6ee4, 0x717b, 0x73e4, 0x761c,
 0x7821, 0x79ef, 0x7b84, 0x7cdf, 0x7dfd, 0x7edd, 0x7f7e, 0x7fdf
};
#endif

#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 64
//-----------------------------------------------------------------------------
#if (NUM_FFT == 64)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[32] =
{
 0x0000, 0x0400, 0x0800, 0x0c00, 0x1000, 0x1400, 0x1800, 0x1c00,
 0x2000, 0x2400, 0x2800, 0x2c00, 0x3000, 0x3400, 0x3800, 0x3c00,
 0x4000, 0x4400, 0x4800, 0x4c00, 0x5000, 0x5400, 0x5800, 0x5c00,
 0x6000, 0x6400, 0x6800, 0x6c00, 0x7000, 0x7400, 0x7800, 0x7c00
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[32] =
{
 0x0000, 0x004e, 0x013a, 0x02c1, 0x04df, 0x078e, 0x0ac9, 0x0e86,
 0x12be, 0x1766, 0x1c71, 0x21d4, 0x2782, 0x2d6b, 0x3383, 0x39ba,
 0x3fff, 0x4645, 0x4c7c, 0x5294, 0x587d, 0x5e2b, 0x638e, 0x6899,
 0x6d41, 0x7179, 0x7536, 0x7871, 0x7b20, 0x7d3e, 0x7ec5, 0x7fb1
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[32] =
{
 0x0a3d, 0x0a86, 0x0b5f, 0x0cc6, 0x0eb8, 0x1131, 0x1429, 0x179a,
 0x1b7c, 0x1fc4, 0x2468, 0x295d, 0x2e96, 0x3407, 0x39a2, 0x3f59,
 0x451e, 0x4ae4, 0x509b, 0x5636, 0x5ba7, 0x60e0, 0x65d4, 0x6a79,
 0x6ec1, 0x72a2, 0x7613, 0x790c, 0x7b84, 0x7d76, 0x7ede, 0x7fb7
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[32] =
{
 0x0000, 0x001c, 0x0073, 0x0107, 0x01df, 0x0301, 0x0476, 0x0648,
 0x0881, 0x0b29, 0x0e48, 0x11e6, 0x1607, 0x1aaa, 0x1fd0, 0x2571,
 0x2b85, 0x31fd, 0x38c9, 0x3fd2, 0x4702, 0x4e3d, 0x5565, 0x5c5d,
 0x6303, 0x693a, 0x6ee4, 0x73e4, 0x7821, 0x7b84, 0x7dfd, 0x7f7e
};
#endif

#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 32
//-----------------------------------------------------------------------------
#if (NUM_FFT == 32)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[16] =
{
 0x0000, 0x0800, 0x1000, 0x1800, 0x2000, 0x2800, 0x3000, 0x3800,
 0x4000, 0x4800, 0x5000, 0x5800, 0x6000, 0x6800, 0x7000, 0x7800
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[16] =
{
 0x0000, 0x013a, 0x04df, 0x0ac9, 0x12be, 0x1c71, 0x2782, 0x3383,
 0x3fff, 0x4c7c, 0x587d, 0x638e, 0x6d41, 0x7536, 0x7b20, 0x7ec5
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[16] =
{
 0x0a3d, 0x0b5f, 0x0eb8, 0x1429, 0x1b7c, 0x2468, 0x2e96, 0x39a2,
 0x451e, 0x509b, 0x5ba7, 0x65d4, 0x6ec1, 0x7613, 0x7b84, 0x7ede
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[16] =
{
 0x0000, 0x0073, 0x01df, 0x0476, 0x0881, 0x0e48, 0x1607, 0x1fd0,
 0x2b85, 0x38c9, 0x4702, 0x5565, 0x6303, 0x6ee4, 0x7821, 0x7dfd
};
#endif

#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 16
//-----------------------------------------------------------------------------
#if (NUM_FFT == 16)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[8] =
{
 0x0000, 0x1000, 0x2000, 0x3000, 0x4000, 0x5000, 0x6000, 0x7000
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[8] =
{
 0x0000, 0x04df, 0x12be, 0x2782, 0x3fff, 0x587d, 0x6d41, 0x7b20
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[8] =
{
 0x0a3d, 0x0eb8, 0x1b7c, 0x2e96, 0x451e, 0x5ba7, 0x6ec1, 0x7b84
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[8] =
{
 0x0000, 0x01df, 0x0881, 0x1607, 0x2b85, 0x4702, 0x6303, 0x7821
};
#endif

#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 8
//-----------------------------------------------------------------------------
#if (NUM_FFT == 8)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[4] =
{
 0x0000, 0x2000, 0x4000, 0x6000
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[4] =
{
 0x0000, 0x12be, 0x3fff, 0x6d41
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[4] =
{
 0x0a3d, 0x1b7c, 0x451e, 0x6ec1
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[4] =
{
 0x0000, 0x0881, 0x2b85, 0x6303
};
#endif

#endif


//-----------------------------------------------------------------------------
// Window Functions for NUM_FFT = 4
//-----------------------------------------------------------------------------
#if (NUM_FFT == 4)

#if (WINDOW_TYPE == 1)
// Triangle Window
unsigned int code WindowFunc[2] =
{
 0x0000, 0x4000
};
#endif

#if (WINDOW_TYPE == 2)
// Hanning Window
unsigned int code WindowFunc[2] =
{
 0x0000, 0x3fff
};
#endif


#if (WINDOW_TYPE == 3)
// Hamming Window
unsigned int code WindowFunc[2] =
{
 0x0a3d, 0x451e
};
#endif


#if (WINDOW_TYPE == 4)
// Blackman Window
unsigned int code WindowFunc[2] =
{
 0x0000, 0x2b85
};
#endif

#endif

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------
