//-----------------------------------------------------------------------------
// FIR_Demo.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
// --------------------
//
// This code implements an FIR filter using the coefficients defined in
// FIR_Constants.h.  Please see the seminar slides
// for a detailed discussion of the algorithm.
//
// The ADC samples N samples (default is 500) and runs the FIR filter
// real time, taking advantage of the Mirror property of FIR filters
// (coefficients are mirrored).  After N samples, the program calculates the
// average value and outputs the frequency and output and input averages.
// These values can be used to graph the characteristics of the filter using
// Excel.
//
// How To Test:
// ------------
// 1) Compile and download the code to a 'F362 device on a TOOLSTICK360DC
//    ToolStick board. Alternatively, download the C8051F360_FIR_Demo.hex file.
// 2) Connect P0.1 and P1.1 together on the 'F362 Daughter Card.
// 3) Replace C5 with a 100 pF capacitor.
// 4) Disconnect in the IDE and connect to the TOOLSTICK360DC board in
//    ToolStick Terminal.
// 5) Save the output to a text file and use the Excel worksheet
//    (FIR_graph.xls) to generate the graph.
//
// Target:         C8051F362 (TOOLSTICK360DC)
// Tool chain:     Keil C51 7.50 / SDCC
// Command Line:   None
//
// Release 1.1
//    -Added SDCC support, optimized ADC ISR (BW)
//    -01 MAY 2007
// Release 1.0
//    -Initial Revision (TP / GV)
//    -04 FEB 2007

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "si_toolchain.h"             // Compiler-specific declarations
                                       // (Keil/SDCC)
#include "C8051F360_defs.h"            // SFR declarations
#include <stdio.h>
#include <math.h>
#include "FIR_Constants.h"

//-----------------------------------------------------------------------------
// Global Constants
//-----------------------------------------------------------------------------

#define BAUDRATE     115200            // Baud rate of UART in bps
#define INTCLK       24500000          // Internal oscillator frequency in Hz
#define SYSCLK       98000000          // Output of PLL derived from (INTCLK*2)
#define SAMPLE_RATE  10000             // Sample frequency in Hz

#define N            500               // Number of samples to capture at
                                       // each DAC frequency

// DAC constants
#define OUTPUT_RATE_DAC  100000L       // DAC output rate in Hz

#define PHASE_PRECISION  65536         // Range of phase accumulator

#define START_FREQUENCY  50            // Define the starting frequency
#define STOP_FREQUENCY   4999          // Define the ending frequency
#define FREQ_STEP        10            // Define the number of Hz the frequency
                                       // will step for the frequency sweep

// A full cycle, 16-bit, 2's complement sine wave lookup table
//int code SINE_TABLE[256] = {
SI_SEGMENT_VARIABLE(SINE_TABLE[256], int, code) = {
   0x0000, 0x0324, 0x0647, 0x096a, 0x0c8b, 0x0fab, 0x12c8, 0x15e2,
   0x18f8, 0x1c0b, 0x1f19, 0x2223, 0x2528, 0x2826, 0x2b1f, 0x2e11,
   0x30fb, 0x33de, 0x36ba, 0x398c, 0x3c56, 0x3f17, 0x41ce, 0x447a,
   0x471c, 0x49b4, 0x4c3f, 0x4ebf, 0x5133, 0x539b, 0x55f5, 0x5842,
   0x5a82, 0x5cb4, 0x5ed7, 0x60ec, 0x62f2, 0x64e8, 0x66cf, 0x68a6,
   0x6a6d, 0x6c24, 0x6dca, 0x6f5f, 0x70e2, 0x7255, 0x73b5, 0x7504,
   0x7641, 0x776c, 0x7884, 0x798a, 0x7a7d, 0x7b5d, 0x7c29, 0x7ce3,
   0x7d8a, 0x7e1d, 0x7e9d, 0x7f09, 0x7f62, 0x7fa7, 0x7fd8, 0x7ff6,
   0x7fff, 0x7ff6, 0x7fd8, 0x7fa7, 0x7f62, 0x7f09, 0x7e9d, 0x7e1d,
   0x7d8a, 0x7ce3, 0x7c29, 0x7b5d, 0x7a7d, 0x798a, 0x7884, 0x776c,
   0x7641, 0x7504, 0x73b5, 0x7255, 0x70e2, 0x6f5f, 0x6dca, 0x6c24,
   0x6a6d, 0x68a6, 0x66cf, 0x64e8, 0x62f2, 0x60ec, 0x5ed7, 0x5cb4,
   0x5a82, 0x5842, 0x55f5, 0x539b, 0x5133, 0x4ebf, 0x4c3f, 0x49b4,
   0x471c, 0x447a, 0x41ce, 0x3f17, 0x3c56, 0x398c, 0x36ba, 0x33de,
   0x30fb, 0x2e11, 0x2b1f, 0x2826, 0x2528, 0x2223, 0x1f19, 0x1c0b,
   0x18f8, 0x15e2, 0x12c8, 0x0fab, 0x0c8b, 0x096a, 0x0647, 0x0324,
   0x0000, 0xfcdc, 0xf9b9, 0xf696, 0xf375, 0xf055, 0xed38, 0xea1e,
   0xe708, 0xe3f5, 0xe0e7, 0xdddd, 0xdad8, 0xd7da, 0xd4e1, 0xd1ef,
   0xcf05, 0xcc22, 0xc946, 0xc674, 0xc3aa, 0xc0e9, 0xbe32, 0xbb86,
   0xb8e4, 0xb64c, 0xb3c1, 0xb141, 0xaecd, 0xac65, 0xaa0b, 0xa7be,
   0xa57e, 0xa34c, 0xa129, 0x9f14, 0x9d0e, 0x9b18, 0x9931, 0x975a,
   0x9593, 0x93dc, 0x9236, 0x90a1, 0x8f1e, 0x8dab, 0x8c4b, 0x8afc,
   0x89bf, 0x8894, 0x877c, 0x8676, 0x8583, 0x84a3, 0x83d7, 0x831d,
   0x8276, 0x81e3, 0x8163, 0x80f7, 0x809e, 0x8059, 0x8028, 0x800a,
   0x8000, 0x800a, 0x8028, 0x8059, 0x809e, 0x80f7, 0x8163, 0x81e3,
   0x8276, 0x831d, 0x83d7, 0x84a3, 0x8583, 0x8676, 0x877c, 0x8894,
   0x89bf, 0x8afc, 0x8c4b, 0x8dab, 0x8f1e, 0x90a1, 0x9236, 0x93dc,
   0x9593, 0x975a, 0x9931, 0x9b18, 0x9d0e, 0x9f14, 0xa129, 0xa34c,
   0xa57e, 0xa7be, 0xaa0b, 0xac65, 0xaecd, 0xb141, 0xb3c1, 0xb64c,
   0xb8e4, 0xbb86, 0xbe32, 0xc0e9, 0xc3aa, 0xc674, 0xc946, 0xcc22,
   0xcf05, 0xd1ef, 0xd4e1, 0xd7da, 0xdad8, 0xdddd, 0xe0e7, 0xe3f5,
   0xe708, 0xea1e, 0xed38, 0xf055, 0xf375, 0xf696, 0xf9b9, 0xfcdc,
};

//-----------------------------------------------------------------------------
// MACROS
//-----------------------------------------------------------------------------

#if defined __C51__
#include <intrins.h>
#define NOP() \
   _nop_();
#elif defined SDCC
#define NOP() \
   _asm \
   nop \
   _endasm;
#endif // defined SDCC

// Single FIR_TAP macro takes advantage of mirroring
// (i.e. the FIR coefficients are mirrored, so the coefficient only needs to be
// loaded into the MAC registers once).
#define FIR_TAP_MIRROR(X,Y,Z) \
   MAC0A = X; \
   MAC0BH = Y.u8[MSB]; \
   MAC0BL = Y.u8[LSB]; \
   MAC0BH = Z.u8[MSB]; \
   MAC0BL = Z.u8[LSB];

// Single FIR_TAP macro
#define FIR_TAP(X,Y) \
   MAC0A = X; \
   MAC0BH = Y.u8[MSB]; \
   MAC0BL = Y.u8[LSB];

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------


bit FIR_On = 1;                        // If this bit is a '0', the FIR
                                       // computation is skipped.

bit Sample_Ready = 0;                  // If '1', the value in 'Sample' has
                                       // been updated.

SI_UU16_t Sample;                           // Filter output

// For the frequency sweep
unsigned int Phase_Add = (unsigned int)((unsigned long)((START_FREQUENCY *
                        PHASE_PRECISION) / OUTPUT_RATE_DAC));

// The 'filtered_samples' variable holds the last 500 results of the filter's
// output (used for power calculation)
//xdata int filtered_samples[N];
SI_SEGMENT_VARIABLE(filtered_samples[N], int, xdata);

// For the FIR filter
// 'x' holds the 'delay line' of input samples
//idata SI_UU16_t x[TAPS];
SI_SEGMENT_VARIABLE(x[TAPS], SI_UU16_t, idata);

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void SYSCLK_Init (void);               // Configure system clock
void PORT_Init (void);                 // Configure port output
void UART0_Init (void);                // Configure UART operation
void ADC0_Init (void);                 // Configure ADC
void IDA0_Init (void);                 // Configure IDAC
void Timer2_Init (int counts);         // Configure Timer 2
void Timer3_Init (int counts);         // Configure Timer 3

int RMS_Calc (int *input_samples, int num_samples);

void Set_DAC_Frequency (unsigned long frequency);

// Define the UART printing functions
#if defined __C51__
char putchar (char c);                 // Define putchar for Keil
#elif defined SDCC
void putchar (char c);
#endif // defined SDCC

SI_INTERRUPT_PROTO(Timer3_ISR, INTERRUPT_TIMER3);
SI_INTERRUPT_PROTO(ADC0_ISR, INTERRUPT_ADC0_EOC);

// Similar to STARTUP.A51 for Keil, this function stub for SDCC allows us to
// disable the WDT before memory is initialized.
#if defined SDCC
void _sdcc_external_startup (void);

void _sdcc_external_startup (void)
{
   PCA0MD &= ~0x40;                    // Disable WDT in startup
}
#endif // defined SDCC

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main (void)
{
   int RMS_Value = 0;
   int average = 0;
   unsigned int sample_count;
   unsigned long frequency;
   unsigned int count;

   PCA0MD &= ~0x40;                    // WDTE = 0 (clear watchdog timer
                                       // enable)

   SYSCLK_Init ();                     // Initialize oscillator
   PORT_Init ();                       // Initialize crossbar and GPIO
   UART0_Init ();                      // Initialize UART1

   // Initialize Timer2 to overflow at the ADC sample rate
   Timer2_Init (SYSCLK/SAMPLE_RATE);

   // Initialize Timer3 to overflow at the DAC sample rate
   Timer3_Init (SYSCLK/OUTPUT_RATE_DAC);

   IDA0_Init ();                       // Initialize the IDAC
   ADC0_Init ();                       // Initialize the ADC

   AD0EN = 1;                          // Enable ADC

   MAC0CF = 0x06;                      // Fractional mode; Saturate
                                       // enabled
   // Initialize the delay line for the FIR filter
   for (count = 0; count < TAPS; count++)
   {
      x[count].s16 = 0;
   }

   // Initialize the sample array
   for (count = 0; count < N; count ++)
   {
      filtered_samples[count] = 0;
   }

   EA = 1;                             // Enable global interrupts

   // Perform the frequency sweep
   // For each frequency, calculate the RMS Value for the output of the
   // FIR filter and the RMS Value for the input of the FIR filter.
   for (frequency = START_FREQUENCY; frequency <= STOP_FREQUENCY;
        frequency += FREQ_STEP)
   {
      Set_DAC_Frequency (frequency);

      // Collect some filtered samples
      FIR_On = 1;                   // Turn on FIR filter
      for (sample_count = 0; sample_count < N; sample_count++)
      {
         Sample_Ready = 0;
         while (!Sample_Ready);

         EA = 0; EA = 0;
         Sample_Ready = 0;
         filtered_samples[sample_count] = Sample.u16;
         EA = 1;
      }
      RMS_Value = RMS_Calc((int *)filtered_samples, N);
      printf("%ld\t%d\t",frequency, RMS_Value);

      // Collect some unfiltered samples
      FIR_On = 0;                   // Turn off FIR filter
      for (sample_count = 0; sample_count < N; sample_count++)
      {
         Sample_Ready = 0;
         while (!Sample_Ready);

         EA = 0; EA = 0;
         Sample_Ready = 0;
         filtered_samples[sample_count] = Sample.u16;
         EA = 1;
      }
      RMS_Value = RMS_Calc((int *)filtered_samples, N);
      printf("%d\n",RMS_Value);
   }

   while (1);
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// This routine initializes the system clock to use the internal 24.5MHz*4
// oscillator as its clock source.
//
//-----------------------------------------------------------------------------
void SYSCLK_Init (void)
{
   char i;
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   OSCICN = 0x83;

   // Step 2. Set the PLLSRC bit (PLL0CN.2) to select the desired
   // clock source for the PLL.
   PLL0CN &= ~0x04;                    // Internal oscillator

   // Step 3. Program the Flash read timing bits, FLRT (FLSCL.5-4) to the
   // appropriate value for the new clock rate (see Section 15. Flash Memory
   // on page 199).
   SFRPAGE = LEGACY_PAGE;
   FLSCL |= 0x30;                      // >= 100 MHz
   SFRPAGE = CONFIG_PAGE;

   // Step 4. Enable power to the PLL by setting PLLPWR (PLL0CN.0) to 1.
   PLL0CN |= 0x01;

   // Step 5. Program the PLL0DIV register to produce the divided reference
   // frequency to the PLL.
   PLL0DIV = 0x01;

   // Step 6. Program the PLLLP3-0 bits (PLL0FLT.3-0) to the appropriate
   // range for the divided reference frequency.
   PLL0FLT |= 0x01;

   // Step 7. Program the PLLICO1-0 bits (PLL0FLT.5-4) to the appropriate
   // range for the PLL output frequency.
   PLL0FLT &= ~0x30;

   // Step 8. Program the PLL0MUL register to the desired clock multiplication
   // factor.
   PLL0MUL = 0x04;

   // Step 9. Wait at least 5 s, to provide a fast frequency lock.
   for (i = 100; i > 0; i--);

   // Step 10. Enable the PLL by setting PLLEN (PLL0CN.1) to 1.
   PLL0CN |= 0x02;

   // Step 11. Poll PLLLCK (PLL0CN.4) until it changes from 0 to 1.
   while ((PLL0CN & 0x10) != 0x10);

   // Step 12. Switch the System Clock source to the PLL using the CLKSEL
   // register.
   CLKSEL = 0x04;

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the Crossbar and GPIO ports.
//
// Pinout:
//
// P0.1 - IDAC0 output
//
// P0.4 - UART TX (push-pull)
// P0.5 - UART RX
//
// P1.1 - ADC0 analog input
//
// Note: for the purposes of this example, P0.1 and P1.1 are tied together
// on the F362 ToolStick Daughter Card board.
//
//-----------------------------------------------------------------------------
void PORT_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   P0MDOUT |= 0x10;                    // Set TX pin to push-pull

   P0MDIN &= ~0x02;                    // Set P0.1 as an analog pin (IDAC)
   P1MDIN &= ~0x02;                    // Set P1.1 as an analog pin (ADC)

   P0SKIP = 0x01;                      // Skip P0.1 for the IDAC output
   P1SKIP = 0x01;                      // Skip P1.1 for the ADC input

   XBR0 = 0x01;                        // Enable UART0
   XBR1 = 0x40;                        // Enable crossbar and weak pull-ups

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//
//-----------------------------------------------------------------------------
void UART0_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   SCON0 = 0x10;                       // SCON0: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits
   if (SYSCLK/BAUDRATE/2/256 < 1)
   {
      TH1 = (char) -(SYSCLK/BAUDRATE/2);
      CKCON |=  0x08;                  // T1M = 1; SCA1:0 = xx
   }
   else if (SYSCLK/BAUDRATE/2/256 < 4)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 01
      CKCON |=  0x01;
   }
   else if (SYSCLK/BAUDRATE/2/256 < 12)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 00
   }
   else if (SYSCLK/BAUDRATE/2/256 < 48)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 10
      CKCON |=  0x02;
   }
   else
   {
      while (1);                       // Error.  Unsupported baud rate
   }

   TL1 = TH1;                          // Init Timer1
   TMOD &= ~0xF0;                      // TMOD: Timer 1 in 8-bit autoreload
   TMOD |=  0x20;
   TR1 = 1;                            // START Timer1
   TI0 = 1;                            // Indicate TX0 ready

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configures ADC0 to make single-ended analog measurements on pin P1.1 for the
// FIR filter.
//
//-----------------------------------------------------------------------------
void ADC0_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   ADC0CN = 0x02;                      // ADC0 disabled, normal tracking,
                                       // conversion triggered on TMR2 overflow

   REF0CN = 0x03;                      // Enable on-chip VREF and buffer

   AMX0P = 0x01;                       // ADC0 positive input = P1.1
   AMX0N = 0x1F;                       // ADC0 negative input = GND
                                       // i.e., single ended mode

   ADC0CF = ((SYSCLK/3000000)-1)<<3;   // Set SAR clock to 3MHz

   ADC0CF |= 0x04;                     // Left-justify results

   EIE1 |= 0x08;                       // Enable ADC0 conversion complete int.

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Timer2_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:
//   1) counts - the number of timer clocks to count before a timer interrupt
//           should occur
//
// Configure Timer2 to 16-bit auto-reload and generate an interrupt at
// <SAMPLE_RATE>.
//
// Timer 2 overflow automatically triggers ADC0 conversion.
//
//-----------------------------------------------------------------------------
void Timer2_Init (int counts)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   TMR2CN = 0x00;                      // Stop Timer2; Clear TF2;
                                       // use SYSCLK as timebase, 16-bit
                                       // auto-reload
   CKCON |= 0x10;                      // Select SYSCLK for Timer 2 source
   TMR2RL = -counts;                   // Init reload value for <SAMPLE_RATE>
   TMR2 = TMR2RL;                      // Init Timer 2 counter
   TR2 = 1;                            // Start Timer2

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  int counts - calculated Timer overflow rate
//                    range is positive range of integer: 0 to 32767
//
// Configure Timer3 to auto-reload at interval specified by <counts>
// using SYSCLK as its time base.  Interrupts are enabled.
//
// Timer3 generates the update rate for the IDAC.
//
//-----------------------------------------------------------------------------
void Timer3_Init(int counts)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   TMR3CN  = 0x00;                     // Resets Timer 3, sets to 16 bit mode
   CKCON  |= 0x40;                     // Use system clock

   TMR3RL  = -counts;                  // Initial reload value
   TMR3    = TMR3RL;                   // Initialize Timer value

   EIE1   |= 0x80;                     // Enable Timer 3 interrupts
   EIP1   |= 0x80;                     // Enable Timer 3 interrupts as high
                                       // priority
   TMR3CN  = 0x04;                     // Start Timer 3

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// IDA0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Enable IDAC0 (VREF is already initialized in ADC0_Init).
//
//-----------------------------------------------------------------------------
void IDA0_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   IDA0CN = 0xB2;                      // Enable IDAC0 for 2.0 mA full-scale
                                       // output; updated on Timer3 overflows

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Timer3_ISR
//-----------------------------------------------------------------------------
//
// This ISR is called on Timer3 overflows.  Timer3 is set to auto-reload mode
// and is used to schedule the DAC output sample rate in this example.
// Note that the value that is written to the IDAC during this ISR call is
// actually transferred to the IDAC at the next Timer3 overflow.
//
//-----------------------------------------------------------------------------
//void Timer3_ISR (void) interrupt 14
SI_INTERRUPT(Timer3_ISR, INTERRUPT_TIMER3)
{
   static SI_UU16_t phase_acc = {0};        // Holds phase accumulator
   int temp1;                          // The temporary value that passes
                                       // through 3 stages before being written
                                       // to the IDAC
   TMR3CN &= ~0x80;                    // Clear Timer3 overflow flag

   phase_acc.u16 += Phase_Add;         // Increment phase accumulator

   temp1 = SINE_TABLE[phase_acc.u8[MSB]];

   // Add a DC bias to make the rails 0 to 65535
   // Note: the XOR with 0x8000 translates the bipolar quantity into
   // a unipolar quantity.

   IDA0 = 0x8000 ^ temp1;
}

//-----------------------------------------------------------------------------
// ADC0_ISR
//-----------------------------------------------------------------------------
//
// ADC0 end-of-conversion ISR
//
// This interrupt service routine is called on ADC0 conversion complete.
// The ADC result is converted to signed and stored in the FIR delay line.
//
// If the global <FIR_On> bit is set to a '1', then the FIR output is computed
// and stored in the global variable 'Sample'.  The 'Sample_Ready' indicator
// bit is set to indicate the value is ready.
//
// If <FIR_On> is set to '0', then the ADC sample is copied to the global
// 'Sample' variable.  The 'Sample_Ready' indicator bit is set to indicate
// that the value is ready.
//
//-----------------------------------------------------------------------------
//void ADC0_ISR (void) interrupt 10
SI_INTERRUPT(ADC0_ISR, INTERRUPT_ADC0_EOC)
{
   volatile SI_UU16_t input;
   static unsigned char delay_index = 0;
   unsigned char coeff_index, sample_index, opposite_sample_index;

   AD0INT = 0;                         // Clear ADC conversion complete
                                       // indicator

   input.s16 = ADC0^0x8000;            // Convert to bipolar value

   // Store ADC result in the delay line
   x[delay_index].u16 = input.u16;
   sample_index = delay_index;         // Sample_index points to newest data

   // Update delay index
   if (delay_index == (TAPS - 1))
   {
      delay_index = 0;
   }
   else
   {
      delay_index++;
   }

   // If FIR is On, compute filter result, otherwise send unfiltered sample
   if (FIR_On == 1)
   {
      MAC0CF |= 0x08;                  // Clear accumulator

      // Mirror algorithm
      if (sample_index == TAPS - 1)
      {
         opposite_sample_index = 0;
      }
      else
      {
         opposite_sample_index = sample_index + 1;
      }

      for (coeff_index = 0; coeff_index < (TAPS / 2); coeff_index++)
      {

         FIR_TAP_MIRROR (B_FIR[coeff_index].u16, x[sample_index],
         x[opposite_sample_index]);

         if (sample_index == 0)
         {
            sample_index = TAPS - 1;
         }
         else
         {
            sample_index--;
         }

         if (opposite_sample_index == TAPS - 1)
         {
            opposite_sample_index = 0;
         }
         else
         {
            opposite_sample_index++;
         }
      }

      if ((TAPS % 2) == 1)             // Handle middle tap of odd order filter
      {
         FIR_TAP (B_FIR[coeff_index].u16, x[sample_index]);
         NOP ();
         NOP ();
         NOP ();
      }

      Sample.u16 = MAC0RND;
   }
   else
   {
      Sample.u16 = input.u16;
   }

   Sample_Ready = 1;                   // Post Sample availability
}

//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Set_DAC_Frequency
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    <frequency>
//   1)  unsigned long frequency - the new frequency for the IDAC frequency
//               sweep
//                    range is 50 to 5000 due to the ADC sampling frequency
//
// This routine sets the DAC to output a sinewave at the given <frequency>.
//
//-----------------------------------------------------------------------------
void Set_DAC_Frequency (unsigned long frequency)
{
   Phase_Add = (unsigned int)((unsigned long)((frequency *
                PHASE_PRECISION) / OUTPUT_RATE_DAC));
}

/*-----------------------------------------------------------------------------
// RMS_Calc
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1) input_samples - pointer to an array of the data to be used for the RMS
//                        Value calculations
//   2) num_samples - the number of data elements in the <input_samples> array
//
// This routine takes a pointer to an array and a number of samples and first
// computes the average value of the data.  Then, it uses this average to
// calculate the RMS Value by using the following equation:
//
//                 N
//                 __
//                 \
//                 /_  (x-x_avg)^2
//                 n=10
// RMS_Value^2 =  -----------------
//                       N-10
//
//
// The above routine skips the first <TAPS> samples where the filter hasn't quite
// settled.
//---------------------------------------------------------------------------*/
int RMS_Calc (int *input_samples, int num_samples)
{
   int count;
   float average;
   float RMS_summation = 0;
   float RMS_Value;
   float temp;


   // Calculate the average value (x_avg) of the <input_samples> array
   average = 0.0;

   for (count = TAPS; count < num_samples; count++)
   {
      average += (float) input_samples[count];
   }
   average = (float)(average / (num_samples-TAPS));

   // Calculate the RMS Value using the average computed above

   // Calculate the sum from 1 to N of (x-x_avg)^2
   for (count = TAPS; count < num_samples; count++)
   {
      // calculate difference from mean
      temp = input_samples[count] - average;
      // square it
      temp *= temp;
      // and add it to sum
      RMS_summation += temp;
   }

   // Calculate sum from above / N
   RMS_summation = (float)RMS_summation / (num_samples-TAPS);

#if defined __C51__
   RMS_Value = sqrt(RMS_summation);
#elif defined SDCC
   RMS_Value = sqrtf(RMS_summation);
#endif

   return (int)RMS_Value;
}

//-----------------------------------------------------------------------------
// putchar
//-----------------------------------------------------------------------------
//
// Return Value:
//   1) char c - returns the char c that was passed as a parameter
// Parameters:
//   1) char c - the character to be printed
//
// Print the character <c> using the UART at <BAUDRATE>.
//
//-----------------------------------------------------------------------------
#if defined __C51__
char putchar (char c)
#elif defined SDCC
void putchar (char c)
#endif
{
   if (c == '\n')                      // Print a carriage return
   {
      while (!TI0);
      TI0 = 0;
      SBUF0 = 0x0d;
   }
   while (!TI0);
   TI0 = 0;
   SBUF0 = c;
#if defined __C51__
   return c;                           // Print the character
#endif
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------