//-----------------------------------------------------------------------------
// C8051F12x_DTMF_Demo.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
// --------------------
//
// This code implements the DTMF Tone generation using the DAC a Goertzel
// algorithm for DTMF decoding.  Please see AN219 for a detailed discussion of
// the algorithm.
//
// The DAC outputs the tones based on a SINE table in code memory.  The two
// parts to the tone are added together to create the combined tones.
//
// The ADC samples N samples (default is 200 for the base frequencies and 250
// for the 2nd harmonics) and runs the Goertzel filters real time.  After N
// samples, the ADC ISR calculates the magnitude of each filter and checks
// for one row tone and one column tone.  Note that both the filters and
// magnitude calculations are completed before the next ADC sample and in
// real time.
//
// This example uses the MAC for the filters and magnitude calculations.
//
// Pinout:
//
// P0.4 - UART0 TX pin
// P0.5 - UART0 RX pin
//
// DAC0 - DAC0 output (DTMF tone)
//
// AIN0.0 - ADC0 input
//
//
// How To Test:
// ------------
// 1) Compile and download the code to a 'F12x device (or download the
//    C8051F12x_DTMF_Demo.hex file).  The code was written for the 'F12x
//    Target Board
// 2) Connect to the 'F12x using a Terminal program (like Hyperterminal) at
//    <BAUDRATE> and 8N1.
// 3) Connect the DAC output pin to the ADC input pin (DAC0, pin 3 on J1, to
//    AIN0.0, pin 7 on J1, on the 'F12x Target Board).
// 4) Navigate through the UART menu options.
//
// Target:         C8051F12x ('F120 Target Board)
// Tool chain:     Keil C51 7.50 / SDCC 7.1
// Command Line:   None
//
// Release 1.0
//    -Initial Revision (TP / SYRO)
//    -1 JUN 2007

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "si_toolchain.h"             // Compiler-specific declarations
                                       // (Keil/SDCC)
#include "C8051F120_defs.h"            // SFR declarations
#include <stdio.h>

// For Keil...
#if defined __C51__                    // Define a "nop" in Keil
   #include <intrins.h>
   #define NOP() \
      _nop_();

// For SDCC...
#elif defined SDCC                     // Define a "nop" in SDCC
   #define NOP() \
   _asm \
       nop \
   _endasm;
#endif

//-----------------------------------------------------------------------------
// Global Constants
//-----------------------------------------------------------------------------

#define SYSCLK       98000000          // SYSCLK frequency in Hz
#define BAUDRATE     115200            // UART0 baud rate
#define SAMPLE_RATE  8000              // Sample rate for ADC0 conversions
#define XMIN         20                // Threshold value for the difference
                                       // between two succesive samples
                                       // (8-bit data from the 12-bit ADC)

// ADC constants
char code dtmfchar[16] =               // DTMF characters
{
    '1','2','3','A',
    '4','5','6','B',
    '7','8','9','C',
    '*','0','#','D'
};

code int coef1[2] = { 235, 437};       // Pre-calculated Goertzel filter
code int coef2[2] = { 181, 421};       // coefficients
code int coef3[2] = { 118, 402};
code int coef4[2] = {  47, 378};
code int coef5[2] = {-165, 298};
code int coef6[2] = {-258, 255};
code int coef7[2] = {-349, 204};
code int coef8[2] = {-429, 146};

// DAC constants
#define SAMPLE_RATE_DAC   100000L      // Update rate of DAC in Hz

#define PHASE_PRECISION   65536        // Range of phase accumulator

// DTMF phase adder values based on SAMPLE_RATE_DAC and PHASE_PRECISION
#define LOW_697           697 * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_770           770 * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_852           852 * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_941           941 * PHASE_PRECISION / SAMPLE_RATE_DAC

#define HIGH_1209         1209 * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1336         1336 * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1477         1477 * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1633         1633 * PHASE_PRECISION / SAMPLE_RATE_DAC

char code SINE_TABLE[256] = {
   0x00, 0x03, 0x06, 0x09, 0x0c, 0x0f, 0x12, 0x15,
   0x18, 0x1c, 0x1f, 0x22, 0x25, 0x28, 0x2b, 0x2e,
   0x30, 0x33, 0x36, 0x39, 0x3c, 0x3f, 0x41, 0x44,
   0x47, 0x49, 0x4c, 0x4e, 0x51, 0x53, 0x55, 0x58,
   0x5a, 0x5c, 0x5e, 0x60, 0x62, 0x64, 0x66, 0x68,
   0x6a, 0x6c, 0x6d, 0x6f, 0x70, 0x72, 0x73, 0x75,
   0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7c,
   0x7d, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
   0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7e, 0x7e,
   0x7d, 0x7c, 0x7c, 0x7b, 0x7a, 0x79, 0x78, 0x77,
   0x76, 0x75, 0x73, 0x72, 0x70, 0x6f, 0x6d, 0x6c,
   0x6a, 0x68, 0x66, 0x64, 0x62, 0x60, 0x5e, 0x5c,
   0x5a, 0x58, 0x55, 0x53, 0x51, 0x4e, 0x4c, 0x49,
   0x47, 0x44, 0x41, 0x3f, 0x3c, 0x39, 0x36, 0x33,
   0x30, 0x2e, 0x2b, 0x28, 0x25, 0x22, 0x1f, 0x1c,
   0x18, 0x15, 0x12, 0x0f, 0x0c, 0x09, 0x06, 0x03,
   0x00, 0xfd, 0xfa, 0xf7, 0xf4, 0xf1, 0xee, 0xeb,
   0xe8, 0xe4, 0xe1, 0xde, 0xdb, 0xd8, 0xd5, 0xd2,
   0xd0, 0xcd, 0xca, 0xc7, 0xc4, 0xc1, 0xbf, 0xbc,
   0xb9, 0xb7, 0xb4, 0xb2, 0xaf, 0xad, 0xab, 0xa8,
   0xa6, 0xa4, 0xa2, 0xa0, 0x9e, 0x9c, 0x9a, 0x98,
   0x96, 0x94, 0x93, 0x91, 0x90, 0x8e, 0x8d, 0x8b,
   0x8a, 0x89, 0x88, 0x87, 0x86, 0x85, 0x84, 0x84,
   0x83, 0x82, 0x82, 0x81, 0x81, 0x81, 0x81, 0x81,
   0x80, 0x81, 0x81, 0x81, 0x81, 0x81, 0x82, 0x82,
   0x83, 0x84, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
   0x8a, 0x8b, 0x8d, 0x8e, 0x90, 0x91, 0x93, 0x94,
   0x96, 0x98, 0x9a, 0x9c, 0x9e, 0xa0, 0xa2, 0xa4,
   0xa6, 0xa8, 0xab, 0xad, 0xaf, 0xb2, 0xb4, 0xb7,
   0xb9, 0xbc, 0xbf, 0xc1, 0xc4, 0xc7, 0xca, 0xcd,
   0xd0, 0xd2, 0xd5, 0xd8, 0xdb, 0xde, 0xe1, 0xe4,
   0xe8, 0xeb, 0xee, 0xf1, 0xf4, 0xf7, 0xfa, 0xfd
};

//-----------------------------------------------------------------------------
// Macros
//-----------------------------------------------------------------------------

// The MAC_Feedback Macro implements the following code:
// Qx[0] = (coeff * Qx[1]) - Qx[2] + x
//
// void MAC_Feedback(SI_UU16_t *Qx, int coeff)
// {
//    // Qhold.l = (long)coefx[base_freq]*(long)Qx[1].i;
//    MAC0A = coeff;
//    MAC0BH = Qx[1].s8[MSB];
//    MAC0BL = Qx[1].s8[LSB];          // First multiply and acc initiated
//
//    _nop_();                         // Wait for the accumulator to be ready
//    _nop_();
//
//    // Qx[0].i = x;
//    // Qx[0].i += Qhold.hold.intval;
//    // Move MAC0ACC1 to MAC0ACC0 and MAC0ACC2 to MAC0ACC1 to access the "middle"
//    // 16-bits (lowest 8 bits are truncated)
//    MAC0ACC0 = MAC0ACC1;
//    MAC0ACC1 = MAC0ACC2;
//    MAC0A = x;
//    MAC0BH = 0x00;
//    MAC0BL = 1;                      // Second multiply and acc initiated
//
//    // Qx[0].i -= Qx[2].i;
//    MAC0A = -1;
//    MAC0BH = Qx[2].s8[MSB];
//    MAC0BL = Qx[2].s8[LSB];          // Third multiply and acc initiated
//
//    _nop_();                         // Wait two SYSCLKs for the final
//    _nop_();                         // results in the accumulator
//
//    Qx[0].s16 = MAC0ACCL;
//    Qx[2].s16 = Qx[1].s16;
//    Qx[1].s16 = Qx[0].s16;
//
//    MAC0CF |= 0x08;                  // Clear the accumulator
// }
//
// NOTE: <Qx> must be a SI_UU16_t type
#define MAC_Feedback(Qx, coeff) \
   MAC0A = coeff; \
   MAC0BH = Qx[1].s8[MSB]; \
   MAC0BL = Qx[1].s8[LSB]; \
   NOP(); \
   NOP(); \
   MAC0ACC0 = MAC0ACC1; \
   MAC0ACC1 = MAC0ACC2; \
   MAC0A = x; \
   MAC0BH = 0x00; \
   MAC0BL = 1; \
   MAC0A = -1; \
   MAC0BH = Qx[2].s8[MSB]; \
   MAC0BL = Qx[2].s8[LSB]; \
   NOP(); \
   NOP(); \
   Qx[0].s16 = MAC0ACCL; \
   Qx[2].s16 = Qx[1].s16; \
   Qx[1].s16 = Qx[0].s16; \
   MAC0CF |= 0x08;

// The MAC_Magnitude Macro implements the following code:
// Power =  Qx[1]^2 + Qx[2]^2 - (coeff * Qx[1] * Qx[2])
//
// void MAC_Magnitude(SI_UU16_t *Qx, int coeff, unsigned int result)
// {
//
//    // "Right shift" the Qx[1] and Qx[2] data by 8 bits by moving the byte
//    // from the MSB to the LSB and sign extending
//    // So Qx[1] = Qx[1] >> 8
//    // and Qx[2] = Qx[2] >> 8
//    Qx[1].s8[LSB] = Qx[1].s8[MSB];
//    if ((Qx[1].s8[MSB] & 0x80) == 0x80)
//    {
//       Qx[1].s8[MSB] = 0xFF;         // Sign extend to the MSB, if necessary
//    }
//    else
//    {
//       Qx[1].s8[MSB] = 0x00;
//    }
//    Qx[2].s8[LSB] = Qx[2].s8[MSB];
//    if ((Qx[2].s8[MSB] & 0x80) == 0x80)
//    {
//       Qx[2].s8[MSB] = 0xFF;         // Sign extend to the MSB, if necessary
//    }
//    else
//    {
//       Qx[2].s8[MSB] = 0x00;
//    }
//
//    MAC0CF |= 0x80;                  // Clear the accumulator
//
//    // Calculate coeff * -1
//    // since -coeff * Qx[1] * Qx[2] =  - (coeff*Qx[1]*Qx[2])
//    MAC0A = coeff;
//    MAC0BH = 0xFF;
//    MAC0BL = 0xFF;
//
//    MAC0A = Qx[1].s16;               // Preload the A register with Qx[1]
//                                     // while the previous multiplication
//                                     // executes
//
//    temp_value.s16 = MAC0ACCL;       // Store -coeff
//
//    MAC0CF |= 0x08;                  // Clear the accumulator
//
//    // Calculate Qx[1] * Qx[2]
//    MAC0BH = Qx[2].s8[MSB];
//    MAC0BL = Qx[2].s8[LSB];
//    _nop_();
//    _nop_();
//
//    // Calculate Qx[1] * Qx[2] * -coeff
//    MAC0A = MAC0ACCL;
//    MAC0CF |= 0x08;                  // Clear the accumulator
//    MAC0BH = temp_value.s8[MSB];
//    MAC0BL = temp_value.s8[LSB];
//    _nop_();
//    _nop_();
//
//    // Use the middle two bytes of the result and sign extend the
//    // Rest of the accumulator
//    MAC0ACC0 = MAC0ACC1;
//    MAC0ACC1 = MAC0ACC2;
//    if ((MAC0ACC2 & 0x80) == 0x80)
//    {
//       MAC0ACC2 = 0xFF;
//       MAC0ACC3 = 0xFF;
//    }
//    else
//    {
//       MAC0ACC2 = 0x00;
//       MAC0ACC3 = 0x00;
//    }
//
//    // Accumulator now holds the -(coeff * Qx[1] * Qx[2]) term
//
//    // Calculate Qx[1]^2 and accumulate
//    MAC0A = Qx[1].s16;
//    MAC0BH = Qx[1].s8[MSB];
//    MAC0BL = Qx[1].s8[LSB];
//
//    // Calculate Qx[2]^2 and accumulate
//    MAC0A = Qx[2].s16;
//    MAC0BH = Qx[2].s8[MSB];
//    MAC0BL = Qx[2].s8[LSB];
//    _nop_();
//    _nop_();
//
//    result = MAC0ACCL;               // result is Qx[1]^2 + Qx[2]^2 -
//                                     // (coeff * Qx[1] * Qx[2])
//
//    MAC0CF |= 0x08;                  // Clear the accumulator
//
// NOTE: <Qtx> must be a SI_UU16_t type and <result> must be an unsigned integer
// NOTE: <temp_value> must be a SI_UU16_t type and must be declared before
// the macro is used.
#define MAC_Magnitude(Qx, coeff, result) \
   Qx[1].s8[LSB] = Qx[1].s8[MSB]; \
   if ((Qx[1].s8[MSB] & 0x80) == 0x80) \
   { \
      Qx[1].s8[MSB] = 0xFF; \
   } \
   else \
   { \
      Qx[1].s8[MSB] = 0x00; \
   } \
   Qx[2].s8[LSB] = Qx[2].s8[MSB]; \
   if ((Qx[2].s8[MSB] & 0x80) == 0x80) \
   { \
      Qx[2].s8[MSB] = 0xFF; \
   } \
   else \
   { \
      Qx[2].s8[MSB] = 0x00; \
   } \
   MAC0CF |= 0x80; \
   MAC0A = coeff; \
   MAC0BH = 0xFF; \
   MAC0BL = 0xFF; \
   MAC0A = Qx[1].s16; \
   temp_value.s16 = MAC0ACCL; \
   MAC0CF |= 0x08; \
   MAC0BH = Qx[2].s8[MSB]; \
   MAC0BL = Qx[2].s8[LSB]; \
   NOP(); \
   NOP(); \
   MAC0A = MAC0ACCL; \
   MAC0CF |= 0x08; \
   MAC0BH = temp_value.s8[MSB]; \
   MAC0BL = temp_value.s8[LSB]; \
   NOP(); \
   NOP(); \
   MAC0ACC0 = MAC0ACC1; \
   MAC0ACC1 = MAC0ACC2; \
   if ((MAC0ACC2 & 0x80) == 0x80) \
   { \
      MAC0ACC2 = 0xFF; \
      MAC0ACC3 = 0xFF; \
   } \
   else \
   { \
      MAC0ACC2 = 0x00; \
      MAC0ACC3 = 0x00; \
   } \
   MAC0A = Qx[1].s16; \
   MAC0BH = Qx[1].s8[MSB]; \
   MAC0BL = Qx[1].s8[LSB]; \
   MAC0A = Qx[2].s16; \
   MAC0BH = Qx[2].s8[MSB]; \
   MAC0BL = Qx[2].s8[LSB]; \
   NOP(); \
   NOP(); \
   result = MAC0ACCL; \
   MAC0CF |= 0x08;

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

// ADC Variables
idata SI_UU16_t Q1[3];                      // These are the elements of the 8
idata SI_UU16_t Q2[3];                      // Goertzel filters.  The filters are
idata SI_UU16_t Q3[3];                      // 2 pole IIR filters and so require
idata SI_UU16_t Q4[3];                      // 3 terms each.
idata SI_UU16_t Q5[3];
idata SI_UU16_t Q6[3];
idata SI_UU16_t Q7[3];
idata SI_UU16_t Q8[3];

idata unsigned int mag_squared1, mag_squared2; // Store output of the
idata unsigned int mag_squared3, mag_squared4; // Goertzel filters.
idata unsigned int mag_squared5, mag_squared6;
idata unsigned int mag_squared7, mag_squared8;


unsigned char TMR0RLH, TMR0RLL;        // Timer0 reload value

bit new_tone;                          // Flag for valid pause between tones

bit start_goertzel;                    // Flag for start of the decoding
                                       // process

char base_freq;                        // Flag for base frequencies /
                                       // 2nd harmonics

unsigned char sample_no;               // Sample counter

unsigned char max_sample;              // Total no. of samples for base freqs
                                       // and 2nd harmonics

unsigned char sig_present;             // For every frequency detected, a bit
                                       // in sig_present is set

unsigned char set1;                    // Hold the no. of freqs. detected
unsigned char set2;

unsigned char dtmf_index;              // Index for dtmfchar array

bit done;                              // Flag for starting character decoding

bit tone_detected = 0;                 // This bit is used by the ADC ISR for
                                       // indicating when a tone has been
                                       // detected.  The tone is the printed
                                       // in main().

// ADC Variables for gain calculation and signal detection
bit gain_calc;                         // Flag for gain computing


int x;                                 // Current signal sample
int x_old;                             // Former signal sample

int high_x, low_x;                     // Minimum and maximum value of the
                                       // signal
int delta_x;

int gain;                              // Gain computed in AGC block

unsigned char gain_cnt;                // Gain stage sample counter

// DAC Variables
bit tone_generated = 0;

unsigned int phase_add1;               // Holds low-tone phase adder
unsigned int phase_add2;               // Holds low-tone phase adder

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------

void SYSCLK_Init(void);                // System clock configuration
void PORT_Init(void);                  // I/O port configuration
void Timer0_Init(void);                // Timer0 configuration
void Timer3_Init(void);                // Timer3 configuration
void Timer4_Init(int counts);          // Configure Timer4
void ADC0_Init(void);                  // ADC0 configuration
void DAC0_Init(void);                  // Configure DAC0
void UART1_Init(void);                 // UART1 configuration
void MAC0_Init(void);                  // Configure MAC0
void Interrupt_Init(void);             // Interrupt configuration
void DTMF_Init(void);                  // Variable initializations

// Define the UART printing functions for Keil/SDCC
#if defined __C51__
char putchar (char c);                 // Define putchar for Keil
#elif defined SDCC
void putchar (char c);                 // Define putchar for SDCC
#endif

char getkey (void);

// Interrupt Service Routines
SI_INTERRUPT_PROTO_USING(ADC0_ISR, INTERRUPT_ADC0_EOC, 1);
SI_INTERRUPT_PROTO_USING(Timer0_ISR, INTERRUPT_TIMER0, 2);
SI_INTERRUPT_PROTO_USING(Timer4_ISR, INTERRUPT_TIMER4, 3);

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
//
void main(void)
{
   bit invalid_character = 0;
   unsigned char input_character;

   WDTCN = 0xDE;                       // Disable watchdog timer
   WDTCN = 0xAD;

   SYSCLK_Init();                      // Configure system clock
   PORT_Init();                        // Condigure I/O port
   Timer0_Init();                      // Configure Timer0
   Timer3_Init();                      // Configure Timer3
   Timer4_Init(SYSCLK/SAMPLE_RATE_DAC);// Using Timer4 as update scheduler
   ADC0_Init();                        // Configure ADC0
   DAC0_Init();
   UART1_Init();                       // Configure UART1
   MAC0_Init();                        // Configure MAC0
   Interrupt_Init();                   // Initialize interrupts
   DTMF_Init();                        // Variable initializations

   printf ("\fWelcome to the DTMF Tone generator and decoder.\n\n");

   while(1)
   {
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| 1 | | 2 | | 3 | | A |\n");
      printf ("|___| |___| |___| |___|\n\n");
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| 4 | | 5 | | 6 | | B |\n");
      printf ("|___| |___| |___| |___|\n\n");
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| 7 | | 8 | | 9 | | C |\n");
      printf ("|___| |___| |___| |___|\n\n");
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| * | | 0 | | # | | D |\n");
      printf ("|___| |___| |___| |___|\n\n");

      while (tone_generated == 1);     // Wait until the current tone is
                                       // done being generated before
                                       // accepting a new value

      printf ("Enter the touch-tone number: ");
      input_character = getkey ();

      invalid_character = 0;

      // Determine the two DTMF tones for the desired number
      switch (input_character)
      {
         case '1':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1209;
            break;

         case '2':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1336;
            break;

         case '3':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1477;
            break;

         case 'A':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1633;
            break;

         case '4':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1209;
            break;

         case '5':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1336;
            break;

         case '6':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1477;
            break;

         case 'B':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1633;
            break;

         case '7':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1209;
            break;

         case '8':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1336;
            break;

         case '9':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1477;
            break;

         case 'C':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1633;
            break;

         case '*':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1209;
            break;

         case '0':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1336;
            break;

         case '#':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1477;
            break;

         case 'D':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1633;
            break;

         default:
            printf ("\fInvalid Character Entered!\n\n");
            invalid_character = 1;

      }

      // If a valid character was entered, generate the DTMF tone
      if (invalid_character == 0)
      {
         tone_generated = 1;

         SFRPAGE = TMR4_PAGE;

         TMR4CN |= 0x04;               // Start Timer 4

         printf ("\fInput character is: %c\n\n", input_character);
      }

      SFRPAGE = LEGACY_PAGE;

      // The ADC ISR does two passes through the Goertzel filters:
      // One pass for the main DTMF frequencies and one pass for the
      // second harmonics of those frequencies.
      while(!done);                    // Wait the feedback stage of
                                       // Goertzel filters for the DTMF
                                       // base frequencies to complete

      done = 0;                        // Clear the done bit to wait until
                                       // the next sample set is done

      while(!done);                    // Wait the feedback stage of
                                       // Goertzel filters for 2nd
                                       // harmonics to complete

      done = 0;                        // Clear the done bit to wait until
                                       // the next sample set is done

      if (tone_detected == 1)
      {
         printf ("Button pressed is: %c\n", dtmfchar[dtmf_index]);
         tone_detected = 0;
      }
   }
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// This routine initializes the system clock to use the internal oscillator
// as its clock source.  Also enables missing clock detector reset.
//
//-----------------------------------------------------------------------------
void SYSCLK_Init (void)
{
   unsigned char i;

   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   OSCICN = 0x83;                      // Configure the internal oscillator to
                                       // 24.5 MHz

   // Step 2. Set the PLLSRC bit (PLL0CN.2) to select the desired
   // clock source for the PLL.
   PLL0CN &= ~0x04;                    // Internal oscillator

   // Step 3. Program the Flash read timing bits, FLRT (FLSCL.5-4) to the
   // appropriate value for the new clock rate (see Section 15. Flash Memory
   // on page 199).
   SFRPAGE = LEGACY_PAGE;
   FLSCL |= 0x30;                      // >= 100 MHz
   SFRPAGE = CONFIG_PAGE;

   // Step 4. Enable power to the PLL by setting PLLPWR (PLL0CN.0) to 1.
   PLL0CN |= 0x01;

   // Step 5. Program the PLL0DIV register to produce the divided reference
   // frequency to the PLL.
   PLL0DIV = 0x01;

   // Step 6. Program the PLLLP3-0 bits (PLL0FLT.3-0) to the appropriate
   // range for the divided reference frequency.
   PLL0FLT |= 0x01;

   // Step 7. Program the PLLICO1-0 bits (PLL0FLT.5-4) to the appropriate
   // range for the PLL output frequency.
   PLL0FLT &= ~0x30;

   // Step 8. Program the PLL0MUL register to the desired clock multiplication
   // factor.
   PLL0MUL = 0x04;

   // Step 9. Wait at least 5 s, to provide a fast frequency lock.
   for (i = 100; i > 0; i--);

   // Step 10. Enable the PLL by setting PLLEN (PLL0CN.1) to 1.
   PLL0CN |= 0x02;

   // Step 11. Poll PLLLCK (PLL0CN.4) until it changes from 0 to 1.
   while ((PLL0CN & 0x10) != 0x10);

   // Step 12. Switch the System Clock source to the PLL using the CLKSEL
   // register.
   CLKSEL = 0x02;

   RSTSRC = 0x06;                      // Enable Missing Clock Detector
                                       // and VDD Monitor

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// PORT_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the Crossbar and GPIO ports.
//
// Pinout:
//
// P0.0 - UART TX1 (push-pull)
// P0.1 - UART RX1
//
// DAC0 - DAC0 output
//
// AIN0.0 - ADC0 analog input
//
//-----------------------------------------------------------------------------
void PORT_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   XBR0     = 0x00;
   XBR1     = 0x00;
   XBR2     = 0x44;                    // Enable crossbar and weak pull-up
                                       // Enable UART1

   P0MDOUT |= 0x01;                    // Set TX1 pin to push-pull

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Timer0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// This routine configures Timer0.  Timer0 is used to space each tone by at
// least 20 ms.  Any new tone received within 20 ms is ignored.
//
//-----------------------------------------------------------------------------
void Timer0_Init(void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = TIMER01_PAGE;             // Switch to the necessary SFRPAGE

   CKCON &= ~0x07;                     // Timer0 clock source
   CKCON |= 0x02;                      // = SYSCLK/48

   TMOD &= ~0x0E;                      // Timer0 in 16 bit mode
   TMOD |= 0x01;

   TH0 = -(SYSCLK/48/50) >> 8;         // Load Timer0 registers
   TL0 = -(SYSCLK/48/50) & 0xFF;
   TMR0RLH = TH0;                      // Save values of Timer0 registers
   TMR0RLL = TL0;                      // for reload

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure Timer3 to auto-reload every <SYSCLK/12/SAMPLE_RATE>.
//
// Timer 3 overflow automatically triggers ADC0 conversion.
//
//-----------------------------------------------------------------------------
void Timer3_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = TMR3_PAGE;

   SFRPAGE = TMR3_PAGE;

   TMR3CN = 0x00;                      // Stop Timer3; Clear TF3;
   TMR3CF = 0x08;                      // use SYSCLK as timebase

   RCAP3 = -(SYSCLK/SAMPLE_RATE);      // Init reload values
   TMR3 = RCAP3;                       // Set to reload immediately
   EIE2 &= ~0x01;                      // Disable Timer3 interrupts
   TR3 = 1;                            // Start Timer3

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// Timer4_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:
//   1) counts - the number of timer clocks to count before a timer interrupt
//           should occur
//
// Configure Timer4 to auto-reload mode and to generate interrupts
// at intervals specified in <counts> using SYSCLK as its time base.
//
// Timer 4 overflow controls the DAC update rate.
//
//-----------------------------------------------------------------------------
void Timer4_Init (int counts)
{
   char SFRPAGE_SAVE = SFRPAGE;     // Save Current SFR page

   SFRPAGE = TMR4_PAGE;

   TMR4CN = 0x00;                   // Stop Timer4; Clear overflow flag (TF4);
                                    // Set to Auto-Reload Mode

   TMR4CF = 0x08;                   // Configure Timer4 to increment;
                                    // Timer4 counts SYSCLKs

   RCAP4 = -counts;                 // Set reload value
   TMR4 = RCAP4;                    // Initialzie Timer4 to reload value

   SFRPAGE = SFRPAGE_SAVE;          // Restore SFR page
}

//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configures ADC0 to make single-ended analog measurements on pin AIN0.0 for
// the DTMF Decoder.  Timer3 overflows are the conversion source and the data
// is left-justified.  This function also enables the ADC end-of-conversion
// interrupt and leaves the ADC disabled.
//
//-----------------------------------------------------------------------------
void ADC0_Init (void)
{
   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = ADC0_PAGE;

   ADC0CN = 0x05;                      // ADC0 disabled; normal tracking
                                       // mode; ADC0 conversions are initiated
                                       // on overflow of Timer3; ADC0 data is
                                       // left-justified
                                       // The algorithm is only using the most
                                       // significant 8 bits of data.

   REF0CN = 0x03;                      // Enable on-chip VREF and VREF output
                                       // buffer

   AMX0SL = 0x00;                      // Select AIN0.0 as ADC mux input

   ADC0CF = (SYSCLK/2500000) << 3;     // ADC conversion clock = 2.5MHz

   AD0EN = 1;                          // Enable ADC0

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// DAC0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure DAC1 to update on Timer4 overflows.  VREF is already enabled by
// the ADC initialization code.
//
//-----------------------------------------------------------------------------
void DAC0_Init(void){

   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = DAC0_PAGE;

   DAC0CN = 0x94;                      // Enable DAC0 in left-justified mode
                                       // managed by Timer4 overflows

   DAC0L = 0x00;                       // Initialize IDA0L to '0' for "8-bit"
                                       // IDAC mode

   SFRPAGE = SFRPAGE_SAVE;             // Restore SFR page
}

//-----------------------------------------------------------------------------
// UART1_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the UART1 using Timer1, for <BAUDRATE> and 8-N-1.
//
//-----------------------------------------------------------------------------
void UART1_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = UART1_PAGE;               // Switch to the necessary SFRPAGE

   SCON1 = 0x10;                       // SCON1: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits

   SFRPAGE = TIMER01_PAGE;

   TMOD   &= ~0xF0;                    // Timer 1, Mode 2, 8-bit reload
   TMOD   |=  0x20;

   if (SYSCLK/BAUDRATE/2/256 < 1)
   {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON |=  0x08;                  // T1M = 1; SCA1:0 = xx
   }
   else if (SYSCLK/BAUDRATE/2/256 < 4)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 01
      CKCON |=  0x01;
   }
   else if (SYSCLK/BAUDRATE/2/256 < 12)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 00
   }
   else if (SYSCLK/BAUDRATE/2/256 < 48)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 10
      CKCON |=  0x02;
   }
   else
   {
      while (1);                       // Error.  Unsupported baud rate
   }

   TL1 = TH1;                          // Init Timer1
   TR1 = 1;                            // START Timer1

   SFRPAGE = UART1_PAGE;

   TI1 = 1;                            // Indicate TX1 ready

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// MAC0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Initialize the MAC.
//
//-----------------------------------------------------------------------------
void MAC0_Init(void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = MAC0_PAGE;                // Switch to the necessary SFRPAGE

   MAC0CF = 0x00;                      // Integer mode;
                                       // Multiply and Accumulate

   MAC0CF |= 0x08;                     // Clear the accumulator

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// DTMF_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Initialize the DTMF system variables.
//
//-----------------------------------------------------------------------------
void DTMF_Init(void)
{
   // Initialize the variables.
   new_tone = 1;
   done = 0;
   base_freq = 1;
   gain_cnt = 0;
   gain_calc = 1;
   sample_no = 0;
   set1 = 0;
   set2 = 0;
   low_x = 255;
   high_x = 0;
   max_sample = 200;

   TR0 = 1;                            // Start Timer0
   TR2 = 1;                            // Start Timer2
}

//-----------------------------------------------------------------------------
// Interrupt_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Enables the interrupts and sets their priorities
//
//-----------------------------------------------------------------------------
void Interrupt_Init(void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   IE = 0;                             // All interrupts disabled
   EIE1 = 0;

   PT0 = 0;                            // Timer0 interrupt low priority
   ET0 = 1;                            // Timer0 interrupt enable

   EIE2   |= 0x04;                     // Enable Timer 4 interrupts
   EIP2   |= 0x04;                     // Enable Timer 4 interrupts as high
                                       // priority so it updates the DAC even
                                       // if the ADC ISR is in progress

   EIE2 |= 0x02;                       // ADC0 interrupt enable
                                       // Leave ADC0 as a low-priority
                                       // interrupt so the DAC can update the
                                       // waveform

   EA = 1;                             // Enable interrupts

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}


//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// ADC0 Conversion Complete Interrupt Service Routine (ISR)
//-----------------------------------------------------------------------------
//
// ADC0 ISR implements the signal detection, automatic gain control,
// DC bias removal and the feedback phase of the Goertzel filters
//
//-----------------------------------------------------------------------------
//void ADC0_ISR(void) interrupt 15 using 1
SI_INTERRUPT_USING(ADC0_ISR, INTERRUPT_ADC0_EOC, 1)
{
   SI_UU16_t temp_value;

   AD0INT = 0;                         // Clear the interrupt flag

   x = ADC0H;                          // Get a new sample

//---------------------------------------------------------------------------//
//                                                                           //
//                          Signal Detection                                 //
//                                                                           //
//---------------------------------------------------------------------------//

   delta_x = x - x_old;

   // The signal variation is big enough
   if( ( delta_x > XMIN ) || ( delta_x < -XMIN ) )
   {
      if(new_tone)                     // The required pause between two
      {                                // tones has passed

         ET0 = 0;                      // Disable Timer0 interrupt
         new_tone = 0;                 // Reset new_tone, only way for this
                                       // code to execute again is if there
                                       // is a 20ms gap in the signal and
                                       // Timer0 overflows.

         start_goertzel = 1;           // A new tone has been detected,
                                       // so start the DTMF detect.
      }

      TH0 = TMR0RLH;                   // Reload Timer0 registers
      TL0 = TMR0RLL;
   }

   x_old = x;                          // Save the current sample value for
                                       // the next DTMF tone detection


//---------------------------------------------------------------------------//
//                                                                           //
//                        Automatic Gain Control                             //
//                                                                           //
//---------------------------------------------------------------------------//

   // Initially, the signal input is passed through the following gain
   // calculation routine.  The gain is produced by dividing the maximum
   // possible signal magnitude by the greatest input magnitude of the
   // current tone.  This produces a gain that when multiplied with the
   // incoming signal will increase the signal amplitude to nearly full
   // scale. Also the routine finds the lowest input magnitude, which will
   // be used to compute the average value of the signal for DC bias removal.

   if(start_goertzel)                  // New tone detected
   {
      if(gain_calc)                    // Gain calculation phase
      {
         if(x > high_x)
            high_x = x;                // Find maximum value
         else
         if(x < low_x)
            low_x = x;                 // Find minimum value
         gain_cnt++;
         if(gain_cnt >= 70)
         {
            gain_cnt = 0;              // Reset gain counter
            gain_calc = 0;             // Reset gain calculation flag

            // compute gain
            gain = ( 256 / (high_x - low_x) );

            // low_x will contain the average value
            low_x += (high_x - low_x)>>1;
         }
      }
      else                             // Gain calculation completed
      {
         x = (x - low_x) * gain;       // Scale input to nearly full scale
                                       // of ADC0, and remove DC bias

//---------------------------------------------------------------------------//
//                                                                           //
//                 Feedback Phase of Goertzel Filters                        //
//                                                                           //
//---------------------------------------------------------------------------//

         SFRPAGE = MAC0_PAGE;

         // Calculate the Goertzel filters using the latest input sample
         MAC_Feedback(Q1, coef1[base_freq]); // Filter 1
         MAC_Feedback(Q2, coef2[base_freq]); // Filter 2
         MAC_Feedback(Q3, coef3[base_freq]); // Filter 3
         MAC_Feedback(Q4, coef4[base_freq]); // Filter 4
         MAC_Feedback(Q5, coef5[base_freq]); // Filter 5
         MAC_Feedback(Q6, coef6[base_freq]); // Filter 6
         MAC_Feedback(Q7, coef7[base_freq]); // Filter 7
         MAC_Feedback(Q8, coef8[base_freq]); // Filter 8

         if(sample_no == max_sample)   // Feed forward and output...
         {
            sample_no = 0;             // Reset sample counter
            done = 1;                  // Indicate that a sample set has
                                       // finished.

//---------------------------------------------------------------------------//
//                                                                           //
//                    Magnitude Phase of Goertzel Filters                    //
//                                                                           //
//---------------------------------------------------------------------------//

            // Calculate the power output (magnitude squared) for each
            // Goertzel filter
            MAC_Magnitude(Q1, coef1[base_freq], mag_squared1); // Filter 1
            MAC_Magnitude(Q2, coef2[base_freq], mag_squared2); // Filter 2
            MAC_Magnitude(Q3, coef3[base_freq], mag_squared3); // Filter 3
            MAC_Magnitude(Q4, coef4[base_freq], mag_squared4); // Filter 4
            MAC_Magnitude(Q5, coef5[base_freq], mag_squared5); // Filter 5
            MAC_Magnitude(Q6, coef6[base_freq], mag_squared6); // Filter 6
            MAC_Magnitude(Q7, coef7[base_freq], mag_squared7); // Filter 7
            MAC_Magnitude(Q8, coef8[base_freq], mag_squared8); // Filter 8

            SFRPAGE = LEGACY_PAGE;

//---------------------------------------------------------------------------//
//                                                                           //
//                          DTMF Tone Decode                                 //
//                                                                           //
//---------------------------------------------------------------------------//

            if(base_freq)              // Base frequencies calculation
            {                          // complete

               // Reset these variables for the decoding process.
               dtmf_index = 0;
               sig_present = 0;
               set1 = 0;
               set2 = 0;

               // If the energy of a given frequency is eight times greater
               // than the sum of the other frequencies from the same group,
               // then it is safe to assume that the signal contains that
               // frequency. If that frequency is present then a unique bit
               // in sig_present is set, dtmf_index is modified so that the
               // correct character in the dtmfchar array will be accessed,
               // and the set1 or set2 flag is incremented to indicate that a
               // low group or high group frequency has been detected.
               if(mag_squared1 > ((mag_squared2+mag_squared3+mag_squared4)<<3))
               {
                  sig_present |= 0x01;
                  dtmf_index += 0;
                  set1 += 1;
               }

               if(mag_squared2 > ((mag_squared1+mag_squared3+mag_squared4)<<3))
               {
                  sig_present |= 0x02;
                  dtmf_index += 4;
                  set1 += 1;
               }

               if(mag_squared3 > ((mag_squared1+mag_squared2+mag_squared4)<<3))
               {
                  sig_present |= 0x04;
                  dtmf_index += 8;
                  set1 += 1;
               }

               if(mag_squared4 > ((mag_squared1+mag_squared2+mag_squared3)<<3))
               {
                  sig_present |= 0x08;
                  dtmf_index += 12;
                  set1 += 1;
               }

               if(mag_squared5 > ((mag_squared6+mag_squared7+mag_squared8)<<3))
               {
                  sig_present |= 0x10;
                  dtmf_index += 0;
                  set2 += 1;
               }

               if(mag_squared6 > ((mag_squared5+mag_squared7+mag_squared8)<<3))
               {
                  sig_present |= 0x20;
                  dtmf_index += 1;
                  set2 += 1;
               }

               if(mag_squared7 > ((mag_squared5+mag_squared6+mag_squared8)<<3))
               {
                  sig_present |= 0x40;
                  dtmf_index += 2;
                  set2 += 1;
               }

               if(mag_squared8 > ((mag_squared5+mag_squared6+mag_squared7)<<3))
               {
                  sig_present |= 0x80;
                  dtmf_index += 3;
                  set2 += 1;
               }

//---------------------------------------------------------------------------//
//                                                                           //
//                             DTMF Output                                   //
//                                                                           //
//---------------------------------------------------------------------------//

               // If both windows have completed, there is 1 and only one of
               // the row tones and 1 and only one of the column tones,
               // and there are no harmonic frequencies present, then indicate
               // that the the proper character should be printed to the
               // terminal.
               if((sig_present != 0)&&(set1 == 1)&&(set2 == 1))
               {
                  tone_detected = 1;
               }
            }
            else                       // 2nd harmonics calculation complete
            {
               // If the sum of all 2nd harmonics energy is greater than a
               // threshold value, we assume that the signal wasn't a pure
               // DTMF.
               if(mag_squared1 + mag_squared2 + mag_squared3 + mag_squared4 +
                  mag_squared5 + mag_squared6 + mag_squared7 + mag_squared8
                  > 125)
               {
                  sig_present = 1;
               }
            }

            // Clear the filter variables for the next data set
            Q1[1].s16 = 0;
            Q1[2].s16 = 0;
            Q2[1].s16 = 0;
            Q2[2].s16 = 0;
            Q3[1].s16 = 0;
            Q3[2].s16 = 0;
            Q4[1].s16 = 0;
            Q4[2].s16 = 0;
            Q5[1].s16 = 0;
            Q5[2].s16 = 0;
            Q6[1].s16 = 0;
            Q6[2].s16 = 0;
            Q7[1].s16 = 0;
            Q7[2].s16 = 0;
            Q8[1].s16 = 0;
            Q8[2].s16 = 0;

            // Each window has 8 filters detecting 8 different frequencies.
            // If the second window has completed, then the flag to run
            // Goertzel is reset, the flag to calculate gain is reset,
            // Timer0 interrupt is reenabled, and the Timer0 high and low
            // bytes are reloaded with a 20ms delay.
            if(base_freq == 0)         // 2nd harmonics filters completed
            {
               ET0 = 1;                // Enable Timer0 interrupt
               TH0 =  TMR0RLH;         // Reload Timer0
               TL0 =  TMR0RLL;
               start_goertzel = 0;     // Clear start of decoding flag
               gain_calc = 1;          // Set gain calculation flag
               low_x = 255;            // Initialize minimum and maximum
               high_x = 0;             // signal values
               max_sample = 200;       // Number of samples for base
                                       // frequencies
            }
            else                       // Base frequencies filters completed
            {
               max_sample = 250;       // Number of samples for 2nd harmonics
            }

            // Each time a window completes, this flag is set or reset to
            // indicate which window (base or 2nd harmonics) is next.
            base_freq^=1;
         }

         // Increment the number of iterations and move on until there
         // have been max_sample iterations.
         else
         {
            sample_no++;
         }
      }
   }
}

//-----------------------------------------------------------------------------
// Timer0 Interrupt Service Routine (ISR)
//-----------------------------------------------------------------------------
//
// Timer0 is set to overflow after 20ms with no signal.  If Timer 0 overflows
// then the next tone received is a new tone.
//
//-----------------------------------------------------------------------------
//void Timer0_ISR (void) interrupt 1 using 2
SI_INTERRUPT_USING(Timer0_ISR, INTERRUPT_TIMER0, 2)
{
   // If Timer0 overflows, then there has been at least 20ms
   // with no signal, that means the next tone will be a new one.
   new_tone = 1;
   done = 1;
}

//-----------------------------------------------------------------------------
// Timer4_ISR
//-----------------------------------------------------------------------------
//
// This ISR is called on Timer3 overflows.  Timer3 is set to auto-reload mode
// and is used to schedule the DAC output sample rate in this example.
// Note that the value that is written to the DAC during this ISR call is
// actually transferred to the DAC at the next Timer3 overflow.
//
// Note: DTMF "twist" is not generated in this routine.  This feature can be
// added by scaling the high tone or low tone before outputting the value on
// the DAC.
//
//-----------------------------------------------------------------------------
//void Timer4_ISR (void) interrupt 16 using 3
SI_INTERRUPT_USING(Timer4_ISR, INTERRUPT_TIMER4, 3)
{
   static SI_UU16_t phase_acc1 = {0};       // Holds low-tone phase accumulator
   static SI_UU16_t phase_acc2 = {0};       // Holds high-tone phase accumulator

   static unsigned int tone_counter = 0; // Counts the number of interrupts
                                         // the tone is generated

   unsigned int temp1;                 // Temp values for table results
   unsigned int temp2;

   TMR4CN &= ~0x80;                    // Clear Timer4 overflow flag

   phase_acc1.u16 += phase_add1;       // Update phase acc1 (low tone)
   temp1 = SINE_TABLE[phase_acc1.u8[MSB]];

   phase_acc2.u16 += phase_add2;       // Update phase acc2 (high tone)

   // Read the table value
   temp2 = SINE_TABLE[phase_acc2.u8[MSB]];

   SFRPAGE = DAC0_PAGE;

   // Now update the DAC value.  Note: the XOR with 0x80 translates
   // the bipolar table look-up into a unipolar quantity.
   DAC0H = 0x80 ^ ((temp1 >> 1) + (temp2 >> 1));

   tone_counter++;

   // If the tone has been generated for 100 ms, stop the tone
   if (tone_counter == (SAMPLE_RATE_DAC/10))
   {
      tone_generated = 0;

      SFRPAGE = TMR4_PAGE;

      TMR4CN &= ~0x04;                 // Stop Timer 4
   }
}

//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// putchar
//-----------------------------------------------------------------------------
//
// Return Value:
//   for Keil:
//      1) char c - returns the char c that was passed as a parameter
//   for SDCC: None
// Parameters:
//   1) char c - the character to be printed
//
// Print the character <c> using the UART at <BAUDRATE>.  The definition for
// the function is different depending on the compiler used.
//
//-----------------------------------------------------------------------------
#if defined __C51__
char putchar (char c)
#elif defined SDCC
void putchar (char c)
#endif
{
   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = UART1_PAGE;

   if (c == '\n')                      // Print a carriage return
   {
      while (!TI1);
      TI1 = 0;
      SBUF1 = 0x0d;
   }
   while (!TI1);
   TI1 = 0;
   SBUF1 = c;

   SFRPAGE = SFRPAGE_SAVE;

#if defined __C51__
   return c;                           // Print the character
#endif
}

//-----------------------------------------------------------------------------
// getkey
//-----------------------------------------------------------------------------
//
// Return Value:
//   1) char c - returns the character received on the UART.
// Parameters: None
//
// Poll on the UART until a character is received and return that character.
//
//-----------------------------------------------------------------------------
char getkey (void)
{
   char c;

   char SFRPAGE_SAVE = SFRPAGE;        // Save Current SFR page

   SFRPAGE = UART1_PAGE;

   while (!RI1);
   c = SBUF1;
   RI1 = 0;

   SFRPAGE = SFRPAGE_SAVE;

   return (c);
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------
