//-----------------------------------------------------------------------------
// C8051F36x_DTMF_Generator.c
//-----------------------------------------------------------------------------
// Copyright 2006 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
// --------------------
//
// This code implements the DTMF Tone generation using the DAC.  Please see
// the seminar slides and AN218 for a detailed discussion of the tones.
//
// The DAC outputs the tones based on a SINE table in code memory.  The two
// parts to the tone are added together to create the combined tones.
//
// Pinout:
//
// P0.1 - IDA0 output (DTMF tone)
//
// P0.4 - UART0 TX pin
// P0.5 - UART0 RX pin
//
//
// How To Test:
// ------------
// 1) Compile and download the code to a 'F362 device on a TOOLSTICK360DC
//    ToolStick board (or download the C8051F36x_DTMF_Generator.hex file).
// 2) Connect P0.1 to a DTMF Decoder.
// 3) Disconnect in the IDE and connect to the TOOLSTICK360DC board in
//    ToolStick Terminal.
// 4) Navigate through the UART menu options.
//
// Target:         C8051F362 (TOOLSTICK360DC)
// Tool chain:     Keil C51 7.50 / Keil EVAL C51 / SDCC
// Command Line:   None
//
// Release 1.0
//    -Initial Revision (TP / SYRO)
//    -26 FEB 2007


//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "si_toolchain.h"             // Compiler-specific declarations
                                       // (Keil/SDCC)
#include "C8051F360_defs.h"            // SFR declarations
#include <stdio.h>

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
#define SYSCLK            24500000     // SYSCLK frequency in Hz

#define BAUDRATE          115200


// DAC constants
#define SAMPLE_RATE_DAC   100000L      // Update rate of DAC in Hz

#define PHASE_PRECISION   65536L       // Range of phase accumulator

// DTMF phase adder values based on SAMPLE_RATE_DAC and PHASE_PRECISION

#define LOW_697           697L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_770           770L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_852           852L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define LOW_941           941L * PHASE_PRECISION / SAMPLE_RATE_DAC

#define HIGH_1209         1209L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1336         1336L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1477         1477L * PHASE_PRECISION / SAMPLE_RATE_DAC
#define HIGH_1633         1633L * PHASE_PRECISION / SAMPLE_RATE_DAC

char code SINE_TABLE[256] = {
   0x00, 0x03, 0x06, 0x09, 0x0c, 0x0f, 0x12, 0x15,
   0x18, 0x1c, 0x1f, 0x22, 0x25, 0x28, 0x2b, 0x2e,
   0x30, 0x33, 0x36, 0x39, 0x3c, 0x3f, 0x41, 0x44,
   0x47, 0x49, 0x4c, 0x4e, 0x51, 0x53, 0x55, 0x58,
   0x5a, 0x5c, 0x5e, 0x60, 0x62, 0x64, 0x66, 0x68,
   0x6a, 0x6c, 0x6d, 0x6f, 0x70, 0x72, 0x73, 0x75,
   0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7c,
   0x7d, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f,
   0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7e, 0x7e,
   0x7d, 0x7c, 0x7c, 0x7b, 0x7a, 0x79, 0x78, 0x77,
   0x76, 0x75, 0x73, 0x72, 0x70, 0x6f, 0x6d, 0x6c,
   0x6a, 0x68, 0x66, 0x64, 0x62, 0x60, 0x5e, 0x5c,
   0x5a, 0x58, 0x55, 0x53, 0x51, 0x4e, 0x4c, 0x49,
   0x47, 0x44, 0x41, 0x3f, 0x3c, 0x39, 0x36, 0x33,
   0x30, 0x2e, 0x2b, 0x28, 0x25, 0x22, 0x1f, 0x1c,
   0x18, 0x15, 0x12, 0x0f, 0x0c, 0x09, 0x06, 0x03,
   0x00, 0xfd, 0xfa, 0xf7, 0xf4, 0xf1, 0xee, 0xeb,
   0xe8, 0xe4, 0xe1, 0xde, 0xdb, 0xd8, 0xd5, 0xd2,
   0xd0, 0xcd, 0xca, 0xc7, 0xc4, 0xc1, 0xbf, 0xbc,
   0xb9, 0xb7, 0xb4, 0xb2, 0xaf, 0xad, 0xab, 0xa8,
   0xa6, 0xa4, 0xa2, 0xa0, 0x9e, 0x9c, 0x9a, 0x98,
   0x96, 0x94, 0x93, 0x91, 0x90, 0x8e, 0x8d, 0x8b,
   0x8a, 0x89, 0x88, 0x87, 0x86, 0x85, 0x84, 0x84,
   0x83, 0x82, 0x82, 0x81, 0x81, 0x81, 0x81, 0x81,
   0x80, 0x81, 0x81, 0x81, 0x81, 0x81, 0x82, 0x82,
   0x83, 0x84, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
   0x8a, 0x8b, 0x8d, 0x8e, 0x90, 0x91, 0x93, 0x94,
   0x96, 0x98, 0x9a, 0x9c, 0x9e, 0xa0, 0xa2, 0xa4,
   0xa6, 0xa8, 0xab, 0xad, 0xaf, 0xb2, 0xb4, 0xb7,
   0xb9, 0xbc, 0xbf, 0xc1, 0xc4, 0xc7, 0xca, 0xcd,
   0xd0, 0xd2, 0xd5, 0xd8, 0xdb, 0xde, 0xe1, 0xe4,
   0xe8, 0xeb, 0xee, 0xf1, 0xf4, 0xf7, 0xfa, 0xfd
};

//-----------------------------------------------------------------------------
// Global VARIABLES
//-----------------------------------------------------------------------------

bit tone_generated = 0;                // Flag used in main() and the DAC ISR
                                       // for determining when a tone is switched
                                       // on and off

unsigned int phase_add1;               // Holds low-tone phase adder
unsigned int phase_add2;               // Holds low-tone phase adder

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void SYSCLK_Init(void);                // Configure system clock
void Port_Init(void);                  // Configure Crossbar and GPIO
void IDA0_Init(void);                  // Configure IDAC0
void Timer3_Init(int counts);          // Configure Timer3
void UART0_Init(void);                 // Configure UART0

void Interrupt_Init(void);             // Initialize interrupts

// Define the UART printing functions for Keil/SDCC
#if defined __C51__
char putchar (char c);                 // Define putchar for Keil
#elif defined SDCC
void putchar (char c);                 // Define putchar for SDCC
#endif

char getkey(void);

// Interrupt Service Routines
SI_INTERRUPT_PROTO_USING(Timer3_ISR, INTERRUPT_TIMER3, 3);

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main(void)
{
   bit invalid_character = 0;
   unsigned char input_character;

   EA = 0;                             // All interrupts disabled
   PCA0MD &= ~0x40;                    // Clear watchdog timer

   SYSCLK_Init();                      // Configure system clock
   Port_Init();                        // Configure Crossbar and GPIO
   Timer3_Init(SYSCLK/SAMPLE_RATE_DAC);// Using Timer3 as update scheduler
   IDA0_Init();
   UART0_Init();

   Interrupt_Init();                   // Initialize interrupts

   printf ("\f\n\n");                  // Clear the screen

   while(1)
   {
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| 1 | | 2 | | 3 | | A |\n");
      printf ("|___| |___| |___| |___|\n\n");
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| 4 | | 5 | | 6 | | B |\n");
      printf ("|___| |___| |___| |___|\n\n");
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| 7 | | 8 | | 9 | | C |\n");
      printf ("|___| |___| |___| |___|\n\n");
      printf (" ___   ___   ___   ___ \n");
      printf ("|   | |   | |   | |   |\n");
      printf ("| * | | 0 | | # | | D |\n");
      printf ("|___| |___| |___| |___|\n\n");

      while (tone_generated == 1);     // Wait until the current tone is
                                       // done being generated before
                                       // accepting a new value

      printf ("Enter the touch-tone number: ");
      input_character = getkey();

      invalid_character = 0;

      // Determine the two DTMF tones for the desired number
      switch (input_character)
      {
         case '1':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1209;
            break;

         case '2':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1336;
            break;

         case '3':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1477;
            break;

         case 'A':
            phase_add1 = LOW_697;
            phase_add2 = HIGH_1633;
            break;

         case '4':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1209;
            break;

         case '5':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1336;
            break;

         case '6':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1477;
            break;

         case 'B':
            phase_add1 = LOW_770;
            phase_add2 = HIGH_1633;
            break;

         case '7':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1209;
            break;

         case '8':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1336;
            break;

         case '9':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1477;
            break;

         case 'C':
            phase_add1 = LOW_852;
            phase_add2 = HIGH_1633;
            break;

         case '*':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1209;
            break;

         case '0':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1336;
            break;

         case '#':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1477;
            break;

         case 'D':
            phase_add1 = LOW_941;
            phase_add2 = HIGH_1633;
            break;

         default:
            printf ("\fInvalid Character Entered!\n\n");
            invalid_character = 1;

      }

      // If a valid character was entered, generate the DTMF tone
      if (invalid_character == 0)
      {
         tone_generated = 1;
         TMR3CN |= 0x04;               // Start Timer 3
         printf ("\fInput character is: %c\n\n", input_character);
      }
   }
}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// This routine initializes the system clock to use the internal oscillator
// as its clock source. Also enables missing clock detector and VDD monitor.
//
//-----------------------------------------------------------------------------
void SYSCLK_Init(void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   OSCICN = 0x83;                      // Configure internal oscillator for
                                       // its highest frequency
   RSTSRC = 0x06;                      // Enable missing clock detector and
                                       // VDD monitor

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}


//-----------------------------------------------------------------------------
// Port_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the Crossbar and GPIO ports.
//
// P0.1 - IDA0 output (DTMF tone)
//
// P0.4 - UART0 TX pin
// P0.5 - UART0 RX pin
//
//-----------------------------------------------------------------------------
void Port_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   P0MDOUT |= 0x10;                    // Set TX pin to push-pull

   P0MDIN &= ~0x02;                    // Set P0.1 as an analog pin (IDAC)

   P0SKIP = 0x01;                      // Skip P0.1 for the IDAC output

   XBR0 = 0x01;                        // Enable UART0
   XBR1 = 0x40;                        // Enable crossbar and weak pull-ups

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Timer3_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   :
//   1)  int counts - calculated Timer overflow rate
//                    range is positive range of integer: 0 to 32767
//
// Configure Timer3 to auto-reload at interval specified by <counts>
// using SYSCLK as its time base.  Interrupts are enabled.
//
//-----------------------------------------------------------------------------
void Timer3_Init(int counts)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   TMR3CN  = 0x00;                     // Resets Timer 3, sets to 16 bit mode
   CKCON  |= 0x40;                     // Use system clock

   TMR3RL  = -counts;                  // Initial reload value
   TMR3    = 0xffff;                   // Sets timer to reload automatically

   EIE1   |= 0x80;                     // Enable Timer 3 interrupts

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// IDA0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Enable IDAC0 and VREF.  The IDAC uses "8-bit" mode by setting IDA0L to
// always be 0.
//
//-----------------------------------------------------------------------------
void IDA0_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   REF0CN = 0x03;                      // Enable on-chip VREF and buffer

   IDA0CN = 0xB2;                      // Enable IDAC0 for 2.0 mA full-scale
                                       // output; updated on Timer3 overflows

   IDA0L = 0x00;                       // Initialize IDA0L to '0' for "8-bit"
                                       // IDAC mode

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Configure the UART0 using Timer1, for <BAUDRATE> and 8-N-1.
//
//-----------------------------------------------------------------------------
void UART0_Init (void)
{
   unsigned char SFRPAGE_save = SFRPAGE; // Save the current SFRPAGE

   SFRPAGE = CONFIG_PAGE;              // Switch to the necessary SFRPAGE

   SCON0 = 0x10;                       // SCON0: 8-bit variable bit rate
                                       //        level of STOP bit is ignored
                                       //        RX enabled
                                       //        ninth bits are zeros
                                       //        clear RI0 and TI0 bits
   if (SYSCLK/BAUDRATE/2/256 < 1)
   {
      TH1 = -(SYSCLK/BAUDRATE/2);
      CKCON |=  0x08;                  // T1M = 1; SCA1:0 = xx
   }
   else if (SYSCLK/BAUDRATE/2/256 < 4)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/4);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 01
      CKCON |=  0x01;
   }
   else if (SYSCLK/BAUDRATE/2/256 < 12)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/12);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 00
   }
   else if (SYSCLK/BAUDRATE/2/256 < 48)
   {
      TH1 = -(SYSCLK/BAUDRATE/2/48);
      CKCON &= ~0x0B;                  // T1M = 0; SCA1:0 = 10
      CKCON |=  0x02;
   }
   else
   {
      while (1);                       // Error.  Unsupported baud rate
   }

   TL1 = TH1;                          // Init Timer1
   TMOD &= ~0xF0;                      // TMOD: Timer 1 in 8-bit autoreload
   TMOD |=  0x20;
   TR1 = 1;                            // START Timer1
   TI0 = 1;                            // Indicate TX0 ready

   SFRPAGE = SFRPAGE_save;             // Restore the SFRPAGE
}

//-----------------------------------------------------------------------------
// Interrupt_Init
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
// Enables the interrupts and sets their priorities.
//
//-----------------------------------------------------------------------------
void Interrupt_Init(void)
{
   IE = 0;                             // All interrupts disabled

   EIE1 |= 0x80;                       // Timer3 interrupt enable

   EA = 1;                             // Enable interrupts
}

//-----------------------------------------------------------------------------
// Interrupt Service Routines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Timer3_ISR
//-----------------------------------------------------------------------------
//
// This ISR is called on Timer3 overflows.  Timer3 is set to auto-reload mode
// and is used to schedule the DAC output sample rate in this example.
// Note that the value that is written to the IDAC during this ISR call is
// actually transferred to the IDAC at the next Timer3 overflow.
//
// Note: DTMF "twist" is not generated in this routine.  This feature can be
// added by scaling the high tone or low tone before outputting the value on
// the DAC.
//
//-----------------------------------------------------------------------------
//void Timer3_ISR (void) interrupt 14 using 3
SI_INTERRUPT_USING(Timer3_ISR, INTERRUPT_TIMER3, 3)
{
   static SI_UU16_t phase_acc1 = {0};       // Holds low-tone phase accumulator
   static SI_UU16_t phase_acc2 = {0};       // Holds high-tone phase accumulator

   static unsigned int tone_counter = 0; // Counts the number of interrupts
                                         // the tone is generated

   unsigned int temp1;                 // Temp values for table results
   unsigned int temp2;

   TMR3CN &= ~0x80;                    // Clear Timer3 overflow flag

   phase_acc1.u16 += phase_add1;       // Update phase acc1 (low tone)
   temp1 = SINE_TABLE[phase_acc1.u8[MSB]];

   phase_acc2.u16 += phase_add2;       // Update phase acc2 (high tone)

   // Read the table value
   temp2 = SINE_TABLE[phase_acc2.u8[MSB]];

   // Now update the DAC value.  Note: the XOR with 0x80 translates
   // the bipolar table look-up into a unipolar quantity.
   IDA0H = 0x80 ^ ((temp1 >> 1) + (temp2 >> 1));

   tone_counter++;

   // If the tone has been generated for 100 ms, stop the tone
   if (tone_counter == (SAMPLE_RATE_DAC/10))
   {
      tone_generated = 0;
      TMR3CN &= ~0x04;                 // Stop Timer 3
   }
}

//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// putchar
//-----------------------------------------------------------------------------
//
// Return Value:
//   for Keil:
//      1) char c - returns the char c that was passed as a parameter
//   for SDCC: None
// Parameters:
//   1) char c - the character to be printed
//
// Print the character <c> using the UART at <BAUDRATE>.  The definition for
// the function is different depending on the compiler used.
//
//-----------------------------------------------------------------------------
#if defined __C51__
char putchar (char c)
#elif defined SDCC
void putchar (char c)
#endif
{
   if (c == '\n')                      // Print a carriage return
   {
      while (!TI0);
      TI0 = 0;
      SBUF0 = 0x0d;
   }
   while (!TI0);
   TI0 = 0;
   SBUF0 = c;
#if defined __C51__
   return c;                           // Print the character
#endif
}

//-----------------------------------------------------------------------------
// getkey
//-----------------------------------------------------------------------------
//
// Return Value:
//   1) char c - returns the character received on the UART.
// Parameters: None
//
// Poll on the UART until a character is received and return that character.
//
//-----------------------------------------------------------------------------
char getkey ()
{
   char c;

   while (!RI0);
   c = SBUF0;
   RI0 = 0;
   return (c);
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------