//-----------------------------------------------------------------------------
// F50x_CAN.c - CAN_BL - Target_BL_FW
//-----------------------------------------------------------------------------
// Copyright (C) 2009 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// This program performs as the Target Bootloader Firmware for the CAN bootloader.
// PC (Data Source) <--> UART <--> MCU (Master) <--> CAN <--> MCU (Target)
//
//
//
// Target:         C8051F500 (Side A of a C8051F500-TB)
// Tool chain:     Keil C51 8.0 / Keil EVAL C51
// Command Line:   None
//
//
// Release 1.0 / 18NOV2009 (PKC)
//    -Initial Revision
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "si_toolchain.h"
#include "C8051F500_defs.h"

#ifdef MASTER_MCU_BL
   #undef MASTER_MCU_BL
#endif

#include "F50x_Master_Interface.h"
#include "F50x_Main.h"
#include "F50x_CAN.h"
#include "F50x_Flash.h"
#include "F50x_CRC.h"

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------
bit CAN_Error = 0;                     // 0 = No Errors during transmission
                                       // 1 = Some error(s) occurred

SI_SEGMENT_VARIABLE(CAN_Rx_Buf[MESSAGE_SIZE], uint8_t, SI_SEG_XDATA);
SI_SEGMENT_VARIABLE(CAN_Tx_Buf[MESSAGE_SIZE], uint8_t, SI_SEG_XDATA);

uint8_t CAN_Rx_Complete_Flag = 0;          // CAN Rx Complete Flag

//-----------------------------------------------------------------------------
// Function Prototypes (Local)
//-----------------------------------------------------------------------------
void CAN0_Send_Message (uint8_t response_to_send);

//-----------------------------------------------------------------------------
// Main Routine
//-----------------------------------------------------------------------------

//=============================================================================
// Function Definitions
//=============================================================================

//-----------------------------------------------------------------------------
// CAN0_Init
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// This function initializes the CAN peripheral and three message objects
//
// CAN Bit Clock : 1 Mbps
// Auto Retransmit : Automatic Retransmission is enabled
// MsgVal        : Set to Valid based on the #define MESSAGE_OBJECTS
// Filtering     : Enabled for all valid message objects
// Message Identifier : 11-bit standard; Each message object is only used by
//                      one message ID
// Direction     : Two buffers for transmit and one for receive are configured
// End of Buffer : All message objects are treated as separate buffers
//
// The following interrupts are enabled and are handled by CAN0_ISR
//
// Error Interrupts
// Status Change Interrupt
// Receive Interrupt
//
//-----------------------------------------------------------------------------

void CAN0_Init (void)
{
   uint8_t SFRPAGE_save = SFRPAGE;
   SFRPAGE  = CAN0_PAGE;               // All CAN register are on page 0x0C

   CAN0CN |= 0x01;                     // Start Intialization mode

   //---------Initialize general CAN peripheral settings

   CAN0CN |= 0x4E;                     // Enable Error and Module
                                       // Enable access to bit timing register

   // See the CAN Bit Timing Spreadsheet for how to calculate this value
   CAN0BT = 0x1402;                    // Based on 24 Mhz CAN clock, set the
                                       // CAN bit rate to 1 Mbps

   //---------Initialize settings for Transmit Message Object

   // Command Mask Register
   CAN0IF1CM = 0x00F0;                 // Write Operation
                                       // Transfer ID Mask, MDir, MXtd
                                       // Transfer ID, Dir, Xtd, MsgVal
                                       // Transfer Control Bits
                                       // Don't set TxRqst or transfer data

   // Mask Registers
   CAN0IF1M1 = 0x0000;                 // Mask Bits 15-0 not used for filtering
   CAN0IF1M2 = 0x5FFC;                 // Ignore Extended Identifier for
                                       // filtering
                                       // Used Direction bit for filtering
                                       // Use ID bits 28-18 for filtering

   // Message Control Registers
   CAN0IF1MC = 0x0080 | MESSAGE_SIZE;  // Disable Transmit Interrupt
                                       // Message Object is a Single Message
                                       // Message Size set by #define
   // Arbitration Registers
   CAN0IF1A1 = 0x0000;                 // 11-bit ID, so lower 16-bits not used

   // Arbitration Registers
   CAN0IF1A2 = 0xA000 | (MSG_ID_TX_BL_RSP << 2);   // Set MsgVal to valid
                                                   // Set Direction to write
                                                   // Set 11-bit Identifier

   CAN0IF1CR = MO_TX_BL_RSP;           // Start command request

   while (CAN0IF1CRH & 0x80) {}        // Poll on Busy bit

   //---------Initialize settings for Receive Message Object 1

   // Can use the same settings for Receive object, so no need to reinitalize the
   // first four CAN registers again

   // Command Mask Register
   //   CAN0IF1CM = 0x00F0;

   // Mask Registers
   //  CAN0IF1M1 = 0x0000;
   //  CAN0IF1M2 = 0x5FFC;

   // Arbitration Registers
   //  CAN0IF1A1 = 0x0000;


   // Message Control Registers
   CAN0IF1MC = 0x1480 | MESSAGE_SIZE;  // Enable Receive Interrupt
                                       // Message Object is a Single Message
                                       // Message Size set by #define
   // Arbitration Registers
   CAN0IF1A2 = 0x8000 | (MSG_ID_RX_BL_CMD << 2);   // Set MsgVal to valid
                                                   // Set Object Direction to read
                                                   // Set 11-bit Identifier

   CAN0IF1CR = MO_RX_BL_CMD;              // Start command request

   while (CAN0IF1CRH & 0x80) {}           // Poll on Busy bit

   //---------Initialize settings for Receive Message Object 2

   // Can use the same settings for Receive object, so no need reinitalize the
   // first four CAN registers again

   // Command Mask Register
   //   CAN0IF1CM = 0x00F0;

   // Mask Registers
   //  CAN0IF1M1 = 0x0000;
   //  CAN0IF1M2 = 0x5FFC;

   // Arbitration Registers
   //  CAN0IF1A1 = 0x0000;


   // Message Control Registers
   CAN0IF1MC = 0x1480 | MESSAGE_SIZE;  // Enable Receive Interrupt
                                       // Message Object is a Single Message
                                       // Message Size set by #define
   // Arbitration Registers
   CAN0IF1A2 = 0x8000 | (MSG_ID_RX_BL_WRITE8 << 2);   // Set MsgVal to valid
                                                      // Set Object Direction to read
                                                      // Set 11-bit Identifier

   CAN0IF1CR = MO_RX_BL_WRITE8;           // Start command request

   while (CAN0IF1CRH & 0x80) {}           // Poll on Busy bit

   //--------- CAN Initalization is complete

   CAN0CN &= ~0x41;                    // Return to Normal Mode and disable
                                       // access to bit timing register

   //EIE2 |= 0x02;                       // Enable CAN interupts
   //Do not enable CAN0 interrupts -- Polled-mode

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// CAN0_SendMessage
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Send the data in the buffer using the message object. This function
// assumes 8 bytes of data for all messages.
//
//-----------------------------------------------------------------------------

void CAN0_Send_Message (uint8_t response_to_send)
{
   // This function assumes that the message object is fully initialized
   // in CAN0_Init and so all it has to do is fill the data registers and
   // initiate transmission

   uint8_t SFRPAGE_save = SFRPAGE;
   SFRPAGE  = CAN0_PAGE;               // All CAN register are on page 0x0C

   CAN0IF1DA1H = response_to_send;     // Initialize data registers
   CAN0IF1DA1L = CAN_Tx_Buf[1];
   CAN0IF1DA2H = CAN_Tx_Buf[2];
   CAN0IF1DA2L = CAN_Tx_Buf[3];
   CAN0IF1DB1H = CAN_Tx_Buf[4];
   CAN0IF1DB1L = CAN_Tx_Buf[5];
   CAN0IF1DB2H = CAN_Tx_Buf[6];
   CAN0IF1DB2L = CAN_Tx_Buf[7];

   CAN0IF1CM = 0x0087;                 // Set Direction to Write
                                       // Write TxRqst, all 8 data bytes

   CAN0IF1CR = MO_TX_BL_RSP;           // Start command request

   while (CAN0IF1CRH & 0x80) {}        // Poll on Busy bit

   SFRPAGE = SFRPAGE_save;             // Restore SFRPAGE
}

//-----------------------------------------------------------------------------
// TGT_Enter_BL_Mode
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Enter_BL_Mode (void)
{
   // Command Format:
   // [0] Command

   // Response Format:
   // [0] Return code (ACK/ERROR etc)

   CAN0_Send_Message (TGT_RSP_BL_MODE);
}

//-----------------------------------------------------------------------------
// TGT_Get_Info
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Get_Info (void)
{
   static uint8_t packet_num = 0;
   uint8_t count;

   // Command Format:
   // [0] Command

   // Response:
   // [0] Return code (ACK/ERROR etc)
   // [1] Packet Number
   // [2] InfoBlock byte (0 + (packetnum*6))
   // [3] InfoBlock byte (1 + (packetnum*6))
   // [4] ...
   // [5] 
   // [6] 
   // [7]

   //if (packet_num >= ((TGT_BL_INFOBLOCK_LENGTH+5)/6) + 1)
   if (packet_num > 3)
   {
      packet_num = 0;
   }

   CAN_Tx_Buf[1] = packet_num;

   for (count = 0; count < 6; count++)
   {
      CAN_Tx_Buf[2+count] = TGT_BL_InfoBlock[TGT_BL_INFOBLOCK_LENGTH-1-count-(packet_num*6)];
   }

   if (packet_num == 0)
   {
      CAN_Tx_Buf[2] = CAN_Tx_Buf[2] + 2; // Add 2 to length (for App FW ver low/high)
   }

   if (packet_num == 2)
   {
      CAN_Tx_Buf[6] = *(uint8_t code*)(APP_LAST_PAGE_END_ADDR-8);
      CAN_Tx_Buf[7] = *(uint8_t code*)(APP_LAST_PAGE_END_ADDR-9);
   }

   packet_num++;

   CAN0_Send_Message (TGT_RSP_OK);
}

//-----------------------------------------------------------------------------
// TGT_Set_Flash_Keys
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Set_Flash_Keys (void)
{
   // Command Format:
   // [0] Command
   // [1] Flash Key byte 0
   // [2] Flash Key byte 1

   Flash_Key_Code0 = CAN_Rx_Buf[1];
   Flash_Key_Code1 = CAN_Rx_Buf[2];

   // Response:
   // [0] Return code (ACK/ERROR etc)

   CAN0_Send_Message (TGT_RSP_OK);
}

//-----------------------------------------------------------------------------
// TGT_Set_Addr
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Set_Addr (void)
{
   SI_UU16_t addr;
   uint8_t response;

   // Command Format:
   // [0] Command
   // [1] Flash code bank number
   // [2] Flash address byte 0
   // [3] Flash address byte 1
   // [4] Flash address byte 2 <-- don't care for MCUs 64k and smaller

   addr.u8[LSB] = CAN_Rx_Buf[2];
   addr.u8[MSB] = CAN_Rx_Buf[3];

   if (  (addr.u16 >= APP_START_ADDR) &&
         (addr.u16 < (APP_LAST_PAGE_START_ADDR+PAGE_SIZE)) &&
         ((addr.u16 % PAGE_SIZE) == 0)
      )
   {
      // Page addr should be within app start and end addresses
      // Page addr should be the start address of a page
      Page_Addr = addr.u16;
      Page_Index = 0;
      response = TGT_RSP_OK;
   }
   else
   {
      response = TGT_RSP_ERROR;
   }

   // Response:
   // [0] Return code (ACK/ERROR etc)

   CAN0_Send_Message (response);
}

//-----------------------------------------------------------------------------
// TGT_Erase_Page
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Erase_Page (void)
{
   // Command Format:
   // [0] Command

   FLASH_PageErase (Page_Addr);

   // Response:
   // [0] Return code (ACK/ERROR etc)

   CAN0_Send_Message (TGT_RSP_OK);
}

//-----------------------------------------------------------------------------
// TGT_Write_Flash
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Write_Flash (void)
{
   uint8_t i;

   // numbytes needs to be 8 because this message is preconfigured for 8 bytes

   // Command Format: [No command byte because this uses a unique message ID]
   // [0] Flash byte 0
   // [1] Flash byte 1
   // [2] Flash byte 2
   // [3] Flash byte 3
   // [4] Flash byte 4
   // [5] Flash byte 5
   // [6] Flash byte 6
   // [7] Flash byte 7

   for (i = 0; i < MESSAGE_SIZE; i++)
   {
      FLASH_ByteWrite ((Page_Addr+Page_Index), CAN_Rx_Buf[i]);
      Page_Index++;
   }

   // Response:
   // [0] Return code (ACK/ERROR etc)

   CAN0_Send_Message (TGT_RSP_OK);
}

//-----------------------------------------------------------------------------
// TGT_Get_Page_CRC
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Get_Page_CRC (void)
{
   SI_UU16_t crc;

   // Command Format:
   // [0] Command

   // Response:
   // [0] Return code (ACK/ERROR etc)
   // [1] CRC byte 0
   // [2] CRC byte 1

   crc.u16 = Get_Page_CRC ();

   CAN_Tx_Buf[1] = crc.u8[LSB];
   CAN_Tx_Buf[2] = crc.u8[MSB];

   CAN0_Send_Message (TGT_RSP_OK);
}

//-----------------------------------------------------------------------------
// TGT_Write_Signature
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_Write_Signature (void)
{
   // Command Format:
   // [0] Command
   // [1] Signature byte 0
   // [2] Signature byte 1
   // [3] Signature byte 2
   // [4] Signature byte 3

   FLASH_ByteWrite (SIG_BYTE0_ADDR, CAN_Rx_Buf[1]);
   FLASH_ByteWrite (SIG_BYTE1_ADDR, CAN_Rx_Buf[2]);
   FLASH_ByteWrite (SIG_BYTE2_ADDR, CAN_Rx_Buf[3]);
   FLASH_ByteWrite (SIG_BYTE3_ADDR, CAN_Rx_Buf[4]);

   // Response:
   // [0] Return code (ACK/ERROR etc)

   CAN0_Send_Message (TGT_RSP_OK);
}

//-----------------------------------------------------------------------------
// TGT_SW_Reset
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void TGT_SW_Reset (void)
{
   uint8_t SFRPAGE_save = SFRPAGE;
   SFRPAGE = ACTIVE_PAGE;

   // Command Format:
   // [0] Command

   // Response:
   // [0] Return code (ACK/ERROR etc)

   CAN0_Send_Message (TGT_RSP_OK);

   // TODO: Delay might be needed here (to allow sending the response)

   RSTSRC = 0x12;    // Initiate software reset with vdd monitor enabled

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// TGT_Send_Error
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
//
//
//-----------------------------------------------------------------------------
void CAN_Send_Error (void)
{
   // Command Format:
   // This is not a command

   // Response:
   // [0] Return code (ACK/ERROR etc)
   CAN0_Send_Message (TGT_RSP_ERROR);
}

//-----------------------------------------------------------------------------
// Wait_For_CAN_Rx_Complete (CAN0_ISR in Polled Mode)
//-----------------------------------------------------------------------------
//
// This function is a substitute for the CAN ISR.
// It polls the CAN Status register to check for an interrupt.
// That register has bits that are set upon any CAN errors or upon a
// complete reception.
//
// If an error occurs, a global flag is updated
//
//-----------------------------------------------------------------------------

void Wait_For_CAN_Rx_Complete (void)
{
   uint8_t status;
   uint8_t Interrupt_ID;

   uint8_t SFRPAGE_save = SFRPAGE;
   SFRPAGE  = CAN0_PAGE;               // All CAN registers are on page 0x0C

   while (CAN_Rx_Complete_Flag == 0)
   {
      // Note: The "CAN0INT" bit mentioned in the datasheet is not preset in CAN0CN.
      // So, the following poll won't work:
      // while ((CAN0CN & 0x80) == 0);    // Wait till the CAN0INT interrupt bit is set

     // That is why we have to use this alternative method of polling the status register:

      do
      {
         status = CAN0STAT;            // Read status, which clears the Status
                                       // Interrupt pending in CAN0IID
                                       // (CAN0IID = 0x8000 for Status Interrupt)
      }
      while ((status & 0xDF) == 0);    // Wait until a status interrupt bit is set
                                       // Monitor BOff, EWarn, RxOk, TxOk and LEC

      Interrupt_ID = CAN0IIDL;         // Read which message object caused
                                       // the interrupt (high byte CAN0IIDH is
                                       // not used)

      CAN0IF1CM = 0x007F;              // Read all of message object to IF1
                                       // Clear IntPnd and newData


      CAN0IF1CR = Interrupt_ID;        // Start command request to actually
                                       // clear the interrupt

      while (CAN0IF1CRH & 0x80) {}     // Poll on Busy bit

      // If receive completed successfully
      if ((status & RxOk) && ((Interrupt_ID == MO_RX_BL_CMD) || (Interrupt_ID == MO_RX_BL_WRITE8)))
      {
          // Read all 8 data bytes to CAN_Rx_Buf, even though they might not be valid

         CAN_Rx_Buf[0] = CAN0IF1DA1H;
         CAN_Rx_Buf[1] = CAN0IF1DA1L;
         CAN_Rx_Buf[2] = CAN0IF1DA2H;
         CAN_Rx_Buf[3] = CAN0IF1DA2L;
         CAN_Rx_Buf[4] = CAN0IF1DB1H;
         CAN_Rx_Buf[5] = CAN0IF1DB1L;
         CAN_Rx_Buf[6] = CAN0IF1DB2H;
         CAN_Rx_Buf[7] = CAN0IF1DB2L;

         CAN_Rx_Complete_Flag = 1;       // Indicate Rx Complete & exit this function

         if (Interrupt_ID == MO_RX_BL_WRITE8)
         {
            Command_Code = TGT_CMD_WRITE_FLASH;
         }
         else if (Interrupt_ID == MO_RX_BL_CMD)
         {
            Command_Code = CAN_Rx_Buf[0];

            // This global variable is used only by the main loop for initial check of
            // Enter_BL_Mode command
            Rx_CAN_Device_Addr = CAN_Rx_Buf[1]; 
         }
         else
         {
            // TODO: Error?
         }
      }

      // If an error occured, simply update the global variable and continue
      if (status & LEC)
      {
          // The LEC bits identify the type of error, but those are grouped here
         if ((status & LEC) != 0x07)
         {
             CAN_Error = 1;
         }
      }

      if (status & BOff)
      {
         CAN_Error = 1;
      }

      if (status & EWarn)
      {
         CAN_Error = 1;
      }
   }

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------
