//-----------------------------------------------------------------------------
// F50x_Main.c - CAN_BL - Target_BL_FW
//-----------------------------------------------------------------------------
// Copyright (C) 2009 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
// This program performs as the Target Bootloader Firmware for the CAN bootloader.
// PC (Data Source) <--> UART <--> MCU (Master) <--> CAN <--> MCU (Target)
//
// How To Test:
//
// 1) Download code to the 'F500 target board
// 2) Ensure that the P1.3 pins are shorted together on the J19 header
// 3) Run the program.
//
//
// Target:         C8051F500 (Side A of a C8051F500-TB)
// Tool chain:     Keil C51 8.0 / Keil EVAL C51
// Command Line:   None
//
//
// Release 1.0 / 18NOV2009 (PKC)
//    -Initial Revision
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "si_toolchain.h"
#include "C8051F500_defs.h"

#ifdef MASTER_MCU_BL
   #undef MASTER_MCU_BL
#endif

#include "F50x_Master_Interface.h"

#include "F50x_Main.h"
#include "F50x_Init.h"
#include "F50x_CAN.h"

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------
SI_SBIT(P1_4_SWITCH, SFR_P1, 4);
SI_SBIT(P1_3_LED, SFR_P1, 3);

//-----------------------------------------------------------------------------
// Global Variables
//-----------------------------------------------------------------------------
uint8_t Device_Mode = MODE_APP;

uint8_t App_FW_Valid = INVALID;

uint16_t Page_Addr;
uint16_t Page_Index;
uint8_t Flash_Key_Code0;
uint8_t Flash_Key_Code1;
uint8_t Command_Code;
uint8_t LED_State;

uint8_t Rx_CAN_Device_Addr;

//-----------------------------------------------------------------------------
// Function Prototypes (Local)
//-----------------------------------------------------------------------------
void Set_LED_State (uint8_t LED_State);

//-----------------------------------------------------------------------------
// Main Routine
//-----------------------------------------------------------------------------
void main (void)
{
   bit BL_Mode_Addressed = 0;
   uint8_t code* codeptr;

   SFRPAGE = ACTIVE_PAGE;              // Configure for PCA0MD

   PCA0MD &= ~0x40;                    // Disable the watchdog timer
   EA = 0;                             // Disable global interrupts

   codeptr = (uint8_t code*)(SIG_BYTE0_ADDR);
   App_FW_Valid = INVALID;

   // The Signature (in Flash) should be valid to allow application FW execution.
   // This is written at the end of the bootloading process by the bootloader.
   if (*codeptr-- == SIG_BYTE0)
   {
     if (*codeptr-- == SIG_BYTE1)
     {
         if (*codeptr-- == SIG_BYTE2)
         {
            if (*codeptr == SIG_BYTE3)
            {
               // All signature bytes match. 
               App_FW_Valid = VALID;
            }
         }
      }
   }

   // Check the Bootloader-Override Pin on the device if this feature is enabled
   #if defined(BOOTLOADER_PIN_OVERRIDE)
      if (P1_4_SWITCH == 0)            // Check P1.4
      {
         // If pin is 0, enter bootload mode and ignore other conditions.
         Device_Mode = MODE_BL_PIN_REQ;
      }
   #endif

   if (Device_Mode == MODE_APP)  // If the mode has not already been decided by the pin state
   {
      if (App_FW_Valid == INVALID)
      {
            // If the signature is invalid, force bootloader mode.
            Device_Mode = MODE_BL_NO_APP;
      }
      else
      {
         // Signature is valid; the next step is to check the Reset Source:

         if (((RSTSRC & PORSF) == 0) && ((RSTSRC & FERROR) != 0))
         {
            // If the Power-on/Brown-out flag is NOT set,
            // AND if the Flash Error Reset flag is set,
            // this indicates a bootloader request from application firmware
            Device_Mode = MODE_BL_APP_REQ;
         }
         else
         {
            // In all other cases, jump to the application FW
            Device_Mode = MODE_APP;       // Setting this variable here is not necessary
            START_APPLICATION ();         // Jump to Application FW
         }
      }

   }

   Device_Init ();

   Set_LED_State (LED_OFF);

   // Initialize Flash Write/Erase Key Codes to zero
   // The actual codes will be set by the Master before calling any write/erase commands
   Flash_Key_Code0 = 0x00;
   Flash_Key_Code1 = 0x00;

   Page_Addr = APP_LAST_PAGE_START_ADDR;

   while(1)                            // Main Command Processor
   {
      Set_LED_State (LED_OFF);

      Wait_For_CAN_Rx_Complete ();     // Wait until a message is received
      CAN_Rx_Complete_Flag = 0;

      if (Command_Code == TGT_CMD_ENTER_BL_MODE && Rx_CAN_Device_Addr == CAN_DEVICE_ADDR)
      {         
         BL_Mode_Addressed = 1;
      }

      if (BL_Mode_Addressed == 1)
      {
         Set_LED_State (LED_ON);

         switch (Command_Code)
         {
            case TGT_CMD_ENTER_BL_MODE:
               TGT_Enter_BL_Mode ();
               break;
            case TGT_CMD_GET_INFO:
               TGT_Get_Info ();
               break;
            case TGT_CMD_SET_FLASH_KEYS:
               TGT_Set_Flash_Keys ();
               break;
            case TGT_CMD_SET_ADDR:
               TGT_Set_Addr ();
               break;
            case TGT_CMD_ERASE_PAGE:
               TGT_Erase_Page ();
               break;
            case TGT_CMD_WRITE_FLASH:
               TGT_Write_Flash ();
               break;
            case TGT_CMD_GET_PAGE_CRC:
               TGT_Get_Page_CRC ();
               break;
            case TGT_CMD_WRITE_SIGNATURE:
               TGT_Write_Signature ();
               break;
            case TGT_CMD_SW_RESET:
               TGT_SW_Reset ();
               break;
            default:
               CAN_Send_Error ();
               break;
         }
      }
   }
}

//=============================================================================
// Function Definitions
//=============================================================================

//-----------------------------------------------------------------------------
// Set_LED_State
//-----------------------------------------------------------------------------
//
// Return Value:  None
// Parameters:    None
//
//
//
//-----------------------------------------------------------------------------
void Set_LED_State (uint8_t state)
{
   uint8_t SFRPAGE_save = SFRPAGE;
   SFRPAGE = CONFIG_PAGE;

   LED_State = state;

   switch (LED_State)
   {
      case LED_OFF:
         P1_3_LED = 0;
         break;
      case LED_ON:
         P1_3_LED = 1;
         break;
      case LED_BLINK_SLOW:
         // Not yet supported
         break;
      case LED_BLINK_FAST:
         // Not yet supported
         break;
      default:
         break;
   }

   SFRPAGE = SFRPAGE_save;
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------
