/**************************************************************************//**
 * @file
 * @brief Energy Modes Exercise Support Function
 * @author Energy Micro AS
* @version 1.10
 ******************************************************************************
 * @section License
 * <b>(C) Copyright 2012 Energy Micro AS, http://www.energymicro.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 * 4. The source and compiled code may only be used on Energy Micro "EFM32"
 *    microcontrollers and "EFR4" radios.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Energy Micro AS has no
 * obligation to support this Software. Energy Micro AS is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Energy Micro AS will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 *****************************************************************************/
#include "em_cmu.h"
#include "em_gpio.h"
#include "em_usart.h"
#include "em_rtc.h"

/* Header file for support functions */
#include "4_exercise.h"

#if defined(_EFM32_GIANT_FAMILY)

#include "leds.h"
/* Defines for Push Button 0 */
#define PB0_PORT                    gpioPortB
#define PB0_PIN                     9

/* Defines for the LED */
#define LED_PORT                    gpioPortE
#define LED_PIN                     2


#else
#include "led.h"
/* Defines for Push Button 0 */
#define PB0_PORT                    gpioPortD
#define PB0_PIN                     8

/* Defines for the LED */
#define LED_PORT                    gpioPortD
#define LED_PIN                     7


#endif

/**************************************************************************//**
 * @brief RTC Handler - simply clear the interrupt
 *****************************************************************************/
void RTC_IRQHandler(void)
{
  /* Clear interrupt source */
  RTC_IntClear(RTC_IFC_COMP0);
}

/**************************************************************************//**
 * @brief Configure RTC
 *****************************************************************************/
void rtc_setup(void)
{
  /* Starting LFXO and waiting until it is stable */
  CMU_OscillatorEnable(cmuOsc_LFXO, true, true);

  /* Routing the LFXO clock to the RTC */
  CMU_ClockSelectSet(cmuClock_LFA, cmuSelect_LFXO);
  CMU_ClockEnable(cmuClock_RTC, true);

  /* Enabling clock to the interface of the low energy modules */
  CMU_ClockEnable(cmuClock_CORELE, true);

  const RTC_Init_TypeDef rtcInit =
  {
    .enable   = true,
    .debugRun = false,
    .comp0Top = true,
  };

  RTC_Init(&rtcInit);

  /* Produce an interrupt every second */
  RTC_CompareSet(0, 32768);

  /* Enable interrupt for compare register 0 */
  RTC_IntEnable(RTC_IFC_COMP0);

  /* Enabling interrupt from RTC */
  NVIC_EnableIRQ(RTC_IRQn);
}

/**************************************************************************//**
 * @brief USART1 Handler - simply clear the interrupt
 *****************************************************************************/
void USART1_TX_IRQHandler(void)
{
  /* Clear interrupt source */
  USART_IntClear(USART1, USART_IF_TXC);
}

/**************************************************************************//**
 * @brief Configure USART
 *****************************************************************************/
void usart_setup(void)
{
  USART_TypeDef           *usart = USART1;
  USART_InitAsync_TypeDef init   = USART_INITASYNC_DEFAULT;

  CMU_ClockEnable(cmuClock_HFPER, true);
  CMU_ClockEnable(cmuClock_USART1, true);
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Output on pin D0 */
  GPIO_PinModeSet(gpioPortD, 0, gpioModePushPull, 1);

  /* Enable pins at default location */
  usart->ROUTE = USART_ROUTE_TXPEN | USART_ROUTE_LOCATION_LOC1;

  /* Configure USART for basic async operation */
  init.enable   = usartDisable;
  init.baudrate = 9600;
  USART_InitAsync(usart, &init);

  /* Enable interrupts */
  NVIC_EnableIRQ(USART1_TX_IRQn);
  USART_IntEnable(USART1, USART_IF_TXC);

  /* Finally enable it */
  USART_Enable(usart, usartEnable);
}

/**************************************************************************//**
 * @brief GPIO Even Handler - simply clear the interrupt
 *****************************************************************************/
void GPIO_EVEN_IRQHandler(void)
{
  /* Clear interrupt source */
  GPIO_IntClear(1 << PB0_PIN);
}

/**************************************************************************//**
 * @brief GPIO Even Handler - simply clear the interrupt
 *****************************************************************************/
void GPIO_ODD_IRQHandler(void)
{
  /* Clear interrupt source */
  GPIO_IntClear(1 << PB0_PIN);
}

/**************************************************************************//**
 * @brief Configure GPIO
 *****************************************************************************/
void gpio_setup(void)
{
  CMU_ClockEnable(cmuClock_GPIO, true);

  /* Configure PD7 (LED) as a push pull, so that we can set its value */
  GPIO_PinModeSet(LED_PORT, LED_PIN, gpioModePushPull, 0);

  /* Configure interrupt for Push Button 0 (D8) */
  GPIO_PinModeSet(PB0_PORT, PB0_PIN, gpioModeInput, 1);
#if defined(_EFM32_GIANT_FAMILY)
  NVIC_EnableIRQ(GPIO_ODD_IRQn);
#else
  NVIC_EnableIRQ(GPIO_EVEN_IRQn);
#endif
  GPIO_IntConfig(PB0_PORT, PB0_PIN, false, true, true);
}