/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Log.c
*
* Description:    Write and erase the log
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Log.h"


/* Flash Read Buffer Control Register */
SFR ( FRBCN, 0xB5 );  

/* Sample */
typedef struct
{
   U32 Timestamp;
   S16 Temperature;
   S16 Humidity;
} SAMPLE;

/* Where the next sample will be written */
VARIABLE_SEGMENT_POINTER( pLog, U8, SEG_XDATA );

/* The number of samples in the log */
U16 SampleCount;


/*****************************************************************************/
/* Log_Init                                                                  */
/*****************************************************************************/

U32 Log_Init( void )
{
   VARIABLE_SEGMENT_POINTER( pSample, SAMPLE, SEG_CODE );
   U32 LastTimestamp=0;
   U16 x;
   
   SFRPAGE = CONFIG_PAGE;

   /* Enable block writes to flash */
   FRBCN = 0x01;

   SFRPAGE = LEGACY_PAGE;

   /* Make sure that the Vdd Monitor is enabled */
	VDM0CN = 0x80;

   /* Wait 100us for the Vdd Monitor to stabilize */
   for ( x=0; x<350; x++ );

   /* Make sure that Vdd Monitor is a reset source */
   RSTSRC = 0x02;
   
   /* Start at the beginning of bank 1 */
   PSBANK  = BANK_1;
   pSample = BANK_ADDRESS; 

   /* Count the number of samples in bank 1 */
   for ( SampleCount=0; SampleCount<SAMPLES_PER_BANK; pSample++,SampleCount++ )
      if ( pSample->Timestamp == 0xFFFFFFFF ) break;

   /* If bank 1 is full */   
   if ( SampleCount == SAMPLES_PER_BANK )
   {
      /* Note the last timestamp in bank 1 in case bank 2 is empty */
      LastTimestamp = (pSample-1)->Timestamp;
   
      /* Start at the beginning of bank 2 */
      PSBANK  = BANK_2;
      pSample = BANK_ADDRESS; 

      /* Count the number of samples in bank 2 */
      for ( ; SampleCount<MAX_SAMPLE_COUNT; pSample++,SampleCount++ )
         if ( pSample->Timestamp == 0xFFFFFFFF ) break;
   } 

   /* This is where the next sample will be written */
   pLog = (U8*)pSample;

   /* If not at the beginning of a bank */
   if ( pSample != BANK_ADDRESS )
      /* Get the timestamp of the last sample */
      LastTimestamp = (pSample-1)->Timestamp;

   return LastTimestamp;   
}


/*****************************************************************************/
/* Log_WriteData                                                             */
/*****************************************************************************/

void Log_WriteData( U8 *pData, U8 DataLength )
{
   U8 x;
   
   for ( x=0; x<DataLength; x++ )
   {
      /* Unlock the flash */
      FLKEY = 0xA5;
      FLKEY = 0xF1;

      /* Write the data to flash */
      *pLog++ = *pData++;
   }
}


/*****************************************************************************/
/* Log_WriteSample                                                           */
/*****************************************************************************/

void Log_WriteSample( U32 Timestamp, S16 Temperature, S16 Humidity )
{
   if ( SampleCount < MAX_SAMPLE_COUNT )
   {
      EA = 0;

      /* Enable writing to flash */
      PSCTL = 0x01;
      
      /* Write the sample to flash */
      Log_WriteData( (U8*)&Timestamp,   4 ); 
      Log_WriteData( (U8*)&Temperature, 2 );
      Log_WriteData( (U8*)&Humidity,    2 );

      /* Disable writing to flash */
      PSCTL = 0x00;

      EA = 1;
      
      SampleCount++;
      
      if ( SampleCount == SAMPLES_PER_BANK )
      {
         PSBANK = BANK_2;
         pLog   = BANK_ADDRESS; 
      }
   }
}


/*****************************************************************************/
/* Log_ErasePage                                                             */
/*****************************************************************************/

void Log_ErasePage( VARIABLE_SEGMENT_POINTER(pPage,U8,SEG_XDATA) )
{
   /* Unlock the flash */
   FLKEY = 0xA5;
   FLKEY = 0xF1;

   /* Write to the page to erase it */
   *pPage = 0;
}


/*****************************************************************************/
/* Log_Erase                                                                 */
/*****************************************************************************/

void Log_Erase( void )
{
   VARIABLE_SEGMENT_POINTER( pPage, U8, SEG_XDATA );

   EA = 0;

   /* Enable erasing flash */
   PSCTL = 0x03;

   /* If currently in bank 2 */
   if ( PSBANK == BANK_2 )
   {
      PSBANK = BANK_1;

      /* Erase all the pages in bank 1 */
      for ( pPage=BANK_ADDRESS; pPage>0; pPage+=PAGE_SIZE )
         Log_ErasePage( pPage );

      PSBANK = BANK_2;
   }

   /* If the log is full */
   if ( pLog == 0 )
   {
      /* Erase all the pages in bank 2 */
      for ( pPage=BANK_ADDRESS; pPage>0; pPage+=PAGE_SIZE )
         Log_ErasePage( pPage );
   }
   else /* The log is not full */
   {
      /* Erase the written pages in the current bank */
      for ( pPage=BANK_ADDRESS; pPage<pLog; pPage+=PAGE_SIZE )
         Log_ErasePage( pPage );
   }   

   /* Disable erasing to flash */
   PSCTL = 0x00;
   
   EA = 1;

   /* The log is now empty */
   SampleCount = 0;

   /* Start at the beginning of bank 1 */
   PSBANK = BANK_1;
   pLog   = BANK_ADDRESS; 
}   

