/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      lcdPutNumber.c
*
* Description:    Display a number, temperature or humidity on the LCD
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "lcdPutChar.h"
#include "lcdPutString.h"
#include "lcdPutNumber.h"
#include "Main.h"


#define SEC_PER_MINUTE   60
#define SEC_PER_HOUR     3600
#define SEC_PER_DAY      86400


/*****************************************************************************/
/* lcdPutNumber                                                              */
/*                                                                           */
/* lcdPutNumber() displays an unsigned 16-bit number on the LCD in decimal.   */
/* If the number has fewer digits than the minimum width then the LCD is     */
/* padded on the left with spaces. The number is right aligned within the    */
/* the minimum width.  If the number has more digits than the minimum width, */
/* then the all the digits are displayed and the minimum width is exceeded.  */
/*                                                                           */
/*****************************************************************************/

void lcdPutNumber( U16 Number, U8 MinWidth, U8 HideLeadingZero )
{
   char Digit;
   U16  Place;
   U8   Width;
   U8   Leading = TRUE;

   /* For each decimal place (greater than one) */
   for ( Place=10000, Width=5; Place>1; Width--,Place/=10 )
   {
      /* Get the digit for the place */
      Digit = '0' + (Number/Place);
      
      /* If it's a leading zero */
      if ( Digit=='0' && Leading )
      {
         /* If within the minimum field width */
         if ( Width <= MinWidth )
         {
            if ( HideLeadingZero )
               lcdPutChar(' ');  /* Display space */
            else
               lcdPutChar('0');  /* Display zero */
         }  
         /* Display nothing if greater than the minimum field width */ 
      }
      else /* It's not a leading zero */
      {
         /* Display the digit */
         lcdPutChar( Digit );
         Leading = FALSE;
      }

      /* Remove the place from the number */
      Number %= Place;
   }
   
   /* One's place */
   lcdPutChar( '0' + Number );
}


/*****************************************************************************/
/* lcdPutTemperature                                                         */
/*****************************************************************************/

void lcdPutTemperature( S16 Temperature, char Scale )
{
   U8 WholeDegree;
   U8 DeciDegree;
   U8 Negative = FALSE;

   /* If the scale is Fahrenheit then do the conversion */
   if ( Scale == 'F' )
      Temperature = ((Temperature*9)/5) + 320;
   
   /* Normalize the temperature to a positive number */
   if ( Temperature < 0 )
   {
      Temperature = -Temperature;
      Negative = TRUE;
   }

   /* Extract the whole and fractional components */
   WholeDegree = Temperature / 10;
   DeciDegree  = Temperature % 10;

   /* If the temperature is negative */
   if ( Negative )
   {
      lcdPutChar('-');
      if ( WholeDegree >= 10 )
      {
         if ( DeciDegree >= 5 )
            WholeDegree++;
         lcdPutNumber( WholeDegree, 2, TRUE );
      }
      else /* Negative but greater than -10 */
      {
         lcdPutNumber( WholeDegree, 1, TRUE );
         lcdPutChar('.');
         lcdPutNumber( DeciDegree, 1, TRUE );
      }
   }
   /* If the temperature is a triple digit whole number */
   else if ( WholeDegree >= 100 )
   {
      if ( DeciDegree >= 5 )
         WholeDegree++;
      lcdPutNumber( WholeDegree, 3, TRUE );
   }
   else /* Positive but less than a hundred */
   {
      lcdPutNumber( WholeDegree, 2, TRUE );
      lcdPutChar('.');
      lcdPutNumber( DeciDegree, 1, TRUE );
   }

   lcdPutChar('\'');
   lcdPutChar( Scale ); 
}


/*****************************************************************************/
/* lcdPutHumidity                                                            */
/*****************************************************************************/

void lcdPutHumidity( S16 Humidity )
{
   U8 WholePercent;
   U8 DeciPercent;

   /* Extract the whole and fractional components */
   WholePercent = Humidity / 10;
   DeciPercent  = Humidity % 10;

   /* Round up to the nearest percent */   
   if ( DeciPercent >= 5 )
      WholePercent++;

   lcdPutNumber( WholePercent, 3, TRUE );
   lcdPutChar('%');
}


/*****************************************************************************/
/* SecondsInYear                                                             */
/*****************************************************************************/

U32 SecondsInYear( U8 Year )
{
   if ( Year % 4 )
      return 365*SEC_PER_DAY;
   else   
      return 366*SEC_PER_DAY;
}


/*****************************************************************************/
/* SecondsInMonth                                                            */
/*****************************************************************************/

U32 SecondsInMonth( U8 Month, U8 Year )
{
   if ( Month==2 )
   {
      if ( Year % 4 )
         return 28*SEC_PER_DAY;
      else
         return 29*SEC_PER_DAY;   
   }   
   else if ( Month==4 || Month==6 || Month==9 || Month==11  )
   {
      return 30*SEC_PER_DAY;
   }
   else
   {
      return 31*SEC_PER_DAY;
   }
}


/*****************************************************************************/
/* lcdPutDate                                                                */
/*****************************************************************************/

void lcdPutDate( U32 SysTime )
{
   U8 Year;
   U8 Month;
   U8 Day;

   /* Find the year */
   for ( Year=0; SysTime>=SecondsInYear(Year); Year++ )
      SysTime -= SecondsInYear( Year );

   /* Find the month */
   for ( Month=1; SysTime>=SecondsInMonth(Month,Year); Month++ )
      SysTime -= SecondsInMonth( Month, Year );

   /* Find the day of the month */
   Day = SysTime / SEC_PER_DAY + 1;  

   lcdPutNumber( Month, 1, TRUE );
   lcdPutChar('/');
   lcdPutNumber( Day, 1, TRUE );
   lcdPutChar('/');
   lcdPutNumber( Year, 2, FALSE );
}


/*****************************************************************************/
/* lcdPutDays                                                                */
/*****************************************************************************/

void lcdPutDays( U32 SysTime )
{
   U16 Days;

   /* Find the number of days */
   Days = SysTime / SEC_PER_DAY;

   lcdPutNumber( Days, 3, TRUE );
   lcdPutString(" DAY");
   if ( Days != 1 )
      lcdPutChar('S');
}


/*****************************************************************************/
/* lcdPutTime                                                                */
/*****************************************************************************/

void lcdPutTime( U32 SysTime, U8 Format )
{
   U8 Hour;
   U8 Minute;
   U8 Second;
   char Suffix;

   /* Find the hour, minute and second */
   SysTime %= SEC_PER_DAY;
   Hour   = SysTime / SEC_PER_HOUR;    SysTime %= SEC_PER_HOUR;
   Minute = SysTime / SEC_PER_MINUTE;  SysTime %= SEC_PER_MINUTE;
   Second = SysTime;
   
   /* In necessary, convert to 12-hour format */
   if ( Format == 12 )       
   {
      if ( Hour == 0 )       /* Midnight */
      {
         Hour   = 12;
         Suffix = 'A';
      }
      else if ( Hour < 12 )  /* Morning */
      {
         Suffix = 'A';
      }
      else if ( Hour == 12 ) /* Noon */
      {
         Suffix = 'P';
      }
      else /* Hour > 12 */   /* Afternoon */
      {
         Hour  -= 12;
         Suffix = 'P';
      }
   }
   else Suffix = ' ';

   lcdPutNumber( Hour, 2, TRUE );
   lcdPutChar('\'');
   lcdPutChar('.');
   lcdPutNumber( Minute, 2, FALSE );
   lcdPutChar('\'');
   lcdPutChar('.');
   lcdPutNumber( Second, 2, FALSE );
   lcdPutChar(' ');
   lcdPutChar( Suffix );
}


/*****************************************************************************/
/* lcdPutSampleCount                                                         */
/*****************************************************************************/

void lcdPutSampleCount( U16 Count, U8 Logging )
{
   lcdPutNumber( Count, 4, TRUE );
   if ( Logging ) lcdPutChar('.');
   lcdPutString(" SMP");
}

