/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myDisplay.c
*
* Description:    Display information on the LCD
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include "myLCD0.h"
#include "myDisplay.h"


#define SEC_PER_MINUTE   60
#define SEC_PER_HOUR     3600
#define SEC_PER_DAY      86400


/*****************************************************************************/
/* Display_Temperature                                                       */
/*****************************************************************************/

void Display_Temperature( int Temperature, char Scale )
{
   int  WholeDegree;
   int  DeciDegree;
   bool Negative = false;

   /* If the scale is Fahrenheit then do the conversion */
   if ( Scale == 'F' )
      Temperature = ((Temperature*9)/5) + 320;
   
   /* Normalize the temperature to a positive number */
   if ( Temperature < 0 )
   {
      Temperature = -Temperature;
      Negative = true;
   }

   /* Extract the whole and fractional components */
   WholeDegree = Temperature / 10;
   DeciDegree  = Temperature % 10;

   /* If the temperature is negative */
   if ( Negative )
   {
      myLCD0_put_char('-');
      if ( WholeDegree >= 10 )
      {
         if ( DeciDegree >= 5 )
            WholeDegree++;
         myLCD0_put_number( WholeDegree, 2, true );
      }
      else /* Negative but greater than -10 */
      {
         myLCD0_put_number( WholeDegree, 1, true );
         myLCD0_put_char('.');
         myLCD0_put_number( DeciDegree, 1, true );
      }
   }
   /* If the temperature is a triple digit whole number */
   else if ( WholeDegree >= 100 )
   {
      if ( DeciDegree >= 5 )
         WholeDegree++;
      myLCD0_put_number( WholeDegree, 3, true );
   }
   else /* Positive but less than a hundred */
   {
      myLCD0_put_number( WholeDegree, 2, true );
      myLCD0_put_char('.');
      myLCD0_put_number( DeciDegree, 1, true );
   }

   myLCD0_put_char('\'');
   myLCD0_put_char( Scale ); 
}


/*****************************************************************************/
/* Display_Humidity                                                          */
/*****************************************************************************/

void Display_Humidity( int Humidity )
{
   int WholePercent;
   int DeciPercent;

   /* Extract the whole and fractional components */
   WholePercent = Humidity / 10;
   DeciPercent  = Humidity % 10;

   /* Round up to the nearest percent */   
   if ( DeciPercent >= 5 )
      WholePercent++;

   myLCD0_put_number( WholePercent, 3, true );
   myLCD0_put_char('%');
}


/*****************************************************************************/
/* SecondsInYear                                                             */
/*****************************************************************************/

int SecondsInYear( int Year )
{
   if ( Year % 4 )
      return 365*SEC_PER_DAY;
   else   
      return 366*SEC_PER_DAY;
}


/*****************************************************************************/
/* SecondsInMonth                                                            */
/*****************************************************************************/

int SecondsInMonth( int Month, int Year )
{
   if ( Month==2 )
   {
      if ( Year % 4 )
         return 28*SEC_PER_DAY;
      else
         return 29*SEC_PER_DAY;   
   }   
   else if ( Month==4 || Month==6 || Month==9 || Month==11  )
   {
      return 30*SEC_PER_DAY;
   }
   else
   {
      return 31*SEC_PER_DAY;
   }
}


/*****************************************************************************/
/* Display_Date                                                              */
/*****************************************************************************/

void Display_Date( uint32_t SysTime )
{
   int Year;
   int Month;
   int Day;

   /* Find the year */
   for ( Year=0; SysTime>=SecondsInYear(Year); Year++ )
      SysTime -= SecondsInYear( Year );

   /* Find the month */
   for ( Month=1; SysTime>=SecondsInMonth(Month,Year); Month++ )
      SysTime -= SecondsInMonth( Month, Year );

   /* Find the day of the month */
   Day = SysTime / SEC_PER_DAY + 1;  

   myLCD0_put_number( Month, 1, true );
   myLCD0_put_char('/');
   myLCD0_put_number( Day, 1, true );
   myLCD0_put_char('/');
   myLCD0_put_number( Year, 2, false );
}


/*****************************************************************************/
/* Display_Days                                                              */
/*****************************************************************************/

void Display_Days( uint32_t SysTime )
{
   int Days;

   /* Find the number of days */
   Days = SysTime / SEC_PER_DAY;

   myLCD0_put_number( Days, 3, true );
   myLCD0_put_string(" DAY");
   if ( Days != 1 )
      myLCD0_put_char('S');
}


/*****************************************************************************/
/* Display_Time                                                              */
/*****************************************************************************/

void Display_Time( uint32_t SysTime, int Format )
{
   int Hour;
   int Minute;
   int Second;
   char Suffix;

   /* Find the hour, minute and second */
   SysTime %= SEC_PER_DAY;
   Hour   = SysTime / SEC_PER_HOUR;    SysTime %= SEC_PER_HOUR;
   Minute = SysTime / SEC_PER_MINUTE;  SysTime %= SEC_PER_MINUTE;
   Second = SysTime;
   
   /* In necessary, convert to 12-hour format */
   if ( Format == 12 )       
   {
      if ( Hour == 0 )       /* Midnight */
      {
         Hour   = 12;
         Suffix = 'A';
      }
      else if ( Hour < 12 )  /* Morning */
      {
         Suffix = 'A';
      }
      else if ( Hour == 12 ) /* Noon */
      {
         Suffix = 'P';
      }
      else /* Hour > 12 */   /* Afternoon */
      {
         Hour  -= 12;
         Suffix = 'P';
      }
   }
   else Suffix = ' ';

   myLCD0_put_number( Hour, 2, true );
   myLCD0_put_char('\'');
   myLCD0_put_char('.');
   myLCD0_put_number( Minute, 2, false );
   myLCD0_put_char('\'');
   myLCD0_put_char('.');
   myLCD0_put_number( Second, 2, false );
   myLCD0_put_char(' ');
   myLCD0_put_char( Suffix );
}


/*****************************************************************************/
/* Display_Count                                                             */
/*****************************************************************************/

void Display_Count( int Count, bool Logging )
{
   myLCD0_put_number( Count, 4, true );
   if ( Logging ) myLCD0_put_char('.');
   myLCD0_put_string(" SMP");
}


