/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myI2C0.c
*
* Description:    Use the I2C protocol to read and write registers on a device
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_I2C_A_Type.h>
#include "myCpu.h"
#include "myI2C0.h"


/* Maximum time to wait for a transfer to complete */
#define MAX_XFER_TIME   300  /* ms */

/* Maximum number of data bytes that can be written by I2C0_WriteData() */
#define MAX_WRITE_LENGTH  8

/* Number of I2C buses */
#define I2C_BUS_COUNT  2

/* If the transfer is in the writing phase then Writing is true, else false */
bool Writing;

/* The number of bytes in the data buffer that still need to be transmitted */ 
int WriteRemaining;

/* The number of bytes that still need to be written to the data buffer */
int ReadRemaining;

/* Pointer within the data buffer of the next data to be transmitted */
uint32_t *pWriteData;

/* Pointer within the buffer where the next received data will be written */
uint32_t *pReadData;

/* The 7-bit I2C address of the slave device */
uint8_t SlaveAddress;

/* The current I2C bus */
int CurrentBus = I2C0_BUS_1;

/* The status of the I2C transfer */
int TransferStatus;


/*****************************************************************************/
/* I2C0_ReadByte                                                             */
/*****************************************************************************/

int I2C0_ReadByte( int Bus, uint8_t Slave, uint8_t Register, uint8_t *Byte )
{
   return I2C0_ReadData( Bus, Slave, Register, Byte, 1 );
}


/*****************************************************************************/
/* I2C0_WriteByte                                                            */
/*****************************************************************************/

int I2C0_WriteByte( int Bus, uint8_t Slave, uint8_t Register, uint8_t Byte )
{
   return I2C0_WriteData( Bus, Slave, Register, &Byte, 1 );
}


/*****************************************************************************/
/* I2C0_ReadData                                                             */
/*****************************************************************************/

int I2C0_ReadData( int Bus, uint8_t Slave, uint8_t Register, uint8_t *Data, int DataLength )
{
   /* Write the register number */
   *Data = Register;

   return I2C0_Transfer( Bus, Slave, Data, 1, DataLength );
}


/*****************************************************************************/
/* I2C0_ReadDataWrite2                                                       */
/*****************************************************************************/

int I2C0_ReadDataWrite2( int Bus, uint8_t Slave, uint8_t Cmd0, uint8_t Cmd1, uint8_t *Data, int DataLength )
{
   /* Write the command bytes */
   *(Data+0) = Cmd0;
   *(Data+1) = Cmd1;

   return I2C0_Transfer( Bus, Slave, Data, 2, DataLength );
}


/*****************************************************************************/
/* I2C0_WriteData                                                            */
/*****************************************************************************/

int I2C0_WriteData( int Bus, uint8_t Slave, uint8_t Register, uint8_t *Data, int DataLength )
{
   uint8_t Buffer[MAX_WRITE_LENGTH+1];
   int x;

   /* Make sure that the DataLength is good */
   if ( DataLength > MAX_WRITE_LENGTH )
      return I2C0_STATUS_BAD_LENGTH;

   /* Combine the register number and the data into a single buffer */
   Buffer[0] = Register;
   for ( x=1; x<=DataLength; x++ )
      Buffer[x] = *Data++;

   return I2C0_Transfer( Bus, Slave, Buffer, DataLength+1, 0 );
}


/*****************************************************************************/
/* I2C0_Transfer                                                             */
/*                                                                           */
/*   Parameters:                                                             */
/*      Bus         : Selects the set of pins to use for I2C communication.  */
/*      Slave       : The 7-bit I2C address of the slave device.             */
/*      Data        : Pointer to a read/write buffer.  This buffer is used   */
/*                    to pass in write data to this routine.  This routine   */
/*                    passes back read data in this buffer.                  */
/*      WriteLength : The number of bytes to write (0 to INT_MAX).           */
/*      ReadLength  : The number of bytes to read  (0 to INT_MAX).           */
/*                                                                           */
/*****************************************************************************/

int I2C0_Transfer( int Bus, uint8_t Slave, uint8_t *Data, int WriteLength, int ReadLength )
{
   uint32_t StartTime;

   /* Make sure that the bus number is good */
   if ( Bus < I2C_BUS_COUNT )
   {
      /* Move the I2C0 module to the specified bus */
      if ( Bus != CurrentBus )
      {
         if ( Bus == I2C0_BUS_1 )           /* Bus 1: PB0.9 and PB0.10 */
            SI32_PBSTD_A_write_pbskipen( SI32_PBSTD_0, 0x01FF );
         else                               /* Bus 2: PB0.7 and PB0.8 */
            SI32_PBSTD_A_write_pbskipen( SI32_PBSTD_0, 0x007F );
      
         CurrentBus = Bus;
      }

      /* Set the transfer parameters */
      SlaveAddress   = Slave;
      pWriteData     = (uint32_t*)Data;
      pReadData      = (uint32_t*)Data;
      WriteRemaining = WriteLength;
      ReadRemaining  = ReadLength;
      
      /* The transfer has not started yet */
      TransferStatus = I2C0_STATUS_NONE;

      /* Start the transfer */
      SI32_I2C_A_set_start( SI32_I2C_0 );

      /* Note when the transfer started */   
      StartTime = get_msTicks();
      
      /* Block until the transfer is finished or timed out */
      while ( TransferStatus == I2C0_STATUS_NONE )
      {
         if ( (get_msTicks()-StartTime) > MAX_XFER_TIME )
         {
            /* Reset the I2C module */
            SI32_I2C_A_reset_module( SI32_I2C_0 );
            
            /* Transfer timed out */
            TransferStatus = I2C0_STATUS_TIMEOUT;
            break;
         }
      }
   }
   else TransferStatus = I2C0_STATUS_BAD_BUS;
      
   return TransferStatus;   
}


/*****************************************************************************/
/* I2C0_start_handler                                                        */
/*****************************************************************************/

void I2C0_start_handler( void )
{
   /* Clear the start flag */
   SI32_I2C_A_clear_start( SI32_I2C_0 );

   /* Set the byte count to one (for the address byte) */
   SI32_I2C_A_set_byte_count( SI32_I2C_0, 1 );

   /* Transmit the address byte with the read/write bit */            
   if ( WriteRemaining )
   {
      SI32_I2C_A_write_data( SI32_I2C_0, SlaveAddress<<1 );
      Writing = true;
   }   
   else /* Reading */
   {   
      SI32_I2C_A_write_data( SI32_I2C_0, (SlaveAddress<<1)+1 );
      Writing = false;
   }   

   /* Arm the transmitter */
   SI32_I2C_A_arm_tx( SI32_I2C_0 );
   
   /* Clear the start interrupt */
   SI32_I2C_A_clear_start_interrupt( SI32_I2C_0 );
}


/*****************************************************************************/
/* I2C0_tx_complete_handler                                                  */
/*****************************************************************************/

void I2C0_tx_complete_handler( void )
{
   /* If the address byte or data byte was acknowledged */
   if ( SI32_I2C_A_is_ack_received(SI32_I2C_0) )
   {
      /* If writing data */
      if ( Writing )
      {
         /* If there is more data to write */
         if ( WriteRemaining )
         {
            /* If there are less than 4 bytes remaining to write */
            if ( WriteRemaining < 4 )
            {
               /* Set the byte count to the number of bytes remaining */
               SI32_I2C_A_set_byte_count( SI32_I2C_0, WriteRemaining );
               WriteRemaining = 0;
            }
            else /* There are 4 or more bytes remaining to write */
            {
               /* Set the byte count to zero (meaning 4 bytes) */
               SI32_I2C_A_set_byte_count( SI32_I2C_0, 0 );
               WriteRemaining -= 4;
            }

            /* Write the data */
            SI32_I2C_A_write_data( SI32_I2C_0, *pWriteData++ );

            /* Arm the transmitter */
            SI32_I2C_A_arm_tx( SI32_I2C_0 );
         }   
         else /* All the data has been written */
         {
            /* If there is data to read */   
            if ( ReadRemaining )
            {
               /* Generate a restart */
               SI32_I2C_A_set_start( SI32_I2C_0 );
            }
            else /* There is no data to read */
            {
               /* Generate a stop */
               SI32_I2C_A_set_stop( SI32_I2C_0 );
               
               /* Write transfer is successful */
               TransferStatus = I2C0_STATUS_SUCCESS;
            }
         }
      }
      else /* The address byte of a read was acknowledged */
      {
         /* If there is data to read */
         if ( ReadRemaining )
         {
            /* If there is less than 4 bytes remaining to read */
            if ( ReadRemaining < 4 )
            {
               /* Set the byte count to the number of bytes remaining */
               SI32_I2C_A_set_byte_count( SI32_I2C_0, ReadRemaining );
               ReadRemaining = 0;
            }
            else /* There are 4 or more bytes remaining to read */
            {
               /* Set the byte count to zero (meaning 4 bytes) */
               SI32_I2C_A_set_byte_count( SI32_I2C_0, 0 );
               ReadRemaining -= 4;
            }

            /* Arm the receiver */
            SI32_I2C_A_arm_rx( SI32_I2C_0 );
         }
         else /* Reading zero bytes */
         {
            /* Generate a stop */
            SI32_I2C_A_set_stop( SI32_I2C_0 );
            
            /* Read transfer of zero bytes is successful */
            TransferStatus = I2C0_STATUS_SUCCESS;
         }
      }
   }
   else /* The address byte or data byte was not acknowledged */
   {
      /* If writing */
      if ( Writing )
      {
         /* Abort the transfer */
         SI32_I2C_A_set_stop( SI32_I2C_0 );

         /* If no data bytes were transmitted */
         if ( pWriteData == pReadData )
         {
            /* No slave acknowledged the address */
            TransferStatus = I2C0_STATUS_ADDR_NAK;
         }   
         else /* Data bytes were transmitted */
         {   
            /* The slave does not want another byte */
            TransferStatus = I2C0_STATUS_DATA_NAK;
         }   
      }
      else /* The address byte of a read was not acknowledged */
      {
         /* Try again (no hold) */
         SI32_I2C_A_set_start( SI32_I2C_0 );
      }
   }
   
   /* Clear the transmit interrupt */
   SI32_I2C_A_clear_tx_interrupt( SI32_I2C_0 );
}


/*****************************************************************************/
/* I2C0_ack_intr_handler                                                     */
/*****************************************************************************/

void I2C0_ack_intr_handler( void )
{
   /* If acknowledge is requested */
   if ( SI32_I2C_A_is_ack_requested(SI32_I2C_0) )
   {
      /* If all data has been received */
      if ( (ReadRemaining == 0) && (SI32_I2C_A_get_byte_count(SI32_I2C_0) == 
            SI32_I2C_A_get_byte_pointer(SI32_I2C_0)) )
      {
         /* Tell the slave not to send another byte */
         SI32_I2C_A_send_nack( SI32_I2C_0 );
      }
      else /* More data is needed */
      {
         /* Tell the slave to send another byte */
         SI32_I2C_A_send_ack( SI32_I2C_0 );
      }
   }   

   /* Clear the ack interrupt */
   SI32_I2C_A_clear_ack_interrupt( SI32_I2C_0 );
}


/*****************************************************************************/
/* I2C0_rx_complete_handler                                                  */
/*****************************************************************************/

void I2C0_rx_complete_handler( void )
{
   uint8_t *pRegister;
   uint8_t *pBuffer;
   int ByteCount,x;

   /* Get the number of bytes received */
   ByteCount = SI32_I2C_A_get_byte_count( SI32_I2C_0 );

   /* If 4 bytes were received */
   if ( ByteCount == 0 )
   {
      /* Copy the 4 bytes to the buffer all at once */
      *pReadData++ = SI32_I2C_A_read_data( SI32_I2C_0 );
   }
   else /* Less than 4 bytes were received */
   {
      /* Point to the data register and the buffer */
      pRegister = (uint8_t*)&SI32_I2C_A_read_data( SI32_I2C_0 );
      pBuffer   = (uint8_t*)pReadData;

      /* Copy the bytes to the buffer a byte at a time */
      for ( x=0; x<ByteCount; x++ )
         *pBuffer++ = *pRegister++;
   }
   
   /* If more data is needed */
   if ( ReadRemaining )
   {
      /* If there is less than 4 bytes remaining to read */
      if ( ReadRemaining < 4 )
      {
         /* Set the byte count to the number of bytes remaining */
         SI32_I2C_A_set_byte_count( SI32_I2C_0, ReadRemaining );
         ReadRemaining = 0;
      }
      else /* There are 4 or more bytes remaining to read */
      {
         /* Set the byte count to zero (meaning 4 bytes) */
         SI32_I2C_A_set_byte_count( SI32_I2C_0, 0 );
         ReadRemaining -= 4;
      }

      /* Arm the receiver */
      SI32_I2C_A_arm_rx( SI32_I2C_0 );
   }
   else /* The read transfer is finished */
   {
      /* Generate a stop */
      SI32_I2C_A_set_stop( SI32_I2C_0 );

      /* Read transfer is successful */
      TransferStatus = I2C0_STATUS_SUCCESS;
   }
   
   /* Clear the receive interrupt */
   SI32_I2C_A_clear_rx_interrupt( SI32_I2C_0 );
}


/*****************************************************************************/
/* I2C0_stop_handler                                                         */
/*****************************************************************************/

void I2C0_stop_handler( void )
{
   /* Clear the stop flag */
   SI32_I2C_A_clear_stop( SI32_I2C_0 );

   /* Clear the stop interrupt */
   SI32_I2C_A_clear_stop_interrupt( SI32_I2C_0 );
}


/*****************************************************************************/
/* I2C0_arb_lost_handler                                                     */
/*****************************************************************************/

void I2C0_arb_lost_handler( void )
{
   /* Arbitration was lost */
   TransferStatus = I2C0_STATUS_ARB_LOST;
   
   /* Clear the arbitration lost interrupt */
   SI32_I2C_A_clear_arblost_interrupt( SI32_I2C_0 );
}


/*****************************************************************************/
/* I2C0_timer3_handler                                                       */
/*****************************************************************************/

void I2C0_timer3_handler(void)
{
   /* Reset the I2C module */
   SI32_I2C_A_reset_module( SI32_I2C_0 );

   /* Clear the timer 3 interrupt */
   SI32_I2C_A_clear_timer3_interrupt( SI32_I2C_0 );
}


