/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myLCD0.c
*
* Description:    Liquid crystal display (LCD) driver
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

// hal
#include <si32_device.h>
#include <SI32_LCD_A_Type.h>
#include <SI32_RTC_B_Type.h>
#include <SI32_CLKCTRL_A_Type.h>


#define RTC_CLOCK_FREQ  16000
#define REFRESH_RATE    60

// application
#include "myLCD0.h"
//==============================================================================
// Local code constant arrays for alpha and Numeric fonts.
// Public extern declartions in header file.
//==============================================================================
const uwide16_t FontMapNumeric[NUM_NUMERIC] =
{
   CHAR_0,
   CHAR_1,
   CHAR_2,
   CHAR_3,
   CHAR_4,
   CHAR_5,
   CHAR_6,
   CHAR_7,
   CHAR_8,
   CHAR_9
};

const uwide16_t FontMapAlpha[NUM_ALPHA] =
{
   CHAR_A,
   CHAR_B,
   CHAR_C,
   CHAR_D,
   CHAR_E,
   CHAR_F,
   CHAR_G,
   CHAR_H,
   CHAR_I,
   CHAR_J,
   CHAR_K,
   CHAR_L,
   CHAR_M,
   CHAR_N,
   CHAR_O,
   CHAR_P,
   CHAR_Q,
   CHAR_R,
   CHAR_S,
   CHAR_T,
   CHAR_U,
   CHAR_V,
   CHAR_W,
   CHAR_X,
   CHAR_Y,
   CHAR_Z
};
//-----------------------------------------------------------------------------
// Internal function prototypes
//-----------------------------------------------------------------------------
void lcdClear(void);
void lcdSetChar(char c, uwide8_t index);
void lcdSetBitMap(uwide16_t bitmap, uwide8_t index);
//-----------------------------------------------------------------------------
// Internal function prototypes
//-----------------------------------------------------------------------------
void myLCD0_enter_default_config (void)
{

   // Enable APB clock to LCD module
   SI32_CLKCTRL_A_enable_apb_to_modules_0 (SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_LCD0CEN_ENABLED_U32);

   SI32_LCD_A_reset_module(SI32_LCD_0);
   SI32_LCD_A_select_4_mux_mode(SI32_LCD_0);
   SI32_LCD_A_select_auto_contrast_constant_contrast_mode(SI32_LCD_0);
   SI32_LCD_A_write_clock_divider(SI32_LCD_0, (RTC_CLOCK_FREQ/REFRESH_RATE/4/4-1));
   SI32_LCD_A_select_rtc_clock_divider_divide_by_1(SI32_LCD_0);
   SI32_LCD_A_write_number_of_segments(SI32_LCD_0, 32);
   SI32_LCD_A_write_number_of_segments(SI32_LCD_0, 40);
   SI32_LCD_A_write_lcd_contrast(SI32_LCD_0, 21);
}
void myLCD0_enter_off_config (void)
{
   // Enable LCD reset
   //SI32_LCD_A_reset_module(SI32_LCD_0);
   SI32_LCD_0->CONFIG.U32       = 0000000000;
   SI32_LCD_0->CLKCONTROL.U32   = 0000000000;
   SI32_LCD_0->BLKCONTROL.U32   = 0x00000000;
   SI32_LCD_0->SEGCONTROL.U32   = 0x00000000;
   SI32_LCD_0->CTRSTCONTROL.U32 = 0x00000000;
   SI32_LCD_0->VBMCONTROL.U32   = 0x00000000;
   SI32_LCD_0->SEGMASK0.U32     = 0x00000000;
   SI32_LCD_0->SEGMASK1.U32     = 0x00000000;
   SI32_LCD_0->SEGDATA0.U32     = 0x00000000;
   SI32_LCD_0->SEGDATA1.U32     = 0x00000000;
   SI32_LCD_0->SEGDATA2.U32     = 0x00000000;
   SI32_LCD_0->SEGDATA3.U32     = 0x00000000;
   SI32_LCD_0->SEGDATA4.U32     = 0x00000000;

   // disable APB clock to LCD module
   SI32_CLKCTRL_A_disable_apb_to_modules_0 (SI32_CLKCTRL_0,
      SI32_CLKCTRL_A_APBCLKG0_LCD0CEN_ENABLED_U32);

}
//-----------------------------------------------------------------------------
// myLCD0_put_string()
//
// This function will output a string to the lcd display. The LCD normally needs
// to be cleared before each string. So the AUTO_CLEAR_DISPLAY feature has been
// added to the put string function. This may be disabled by commenting out the
// build option in the header file.
//
// A generic character pointer is used. The string may be located in xdata or
// in code space.
//
// Note that the putString function depends on a terminating nul character.
// If the string does not have a terminating null character this function
// will continue to put characters until it hits a null byte in memory.
// This is normal behavdior for the putString function.
//
//-----------------------------------------------------------------------------
void myLCD0_put_string (char * string)
{
#ifdef  AUTO_CLEAR_DISPLAY
   myLCD0_put_char('\r');                   // carraige return resets index
   myLCD0_put_char('\n');                   // new line clears display
#endif

   while(*string)
   {
      myLCD0_put_char(*string);
      string++;
   }
}
//-----------------------------------------------------------------------------
// myLCD0_put_number()
//
// lcdPutNumber() displays an unsigned 16-bit number on the LCD in decimal.
// If the number has fewer digits than the minimum width then the LCD is
// padded on the left with spaces. The number is right aligned within the
// the minimum width.  If the number has more digits than the minimum width,
// then the all the digits are displayed and the minimum width is exceeded.
//
//-----------------------------------------------------------------------------
void myLCD0_put_number( int number, int min_width, bool hide_leading_zero )
{
   char digit;
   int  place;
   int  width;
   bool leading = true;

   /* For each decimal place (greater than one) */
   for ( place=10000, width=5; place>1; width--,place/=10 )
   {
      /* Get the digit for the place */
      digit = '0' + (number/place);
      
      /* If it's a leading zero */
      if ( digit=='0' && leading )
      {
         /* If within the minimum field width */
         if ( width <= min_width )
         {
            if ( hide_leading_zero )
               myLCD0_put_char(' ');  /* Display space */
            else
               myLCD0_put_char('0');  /* Display zero */
         }  
         /* Display nothing if greater than the minimum field width */ 
      }
      else /* It's not a leading zero */
      {
         /* Display the digit */
         myLCD0_put_char( digit );
         leading = false;
      }

      /* Remove the place from the number */
      number %= place;
   }
   
   /* One's place */
   myLCD0_put_char( '0' + number );
}
//-----------------------------------------------------------------------------
// myLCD0_put_char()
//
// The myLCD0_put_char() function is specific to the particular display.
//
// A carraige return character resets the index to zero.
//
// A new line character will clear the display. It is necessary to clear the
// display before new string. Each putchar operation will use a logical or
// operation to set the active segment bits.
//
// Any decimal points or colons in the string are treated as special
// characters. The behavior of special character depends on the display.
//
// For the VIM878, the decimal point is associated with the character to the
// left. So the decimal point will set the bit of the previous character,
// and does not increment the index. The decimal point is only valid for
// an index of 1 through 8.
//
// For the VIM878, the apostrophe is associated with the character to the
// right. So the decimal point will set the bit of the current character,
// and does not increment the index. The apostrophe is only valid for
// an index of 0 through 7.
//
// Note that repeated decimal points or appostrophes will not increment
// the index. So a space is needed between repeated decimal points
// or appostrophes.
//
//-----------------------------------------------------------------------------
void myLCD0_put_char (char c)
{
   static uwide8_t index;

   if(c=='\r')
   {
      index=0;                         // carraige return resets index
   }
   else if(c=='\n')
   {
      lcdClear();                      // new line clears display
   }
   else if(c=='.')
   {
      if ((index>0)&&(index<9))
      {
         lcdSetChar(c, index-1);       // set period in previous character                                          // index not incremented
      }
   }
   else if(c=='\'')
   {
      if(index < 8)
      {
         lcdSetChar(c, index);         // set apostrophe
      }                                // index not incremented
   }
   else if(index<8)                    // display overflow ignored
   {
      lcdSetChar(c, index);
      index++;
   }
}
//-----------------------------------------------------------------------------
// lcdClear()
//
// This function will clear all LCD data bits, clearing the display. A bitwise
// logical OR is used to set bits. So clearing the display is necessary for
// each new string.
//-----------------------------------------------------------------------------
void lcdClear(void)
{
   SI32_LCD_A_clear_all_segment_data(SI32_LCD_0);
}
//-----------------------------------------------------------------------------
// lcdSetChar()
//
// This function will use the VIM878 font map character constants to look-up
// the corresponding bitmap for the current character. The VI401 uses a
// 16-bit bitmap.
//
// The VIM878 font map in the header file supports numbers 0-9 and upper
// case alpha characters A-Z. Lower case a-z are displayed as upper-case
// characters.
//
// Supported special characters are PERIOD, APOSTROPHE, PLUS, MINUS, and
// ASTERIX.
//
// The AT_SIGN character '@' will be displayed as a starburst pattern will all
// alphanumeric segments on, not including the appostrophes or decimal points.
// This can be used to test the display. To turn on all segment, including
// appostrophes and decimal points use this string:
// "'@.'@.'@.'@.'@.'@.'@.'@."
//
// Other characters are treated as spaces.
//
//-----------------------------------------------------------------------------
void lcdSetChar(char c, uwide8_t index)
{
   uint16_t bitmap;

   if(c >= '0' && c <= '9')
   {
      bitmap = FontMapNumeric[c - '0'];
   }
   else if(c >= 'A' && c <= 'Z')
   {
      bitmap = FontMapAlpha[c - 'A'];
   }
   else if(c >= 'a' && c <= 'z')
   {
      bitmap = FontMapAlpha[c - 'a'];
   }
   // special character handling
   else if(c =='.')
   {
      bitmap = CHAR_PERIOD;
   }
   else if(c =='\'')
   {
      bitmap = CHAR_APOSTROPHE;
   }
   else if(c =='+')
   {
      bitmap = CHAR_PLUS;
   }
   else if(c =='-')
   {
      bitmap = CHAR_MINUS;
   }
   else if(c =='*')
   {
      bitmap = CHAR_ASTERIX;
   }
   else if(c =='/')
   {
      bitmap = CHAR_SLASH;
   }
   else if(c =='%')
   {
      bitmap = CHAR_PERCENT;
   }
   else if(c =='@')
   {
      bitmap = CHAR_AT_SIGN;
   }
   else
   {
      bitmap = 0x0000;
   }

   lcdSetBitMap(bitmap, index);
}
//-----------------------------------------------------------------------------
// lcdSetBitMap()
//
// This function uses the character bitmap to set bits in the LCD data sfrs.
//
// The data organization depends on the nature of the display.
//
// For a 4-mux display , a 16-bit word represents one character. All bits
// are used.
//
//-----------------------------------------------------------------------------
void lcdSetBitMap(uwide16_t bitmap, uwide8_t index)
{
   uwide16_t value;

   value = SI32_LCD_A_read_segment_data_u16(SI32_LCD_0,index);

   value |= bitmap;

   SI32_LCD_A_write_segment_data_u16(SI32_LCD_0,value, index);
}
//=============================================================================
// end of file
//=============================================================================

