/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myPB.c
*
* Description:    Use port match interrupt to handle button presses
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_PMU_A_Type.h>
#include "main.h"
#include "myPB.h"
#include "myRTC0.h"


/* Time of the previous joystick match interrupt */
uint32_t PreviousJoyMatch = 0;

/* Time of the previous Sx match interrupt */
uint32_t PreviousSxMatch  = 0;


//==============================================================================
// 2nd Level Interrupt Handlers (Called from generated code)
//==============================================================================
void PMATCH_handler( void )
{
   uint32_t CurrentTime;
   uint32_t Pins;
   uint32_t Mask;
   uint32_t Match;

   /* Get the current time from the RTC timer */
   CurrentTime = RTC0_ReadTimer();

   /* If the joystick button caused the pin wake interrupt */
   if ( SI32_PMU_A_is_pin_wakeup_flag_set(SI32_PMU_0) )
   {
      /* Clear the pin wake event */
      SI32_PMU_A_clear_wakeup_flags( SI32_PMU_0 );

      /* Debounce joystick by ignoring interrupts that happen within 20 ms */
      if ( RTC0_ElapsedTime(&PreviousJoyMatch,&CurrentTime) > 660 )
      {
         /* Change the match for the joystick button */
         Match = SI32_PMU_A_read_pwpol( SI32_PMU_0 );
         SI32_PMU_A_write_pwpol( SI32_PMU_0, Match^JOY_EVENT );

         /* If the center joystick button is pressed then set the joy event */
         if ( Match & JOY_EVENT )
            Events |= JOY_EVENT;   

         /* Note the time of the joystick match */
         PreviousJoyMatch = RTC0_ReadTimer();
      }
   } 

   /* Debounce Sx button by ignoring interrupts that happen within 20 ms */
   if ( RTC0_ElapsedTime(&PreviousSxMatch,&CurrentTime) > 660 )
   {
      /* Read PB1 pins */
      Pins = SI32_PBSTD_A_read_pins( SI32_PBSTD_1 );

      /* For each button on PB1 */
      for ( Mask=S1_EVENT; Mask<=S3_EVENT; Mask<<=1 )
      {
         /* Read the PB1 match register */
         Match = SI32_PBSTD_A_read_pm( SI32_PBSTD_1 );

         /* If the button caused the port match interrupt */
         if ( (Pins & Mask) == (Match & Mask) )
         {
            /* Change the match for the button */
            SI32_PBSTD_A_write_pm( SI32_PBSTD_1, Match^Mask );

            /* If the button is pressed then set the Sx event */
            if ( (Pins & Mask) == 0 )
               Events |= Mask;   
   
            /* Note the time of the Sx match */
            PreviousSxMatch = RTC0_ReadTimer();
         }
      }
   }
}

