/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Sensor_Si7005.c
*
* Description:    Read the temperature and humidity from a Si7005 device
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include "myCpu.h"
#include "mySensor.h"


/* Chip Select (CS) lines for bus 1 and bus 2 */
#define CS1                0x00000800
#define CS2                0x00000004

/* I2C slave address of Si7005 */
#define SI7005_ADDR        0x40

/* Si7005 Registers */
#define REG_STATUS         0x00
#define REG_DATA           0x01
#define REG_CONFIG         0x03
#define REG_ID             0x11

/* Status Register */
#define STATUS_NOT_READY   0x01

/* Config Register */
#define CONFIG_START       0x01
#define CONFIG_HEAT        0x02
#define CONFIG_HUMIDITY    0x00
#define CONFIG_TEMPERATURE 0x10
#define CONFIG_FAST        0x20

/* ID Register */
#define ID_Si7005          0x50
#define ID_SI7015          0xF0

/* Coefficients */
#define TEMPERATURE_OFFSET   50
#define TEMPERATURE_SLOPE    32
#define HUMIDITY_OFFSET      24
#define HUMIDITY_SLOPE       16
#define SCALAR            16384
#define A0              (-78388)    /* -4.7844  * SCALAR */
#define A1                 6567     /*  0.4008  * SCALAR */
#define A2                 (-64)    /* -0.00393 * SCALAR */
#define Q0                 3233     /*  0.1973  * SCALAR */
#define Q1                   39     /*  0.00237 * SCALAR */


/*****************************************************************************/
/* Sensor_enter_default_mode_from_reset                                      */
/*****************************************************************************/

void Sensor_enter_default_mode_from_reset( void )
{
   /* Put the Si7005s to sleep */   
   SI32_PBSTD_A_write_pins_high( SI32_PBSTD_0, CS1|CS2 );
}


/*****************************************************************************/
/* Sensor_WakeUp                                                             */
/*****************************************************************************/

void Sensor_WakeUp( int Bus )
{
   /* Run the Si7005 by making the CS line low */
   if ( Bus == I2C0_BUS_1 )
      SI32_PBSTD_A_write_pins_low( SI32_PBSTD_0, CS1 );
   else
      SI32_PBSTD_A_write_pins_low( SI32_PBSTD_0, CS2 );

   /* Give the Si7005 a chance to wake up */
   Delay( 20 );
}


/*****************************************************************************/
/* Sensor_Sleep                                                              */
/*****************************************************************************/

void Sensor_Sleep( int Bus )
{
   /* Put the Si7005 to sleep by making the CS line high */
   if ( Bus == I2C0_BUS_1 )
      SI32_PBSTD_A_write_pins_high( SI32_PBSTD_0, CS1 );
   else
      SI32_PBSTD_A_write_pins_high( SI32_PBSTD_0, CS2 );
}


/*****************************************************************************/
/* Sensor_Measure                                                            */
/*****************************************************************************/

int Sensor_Measure( int Bus, uint8_t Config, int *Value )
{
   uint32_t StartTime;
   uint8_t  Status;
   uint8_t  Data[2];
   int      Result;

   /* Start the conversion */
   Result = I2C0_WriteByte( Bus, SI7005_ADDR, REG_CONFIG, CONFIG_START|Config );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Initialize working variables */
   Status    = STATUS_NOT_READY;
   StartTime = get_msTicks();

   /* Poll until the conversion is ready */   
   while( Status & STATUS_NOT_READY )
   {
      Result = I2C0_ReadByte( Bus, SI7005_ADDR, REG_STATUS, &Status );
      if ( Result != I2C0_STATUS_SUCCESS ) return Result;
      if ( ElapsedTime(StartTime) > 500 ) return I2C0_STATUS_TIMEOUT;
   } 

   /* Stop the conversion */
   Result = I2C0_WriteByte( Bus, SI7005_ADDR, REG_CONFIG, Config );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Read the data */
   Result = I2C0_ReadData( Bus, SI7005_ADDR, REG_DATA, Data, 2 );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Return the data value */
   *Value = (Data[0]*256) + Data[1];
   
   return I2C0_STATUS_SUCCESS;
}


/*****************************************************************************/
/* Sensor_ReadTemperature                                                    */
/*                                                                           */
/* Temperature is returned in deci-degrees Celsius.                          */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

int Sensor_ReadTemperature( int Bus, int *Temperature )
{
   int Value;
   int Result;

   /* Measure the temperature */
   Result = Sensor_Measure( Bus, CONFIG_TEMPERATURE, &Value );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Convert the temperature to deci-degree Celsius */
   *Temperature = ((Value>>2)*10)/TEMPERATURE_SLOPE - TEMPERATURE_OFFSET*10;

   return I2C0_STATUS_SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ReadHumidity                                                       */
/*                                                                           */
/* Humidity is returned in deci-percent relative humidity.                   */
/* For example: 48.2 percent relative humidity is returned as 482.           */
/*                                                                           */
/*****************************************************************************/

int Sensor_ReadHumidity( int Bus, int Temperature, int *Humidity )
{
   int Value;
   int Curve;
   int Linear;
   int Result;

   /* Measure the humidity */
   Result = Sensor_Measure( Bus, CONFIG_HUMIDITY, &Value );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Convert the humidity to deci-percent */
   Curve = ((Value>>4)*10)/HUMIDITY_SLOPE - HUMIDITY_OFFSET*10;
   
   /* Linearization */
   Linear = (Curve*SCALAR - (Curve*Curve*A2)/10 - Curve*A1 - A0*10) / SCALAR;
   
   /* Temperature Compensation */
   Linear = (Linear*SCALAR + (Temperature-300)*((Linear*Q1)/10 + Q0)) / SCALAR;

   /* Limit the humidity to valid values */
   if ( Linear < 0 )
      *Humidity = 0;
   else if ( Linear > 1000 )
      *Humidity = 1000;
   else
      *Humidity = Linear;      

   return I2C0_STATUS_SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ID                                                                 */
/*****************************************************************************/

uint8_t Sensor_ID( int Bus )
{
   uint8_t ID = 0;
 
   I2C0_ReadByte( Bus, SI7005_ADDR, REG_ID, &ID );

   return ID;
}

