/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Form1.cs
*
* Description:    The Form1 class is the main window of the Data Logger GUI
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

using System;
using System.Drawing;
using System.Windows.Forms;
using System.IO;
using System.IO.Ports;
using System.Reflection;
using ZedGraph;
using Silabs.DataLogger;


namespace Silabs.DataLogger
{
    public partial class Form1 : Form
    {
        /* Private constants */
        const int   STATUS_HIGH_TEMPERATURE  = 0x01;
        const int   STATUS_LOW_TEMPERATURE   = 0x02;
        const int   STATUS_HIGH_HUMIDITY     = 0x04;
        const int   STATUS_LOW_HUMIDITY      = 0x08;
        const int   STATUS_COMM_ERROR        = 0x10;
        const float NO_THRESHOLD             = 3276.7F;

        /* Time zone names */
        string[] TimezoneName = { "Baker","Samoa","Hawaii","Alaska",
         "Pacific","Mountain","Central","Eastern","Atlantic","Greeland",
         "Mid-Atlantic","Central African","Western European",
         "Central European","Eastern European","Eastern African","Moscow",                 
         "India","Bangladesh","Vietnam","China","Japan","Australia Eastern",       
         "Solomon","New Zealand" };            

        /* Data members */
        DataLogger Logger = null;
        GraphPane  HumidityPane;
        GraphPane  TemperaturePane;
        LineItem   HumidityCurve;   
        LineItem   TemperatureCurve;
        int        PortNumber = 0;
        bool       Initialized = false;
        DateTime   LoggerTime;
        
        /* Constructor */
        public Form1()
        {
            InitializeComponent();
        }
        
        /* Open the Data Logger */
        private void OpenDataLogger()
        {
            int x;
            
            /* If the port number is known then try to open it */
            if ( PortNumber != 0 )
            {
                try
                {
                    Logger = new DataLogger( "COM" + PortNumber.ToString() );
                    if ( Logger.ID != 0 )
                       return;
                }
                catch {}
            }

            /* Get a list of the serial ports on this PC */
            string[] port = SerialPort.GetPortNames();

            /* Search the serial ports for the Data Logger */
            for ( x=0; x<port.Length; x++ )
            {
                try
                {
                    Logger = new DataLogger( port[x] );
                    if ( Logger.ID == 0 )
                    {
                        Logger.Dispose();
                        Logger = null;
                    }   
                    else break;
                }
                catch
                {
                    if ( Logger != null )
                    {
                        Logger.Dispose(); 
                        Logger = null;
                    }   
                }
            }

            /* If the Data Logger was not found */
            if ( x == port.Length )  
            {
                labelStatus.Text = "Data Logger not found";
                throw new IOException("Cannot find the Data Logger");
            }   

            /* Save the port number of the port */
            PortNumber = Int32.Parse( port[x].Substring(3) );
        }
        
        /* Initialize the GUI */
        private void Initialize()
        {
            /* Call the timer1_Tick() once a second */
            timer1.Interval = 1000;

            /* Get the GUI configuration */
            GetConfig();

            /* Configure the humidity graph */
            HumidityPane = HumidityGraph.GraphPane;
            HumidityPane.Title.Text = "Humidity";
            HumidityPane.XAxis.Title.Text = "Time";
            HumidityPane.XAxis.Type = AxisType.Date;
            HumidityPane.XAxis.MajorGrid.IsVisible = true;
            HumidityPane.XAxis.Scale.MinGrace = 0;
            HumidityPane.XAxis.Scale.MaxGrace = 0;
            HumidityPane.YAxis.Title.Text = "Humidity. % RH";
            HumidityPane.YAxis.MajorGrid.IsVisible = true;
            HumidityPane.Legend.IsVisible = false;
            HumidityPane.Chart.Fill = new Fill( Color.LightSkyBlue, Color.White, 90 );
            HumidityPane.Border.IsVisible = false;
            HumidityPane.Fill = new Fill( Color.FromArgb(212,208,200) );
            HumidityGraph.PointValueFormat = "0.0";

            /* Configure the temperature graph */
            TemperaturePane = TemperatureGraph.GraphPane;
            TemperaturePane.Title.Text = "Temperature";
            TemperaturePane.XAxis.Title.Text = "Time";
            TemperaturePane.XAxis.Type = AxisType.Date;
            TemperaturePane.XAxis.MajorGrid.IsVisible = true;
            TemperaturePane.XAxis.Scale.MinGrace = 0;
            TemperaturePane.XAxis.Scale.MaxGrace = 0;
            if ( radioFahrenheit.Checked )
                TemperaturePane.YAxis.Title.Text = "Temperature, F";
            else   
                TemperaturePane.YAxis.Title.Text = "Temperature, C";
            TemperaturePane.YAxis.MajorGrid.IsVisible = true;
            TemperaturePane.Legend.IsVisible = false;
            TemperaturePane.Chart.Fill = new Fill( Color.LightPink, Color.White, 90 );
            TemperaturePane.Border.IsVisible = false;
            TemperaturePane.Fill = new Fill( Color.FromArgb(212,208,200) );
            TemperatureGraph.PointValueFormat = "0.0";

            /* Show the graphs */
            TemperatureGraph.Invalidate();
            HumidityGraph.Invalidate();

            /* Open the Data Logger */
            OpenDataLogger();

            /* Display the time and time zone */
            DisplayTime();

            /* Display the GUI version from AssemblyInfo.cs */
            Version GUIVersion = Assembly.GetExecutingAssembly().GetName().Version;
            labelGUIVersion.Text = String.Format("{0}.{1}", 
                GUIVersion.Major, GUIVersion.Minor );

            /* Display the firmware version */    
            labelFWVersion.Text = String.Format("{0:F1}", Logger.Version );    

            /* Get the Data Logger ID */
            int ID = Logger.ID;

            /* The second byte of ID came from MCU's DEVICEID register */
            int MCU = ID >> 8;
            switch ( MCU )
            {
                case 0xD0:
                    labelMCU.Text = "C8051F960";
                    break;
                case 167:
                    labelMCU.Text = "SiM3L167";
                    break;
                default:
                    labelMCU.Text = String.Format("DEVICEID 0x{0:X}", MCU );
                    break;
            }

            /* The first byte of ID came from the sensor's ID register */
            switch ( ID & 0xFF )
            {
                case 0xFF:
                    labelSensor.Text = "Sample";
                    break;
                case 0x10:
                    labelSensor.Text = "Si7001";
                    break;
                case 0x50:
                    labelSensor.Text = "Si7005";
                    break;
                case 0x0D:
                    labelSensor.Text = "Si7013";
                    break;
                case 0xF0:
                    labelSensor.Text = "Si7015";
                    break;
                case 0x14:
                    labelSensor.Text = "Si7020";
                    break;
                case 0x15:
                    labelSensor.Text = "Si7021";
                    break;
                default:
                    labelSensor.Text = String.Format("Si70xx ID 0x{0:X}", ID&0xFF );
                    break;
            }

            /* Get the sample interval */
            int Interval = Logger.SampleInterval;
            
            /* Display the sample interval */
            if ( (Interval % 60) == 0 )
            {
                radioMinutes.Checked = true;
                textSampleInterval.Text = (Interval/60).ToString();
            }
            else
            {
                radioSeconds.Checked = true;
                textSampleInterval.Text = Interval.ToString();
            }
           
            /* Display the tresholds */
            DisplayThresholds();

            /* The GUI is now initialized */
            Initialized = true;
        }

        /* This method is called once a second to refresh the display */
        private void timer1_Tick(object sender, EventArgs e)
        {
            try
            {
                if ( !Initialized )
                {
                    Initialize();
                }
                else /* Initialized */
                {
                    /* Increment the local copy of the time */
                    LoggerTime = LoggerTime.AddSeconds(1);
                }
           
                /* Get the status of the Data Logger */
                int Status = Logger.Status;

                /* Display the status */
                if ( (Status & STATUS_COMM_ERROR) != 0 ) 
                {
                    labelStatus.Text      = "Communication Error";
                    labelHumidity.Text    = "";
                    labelTemperature.Text = "";
                    labelMCU.Text         = "";
                    labelSensor.Text      = "";
                }
                else
                {
                    if ( Status != 0 )
                    {
                       if ( (Status & STATUS_LOW_HUMIDITY)     != 0 ) 
                          labelStatus.Text = "Low Humidity Alarm";
                       if ( (Status & STATUS_HIGH_HUMIDITY)    != 0 ) 
                          labelStatus.Text = "High Humidity Alarm";
                       if ( (Status & STATUS_LOW_TEMPERATURE)  != 0 ) 
                          labelStatus.Text = "Low Temperature Alarm";
                       if ( (Status & STATUS_HIGH_TEMPERATURE) != 0 ) 
                          labelStatus.Text = "High Temperature Alarm";
                    }
                    else labelStatus.Text = "OK";

                    /* Display the humidity */
                    labelHumidity.Text = String.Format("{0:F1} % RH", Logger.Humidity );

                    /* Display the temperature */
                    if ( radioFahrenheit.Checked )
                        labelTemperature.Text = String.Format("{0:F1} F", Math.Round(Logger.Temperature*9F)/5F + 32F);
                    else
                        labelTemperature.Text = String.Format("{0:F1} C", Logger.Temperature);
                }
                
                /* Display the time */
                if ( radio12Hour.Checked )
                    labelTime.Text = String.Format("{0:M/dd/yy   h:mm:ss tt}", LoggerTime );
                else    
                    labelTime.Text = String.Format("{0:M/dd/yy   H:mm:ss}", LoggerTime );

                /* Display the sample count */
                labelSampleCount.Text = Logger.SampleCount.ToString();  

                /* Display logging */
                if ( Logger.Logging )
                    buttonLogging.Text = "Stop Logging";
                else
                    buttonLogging.Text = "Start Logging";
            }
            catch (Exception Error)
            {
                timer1.Enabled = false;
                MessageBox.Show(Error.Message);
                Application.Exit();
            }
        }

        /* When the form is closed */
        private void Form1_FormClosed(object sender, FormClosedEventArgs e)
        {
            /* Clean up and save the GUI configuration */
            timer1.Enabled = false;
            if ( Logger != null )
                Logger.Dispose();
            SaveConfig();   
        }
        
        /* Display the time and time zone */
        private void DisplayTime()
        {
            /* Get the time zone from the Data Logger */
            int Timezone = Logger.Timezone;

            /* Display the time zone */
            if ( Timezone == DataLogger.NO_TIMEZONE )
               labelTimezone.Text = "Not Set";
            else if ( DateTime.Now.IsDaylightSavingTime() )
               labelTimezone.Text = String.Format("{0} Time  (UTC{1:+0;-#})", 
                  TimezoneName[Timezone+11], Timezone );
            else      
               labelTimezone.Text = String.Format("{0} Time  (UTC{1:+0;-#})", 
                  TimezoneName[Timezone+12], Timezone );

            /* Get a local copy of the Data Logger time */
            LoggerTime = Logger.Time;

            /* Display the time */
            if ( radio12Hour.Checked )
                labelTime.Text = String.Format("{0:M/dd/yy   h:mm:ss tt}", LoggerTime );
            else    
                labelTime.Text = String.Format("{0:M/dd/yy   H:mm:ss}", LoggerTime );
        }

        /* Display the thresholds */
        private void DisplayThresholds()
        {
            float Threshold;
            
            /* Display the high humidity threshold */
            Threshold = Logger.HighHumidityThreshold;
            if ( Threshold >= NO_THRESHOLD )
                textHighHumidity.Text = "";
            else   
                textHighHumidity.Text = String.Format("{0:F1}", Threshold );
               
            /* Display the low humidity threshold */
            Threshold = Logger.LowHumidityThreshold;
            if ( Threshold >= NO_THRESHOLD )
                textLowHumidity.Text = "";
            else   
                textLowHumidity.Text = String.Format("{0:F1}", Threshold );

            /* Display the high temperature threshold */
            Threshold = Logger.HighTemperatureThreshold;
            if ( Threshold >= NO_THRESHOLD )
                textHighTemperature.Text = "";
            else if ( radioFahrenheit.Checked )
                textHighTemperature.Text = String.Format("{0:F1}", Math.Round(Threshold*9F)/5F + 32F );
            else   
                textHighTemperature.Text = String.Format("{0:F1}", Threshold );
               
            /* Display the low temperature threshold */
            Threshold = Logger.LowTemperatureThreshold;
            if ( Threshold >= NO_THRESHOLD )
                textLowTemperature.Text = "";
            else if ( radioFahrenheit.Checked )
                textLowTemperature.Text = String.Format("{0:F1}", Math.Round(Threshold*9F)/5F + 32F);
            else   
                textLowTemperature.Text = String.Format("{0:F1}", Threshold );

            /* Make sure the labels have the corrent temperature scale */
            if ( radioFahrenheit.Checked )
            {
                labelHighTemperature.Text = "F";
                labelLowTemperature.Text  = "F";
            }    
            else /* Celsius */
            {                    
                labelHighTemperature.Text = "C";
                labelLowTemperature.Text  = "C";
            }    
        }

        /* Display the log data */
        private void DisplayLog()
        {
            /* If there is no log data then return */
            if ( Logger.Sample == null )
               return;

            /* Create a list of points for the curves */   
            PointPairList HumidityList    = new PointPairList();
            PointPairList TemperatureList = new PointPairList();
            
            /* Initialize statistics */
            float  Temperature;
            float  MinTemperature  = 150;
            float  MaxTemperature  = -50;
            float  MinHumidity     = 100;
            float  MaxHumidity     = 0;
            double SumTemperature  = 0;
            double SumHumidity     = 0;

            /* Populate the list of points from the log data */
            foreach ( Sample sample in Logger.Sample )
            {
                HumidityList.Add( new XDate(sample.TimeStamp), sample.Humidity );
                
                Temperature = sample.Temperature;
                
                if ( radioFahrenheit.Checked )
                    Temperature = (float)Math.Round(Temperature*9F)/5F + 32F;
                
                TemperatureList.Add( new XDate(sample.TimeStamp), Temperature );

                /* Gather statistics */
                if ( Temperature < MinTemperature )
                   MinTemperature = Temperature;
                   
                if ( Temperature > MaxTemperature )
                   MaxTemperature = Temperature;

                if ( sample.Humidity < MinHumidity )
                   MinHumidity = sample.Humidity;
                   
                if ( sample.Humidity > MaxHumidity )
                   MaxHumidity = sample.Humidity;
                   
                SumTemperature += Temperature;
                SumHumidity    += sample.Humidity;
            }

            /* Erase any previous curves */
            if ( TemperatureCurve != null )
                TemperatureCurve.Clear();
            
            if ( HumidityCurve != null )
                HumidityCurve.Clear();
            
            /* Create the curves */
            TemperatureCurve = TemperaturePane.AddCurve("Temperature", TemperatureList,Color.Black, SymbolType.None );
            HumidityCurve    = HumidityPane.AddCurve(   "Humidity",    HumidityList,   Color.Black ,SymbolType.None );

            /* Find the duration of the log data */
            TimeSpan Duration = Logger.Sample[Logger.Sample.Length-1].TimeStamp - Logger.Sample[0].TimeStamp;

            /* Format the x axis and the tool tips based on the duration */
            if ( Duration.Days > 7 )
            {
                if ( radio12Hour.Checked )
                {
                    TemperaturePane.XAxis.Scale.Format = "d/M h tt";
                    TemperatureGraph.PointDateFormat   = "d/M h tt";
                    HumidityPane.XAxis.Scale.Format    = "d/M h tt";
                    HumidityGraph.PointDateFormat      = "d/M h tt";
                }
                else /* 24 hour */
                {
                    TemperaturePane.XAxis.Scale.Format = "d/M H";
                    TemperatureGraph.PointDateFormat   = "d/M H";
                    HumidityPane.XAxis.Scale.Format    = "d/M H";
                    HumidityGraph.PointDateFormat      = "d/M H";
                }
            }
            /* Duration is 1 to 7 days */
            else if ( Duration.Days > 0 )
            {
                if ( radio12Hour.Checked )
                {
                    TemperaturePane.XAxis.Scale.Format = "ddd h:mm tt";
                    TemperatureGraph.PointDateFormat   = "ddd h:mm tt";
                    HumidityPane.XAxis.Scale.Format    = "ddd h:mm tt";
                    HumidityGraph.PointDateFormat      = "ddd h:mm tt";
                }
                else /* 24 hour */
                {
                    TemperaturePane.XAxis.Scale.Format = "ddd H:mm";
                    TemperatureGraph.PointDateFormat   = "ddd H:mm";
                    HumidityPane.XAxis.Scale.Format    = "ddd H:mm";
                    HumidityGraph.PointDateFormat      = "ddd H:mm";
                }
            }
            /* Duration is 1 to 24 hours */
            else if ( Duration.Hours > 0 )
            {
                if ( radio12Hour.Checked )
                {
                    TemperaturePane.XAxis.Scale.Format = "h:mm tt";
                    TemperatureGraph.PointDateFormat   = "h:mm tt";
                    HumidityPane.XAxis.Scale.Format    = "h:mm tt";
                    HumidityGraph.PointDateFormat      = "h:mm tt";
                }
                else /* 24 hour */
                {
                    TemperaturePane.XAxis.Scale.Format = "H:mm";
                    TemperatureGraph.PointDateFormat   = "H:mm";
                    HumidityPane.XAxis.Scale.Format    = "H:mm";
                    HumidityGraph.PointDateFormat      = "H:mm";
                }
            }
            /* Duration is less than an hour */
            else
            {
                if ( radio12Hour.Checked )
                {
                    TemperaturePane.XAxis.Scale.Format = "h:mm:ss tt";
                    TemperatureGraph.PointDateFormat   = "h:mm:ss tt";
                    HumidityPane.XAxis.Scale.Format    = "h:mm:ss tt";
                    HumidityGraph.PointDateFormat      = "h:mm:ss tt";
                }
                else /* 24 hour */
                {
                    TemperaturePane.XAxis.Scale.Format = "H:mm:ss";
                    TemperatureGraph.PointDateFormat   = "H:mm:ss";
                    HumidityPane.XAxis.Scale.Format    = "H:mm:ss";
                    HumidityGraph.PointDateFormat      = "H:mm:ss";
                }
            }

            /* Display the curves on the graph */
            TemperatureGraph.AxisChange();
            HumidityGraph.AxisChange();
            TemperatureGraph.Invalidate();
            HumidityGraph.Invalidate();

            /* Display the statistics */
            if ( radioFahrenheit.Checked )
            {
                labelMinTemperature.Text = String.Format("{0:F1} F", MinTemperature );
                labelMaxTemperature.Text = String.Format("{0:F1} F", MaxTemperature );
                labelAvgTemperature.Text = String.Format("{0:F1} F", SumTemperature/Logger.Sample.Length );
            }
            else
            {
                labelMinTemperature.Text = String.Format("{0:F1} C", MinTemperature );
                labelMaxTemperature.Text = String.Format("{0:F1} C", MaxTemperature );
                labelAvgTemperature.Text = String.Format("{0:F1} C", SumTemperature/Logger.Sample.Length );
            }

            labelMinHumidity.Text = String.Format("{0:F1} % RH", MinHumidity );
            labelMaxHumidity.Text = String.Format("{0:F1} % RH", MaxHumidity );
            labelAvgHumidity.Text = String.Format("{0:F1} % RH", SumHumidity/Logger.Sample.Length );
            
            /* The "Save" button can now be used */
            buttonSave.Enabled = true;
        }

        /* Get Log Data button */
        private void buttonGetLog_Click(object sender, EventArgs e)
        {
            try
            {
                Progress dialog = new Progress( Logger );
                
                dialog.ShowDialog();

                DisplayLog();
            }
            catch ( Exception Error ) { MessageBox.Show(Error.Message); }
        }

        /* Save Log Data button */
        private void buttonSave_Click(object sender, EventArgs e)
        {
            if ( Logger.Sample != null )
            {
                SaveFileDialog dialog = new SaveFileDialog();

                dialog.Filter = "CSV files (*.csv)|*.csv|All files (*.*)|*.*";
                dialog.FilterIndex = 1;

                if ( dialog.ShowDialog() == DialogResult.OK )
                {
                    try
                    {
                        Logger.SaveLog( dialog.FileName );
                    }
                    catch ( Exception Error ) { MessageBox.Show(Error.Message); }
                }    
            }    
        }

        /* Restore Log Data button */
        private void buttonRestore_Click(object sender, EventArgs e)
        {
            OpenFileDialog dialog = new OpenFileDialog();

            dialog.Filter = "CSV files (*.csv)|*.csv|All files (*.*)|*.*";
            dialog.FilterIndex = 1;

            if ( dialog.ShowDialog() == DialogResult.OK )
            {
                try 
                { 
                    Logger.RestoreLog( dialog.FileName ); 
                    DisplayLog();
                }
                catch ( Exception Error ) { MessageBox.Show(Error.Message); }
            }    
        }

        /* Set Sample Interval button */
        private void buttonSetInterval_Click(object sender, EventArgs e)
        {
            int Interval;
            
            try
            {
                /* Get the sample interval from the text box */
                Interval = Int32.Parse( textSampleInterval.Text );
            }
            catch ( Exception Error ) 
            { 
                MessageBox.Show( Error.Message ); 
                return;
            }

            /* Convert the sample interval to seconds */
            if ( radioMinutes.Checked )
               Interval *= 60;

            /* Validate the interval */
            if ( (Interval<1) || (Interval>65535) )
            {
                MessageBox.Show("Sample interval must be from 1 to 65535 seconds"); 
                return;
            }

            /* Set the sample interval */   
            Logger.SampleInterval = Interval;   

            /* Get the sample interval */
            Interval = Logger.SampleInterval;
            
            /* Display the sample interval */
            if ( (Interval % 60) == 0 )
            {
                radioMinutes.Checked = true;
                textSampleInterval.Text = (Interval/60).ToString();
            }
            else
            {
                radioSeconds.Checked = true;
                textSampleInterval.Text = Interval.ToString();
            }
        }

        /* Start/Stop Logging button */
        private void buttonLogging_Click(object sender, EventArgs e)
        {
            if ( buttonLogging.Text.Equals("Start Logging") )
            {
                Logger.Logging = true;
                buttonLogging.Text = "Stop Logging";
            }    
            else /* Stop logging */
            {
                Logger.Logging = false;    
                buttonLogging.Text = "Start Logging";
            }    
        }

        /* Erase Log button */
        private void buttonEraseLog_Click(object sender, EventArgs e)
        {
            DialogResult Result = MessageBox.Show("Are you sure you want to erase the log?",
                "Erase Log", MessageBoxButtons.YesNo, MessageBoxIcon.Warning );
               
            if ( Result == DialogResult.Yes )
                Logger.EraseLog();    
        }

        /* Set Time button */
        private void buttonSetTime_Click(object sender, EventArgs e)
        {
            /* Set the Data Logger time to PC time */
            Logger.Time = DateTime.Now;

            /* Set the Data Logger timezone to the time zone of the PC */
            Logger.Timezone = TimeZone.CurrentTimeZone.GetUtcOffset(DateTime.Now).Hours;
            
            DisplayTime();
        }

        /* Set Thresholds button */
        private void buttonSetThresholds_Click(object sender, EventArgs e)
        {
            float HighHumidityThreshold;
            float LowHumidityThreshold;
            float HighTemperatureThreshold;
            float LowTemperatureThreshold;

            /* Get the high humidity threshold from the text box */
            if ( textHighHumidity.Text.Equals("") )
            {
                HighHumidityThreshold = NO_THRESHOLD;
            }
            else
            {
                try
                {
                    HighHumidityThreshold = Single.Parse( textHighHumidity.Text );
                }
                catch
                { 
                    MessageBox.Show("High Humidity value is incorrectly formated"); 
                    return;
                }
                
                if ( (HighHumidityThreshold < 0) || (HighHumidityThreshold > 100) )
                {
                    MessageBox.Show("High Humidity must be from 0 to 100");
                    return;
                }
            }

            /* Get the low humidity threshold from the text box */
            if ( textLowHumidity.Text.Equals("") )
            {
                LowHumidityThreshold = NO_THRESHOLD;
            }
            else
            {
                try
                {
                    LowHumidityThreshold = Single.Parse( textLowHumidity.Text );
                }
                catch
                { 
                    MessageBox.Show("Low Humidity value is incorrectly formated"); 
                    return;
                }
                
                if ( (LowHumidityThreshold < 0) || (LowHumidityThreshold > 100) )
                {
                    MessageBox.Show("Low Humidity must be from 0 to 100");
                    return;
                }
            }
            
            /* Make sure that high and low thresholds are in the right order */
            if ( (LowHumidityThreshold  != NO_THRESHOLD) && 
                 (HighHumidityThreshold != NO_THRESHOLD) &&
                 (LowHumidityThreshold   > HighHumidityThreshold) )
            {
                MessageBox.Show("High Humidity must be greater than the Low Humidity");
                return;
            }

            /* Get the high temperature threshold from the text box */
            if ( textHighTemperature.Text.Equals("") )
            {
                HighTemperatureThreshold = NO_THRESHOLD;
            }
            else
            {
                try
                {
                    HighTemperatureThreshold = Single.Parse( textHighTemperature.Text );
                }
                catch
                { 
                    MessageBox.Show("High Temperature value is incorrectly formated"); 
                    return;
                }
                
                if ( radioCelsius.Checked )
                {
                    if ( (HighTemperatureThreshold < -40F) || (HighTemperatureThreshold > 85F) )
                    {
                        MessageBox.Show("High Temperature must be from -40C to 85C");
                        return;
                    }
                }
                else /* Fahrenheit */
                {
                    if ( (HighTemperatureThreshold < -40F) || (HighTemperatureThreshold > 185F) )
                    {
                        MessageBox.Show("High Temperature must be from -40F to 185F");
                        return;
                    }
                }

                if ( radioFahrenheit.Checked )
                    HighTemperatureThreshold = (HighTemperatureThreshold - 32F)*5F/9F;
            }

            /* Get the low temperature threshold from the text box */
            if ( textLowTemperature.Text.Equals("") )
            {
                LowTemperatureThreshold = NO_THRESHOLD;
            }
            else
            {
                try
                {
                    LowTemperatureThreshold = Single.Parse( textLowTemperature.Text );
                }
                catch
                { 
                    MessageBox.Show("Low Temperature value is incorrectly formated"); 
                    return;
                }
                
                if ( radioCelsius.Checked )
                {
                    if ( (LowTemperatureThreshold < -40F) || (LowTemperatureThreshold > 85F) )
                    {
                        MessageBox.Show("Low Temperature must be from -40C to 85C");
                        return;
                    }
                }
                else /* Fahrenheit */
                {
                    if ( (LowTemperatureThreshold < -40F) || (LowTemperatureThreshold > 185F) )
                    {
                        MessageBox.Show("Low Temperature must be from -40F to 185F");
                        return;
                    }
                }

                if ( radioFahrenheit.Checked )
                    LowTemperatureThreshold = (LowTemperatureThreshold - 32F)*5F/9F;
            }

            /* Make sure that high and low thresholds are in the right order */
            if ( (LowTemperatureThreshold  != NO_THRESHOLD) && 
                 (HighTemperatureThreshold != NO_THRESHOLD) &&
                 (LowTemperatureThreshold   > HighTemperatureThreshold) )
            {
                MessageBox.Show("High Temperature must be greater than the Low Temperature");
                return;
            }

            /* Finally! Set the thresholds */
            Logger.HighHumidityThreshold    = HighHumidityThreshold;
            Logger.LowHumidityThreshold     = LowHumidityThreshold;
            Logger.HighTemperatureThreshold = HighTemperatureThreshold;
            Logger.LowTemperatureThreshold  = LowTemperatureThreshold;
            
            DisplayThresholds();
        }

        /* Celsius radio button has changed */
        private void radioCelsius_CheckedChanged(object sender, EventArgs e)
        {
            if ( Initialized )
            {
                /* Redisplay the temperature if there is not a communicatons error */
                if ( (Logger.Status & STATUS_COMM_ERROR) == 0 )
                { 
                    if ( radioFahrenheit.Checked )
                        labelTemperature.Text = String.Format("{0:F1} F", Math.Round(Logger.Temperature*9F)/5F + 32F);
                    else
                        labelTemperature.Text = String.Format("{0:F1} C", Logger.Temperature);
                }    

                /* Set the scale on the Y axis of the graphs */
                if ( radioFahrenheit.Checked )
                    TemperaturePane.YAxis.Title.Text = "Temperature, F";
                else   
                    TemperaturePane.YAxis.Title.Text = "Temperature, C";

                /* Display thresholds */    
                DisplayThresholds();    

                /* Redisplay the graphs if there is log data */
                if ( Logger.Sample != null )
                    DisplayLog();
                else    
                    TemperatureGraph.Invalidate();
            }
        }

        /* 12 Hour radio button has changed */
        private void radio12Hour_CheckedChanged(object sender, EventArgs e)
        {
            if ( Initialized )
            {
                /* Redisplay the time */
                if ( radio12Hour.Checked )
                    labelTime.Text = String.Format("{0:M/dd/yy   h:mm:ss tt}", LoggerTime );
                else    
                    labelTime.Text = String.Format("{0:M/dd/yy   H:mm:ss}", LoggerTime );
                
                /* Redisplay the graphs if there is log data */
                if ( Logger.Sample != null )
                    DisplayLog();
            }
        }

        /* Save the GUI configuration to a file */
        private void SaveConfig()
        {
            FileStream stream;

            try   { stream = new FileStream("DataLogger.cfg", FileMode.Create ); }
            catch { return; }

            BinaryWriter writer = new BinaryWriter( stream );
            
            writer.Write( PortNumber );
            writer.Write( radioCelsius.Checked );
            writer.Write( radio12Hour.Checked  );

            writer.Close();
            stream.Close();
        }

        /* Get the GUI configuration from a file */
        private void GetConfig()
        {
            FileStream stream;

            try   { stream = new FileStream("DataLogger.cfg", FileMode.Open ); }
            catch { return; }

            BinaryReader reader = new BinaryReader( stream );
            
            PortNumber = reader.ReadInt32();
            
            if ( reader.ReadBoolean() )
                radioCelsius.Checked = true;
            else
                radioFahrenheit.Checked = true;   
            
            if ( reader.ReadBoolean() )
                radio12Hour.Checked  = true;
            else
                radio24Hour.Checked = true;    

            reader.Close();
            stream.Close();
        }
    }
}

