/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Sample.cs
*
* Description:    The Sample class represents a sample in the log
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

using System;
using System.IO;


namespace Silabs.DataLogger
{
    public class Sample
    {
        /* Private constants */
        private const int SEC_PER_MINUTE = 60;
        private const int SEC_PER_HOUR   = SEC_PER_MINUTE * 60;
        private const int SEC_PER_DAY    = SEC_PER_HOUR * 24;
        private const int TIMER_MODE     =  16;
        private const int NO_TIMEZONE    = -16;

        /* Private data members */
        private int      timezone;
        private DateTime timestamp;
        private TimeSpan timerstamp;
        private float    temperature;
        private float    humidity;

        /* Constructor */
        public Sample(PacketSample packetSample, int Timezone)
        {
            timezone = Timezone;

            /* If in timer mode */
            if (Timezone == TIMER_MODE)
            {
                int Timer   = (int)packetSample.TimeStamp;
                int Days    = Timer / SEC_PER_DAY;    Timer %= SEC_PER_DAY;
                int Hours   = Timer / SEC_PER_HOUR;   Timer %= SEC_PER_HOUR;
                int Minutes = Timer / SEC_PER_MINUTE; Timer %= SEC_PER_MINUTE;
                int Seconds = Timer;

                timerstamp = new TimeSpan(Days, Hours, Minutes, Seconds);
            }
            else /* In clock mode */
            {
                uint SysTime = packetSample.TimeStamp;
                int Year;
                int Month;

                for (Year = 0; SysTime >= SecondsInYear(Year); Year++)
                    SysTime -= SecondsInYear(Year);

                for (Month = 1; SysTime >= SecondsInMonth(Month, Year); Month++)
                    SysTime -= SecondsInMonth(Month, Year);

                int Day    = (int)(SysTime / SEC_PER_DAY) + 1; SysTime %= SEC_PER_DAY;
                int Hour   = (int)(SysTime / SEC_PER_HOUR);    SysTime %= SEC_PER_HOUR;
                int Minute = (int)(SysTime / SEC_PER_MINUTE);  SysTime %= SEC_PER_MINUTE;
                int Second = (int)SysTime;

                timestamp = new DateTime(2000 + Year, Month, Day, Hour, Minute, Second);
            }

            temperature = ((float)packetSample.Temperature) / 10F;

            humidity = ((float)packetSample.Humidity) / 10F;
        }

      /* Constructor */
      public Sample( DateTime Timestamp, float Temperature, float Humidity )
      {
         timezone    = NO_TIMEZONE;
         timestamp   = Timestamp;
         temperature = Temperature;
         humidity    = Humidity;
      }

      /* Constructor */
      public Sample( TimeSpan Timerstamp, float Temperature, float Humidity )
      {
         timezone    = TIMER_MODE;
         timerstamp  = Timerstamp;
         temperature = Temperature;
         humidity    = Humidity;
      }

        /* TimeStamp property */
        public DateTime TimeStamp
        {
            get
            {
                if (timezone == TIMER_MODE)
                    throw new IOException("Must be in clock mode to get TimeStamp");

                return timestamp;
            }
        }

        /* TimerStamp property */
        public TimeSpan TimerStamp
        {
            get
            {
                if (timezone != TIMER_MODE)
                    throw new IOException("Must be in timer mode to get TimerStamp");

                return timerstamp;
            }
        }

        /* Temperature property */
        public float Temperature
        {
            get { return temperature; }
        }

        /* Humidity property */
        public float Humidity
        {
            get { return humidity; }
        }

        /* SecondsInYear method */
        private uint SecondsInYear(int Year)
        {
            if ((Year % 4) == 0)
                return 366 * SEC_PER_DAY;
            else
                return 365 * SEC_PER_DAY;
        }

        /* SecondsInMonth method */
        private uint SecondsInMonth(int Month, int Year)
        {
            if (Month == 2)
            {
                if ((Year % 4) == 0)
                    return 29 * SEC_PER_DAY;
                else
                    return 28 * SEC_PER_DAY;
            }
            else if (Month == 4 || Month == 6 || Month == 9 || Month == 11)
            {
                return 30 * SEC_PER_DAY;
            }
            else
            {
                return 31 * SEC_PER_DAY;
            }
        }
    }
}

