/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Demo
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Main.c
*
* Description:    Read temperature and humidity from the sensor device
*                 and display them on the LCD
*
* Revision History:
*
*   09/17/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "lcdPutString.h"
#include "lcdConfig.h"
#include "lcdPutNumber.h"
#include "SmaRTClock.h"
#include "PowerManagement.h"
#include "Tick.h"
#include "I2C.h"
#include "Sensor.h"
#include "PortMatch.h"


/* Pause button */
#define SW4  0x08


/*****************************************************************************/
/* main                                                                      */
/*****************************************************************************/

void main (void)
{
   S16 Temperature;
   S16 Humidity;
   U8  Forever = 1;
   S8  Result;
   
   /* Disable the watchdog timer and the VBAT monitor */
   SFRPAGE = LEGACY_PAGE;
   PCA0MD &= ~0x40;
	VDM0CN  =  0x00;

   /* Use the internal low power oscillator (SYSCLK = 20 MHz) */   
   CLKSEL = 0x04;

   /* Configure the LCD */
   LCD0_Config( 4, 32, 1, 2900, 60, 1 );

   /* Initialize the other components */
   RTC_Init();
   Tick_Init();
   I2C_Init();
   PortMatch_Init();
   LPM_Init();

   /* Enable interrupts */
   EA = 1;

   /* Initialize the sensor */
   Sensor_Init();

   /* Configure RTC alarm 0 for 500 ms with auto reset */
   RTC_Write( RTC0CF, 0x00 );
   RTC_WriteAlarm( 0, 16384 );
   RTC_Write( RTC0CF, 0x09 );

   /* Start the RTC timer */
   RTC_Write( RTC0CN, 0x90 );

   /* Wake up from sleep mode on RTC alarm or port match */
   LPM_Enable_Wakeup( RTC|PORT_MATCH );

   /* If the SW4 button is pressed */
   if ( (P0 & SW4) == 0 )
   {
      lcdPutString("PAUSED");
      
      /* Pause until SW4 is released */
      /* This gives the IDE or flash utility a chance to */
      /* get control before the MCU enters sleep mode    */
      while ( (P0 & SW4) == 0 );
   }

   while ( Forever )
   {
      /* Sleep for a while */
      LPM( SLEEP );
      
      /* If woke up because of an RTC alarm 0 */
      if ( RTC_Alarm0_WakeUp )
      {
         /* Clear the RTC alarm 0 event flag */
         RTC_Write( RTC0CF, 0x00 );
         RTC_Write( RTC0CF, 0x09 );

         /* Wake up the sensor */
         Sensor_WakeUp( I2C_BUS_1 );

         /* Read the temperature */
         if ( Thermistor )
            Result = Sensor_ReadThermistor( I2C_BUS_1, &Temperature );
         else   
            Result = Sensor_ReadTemperature( I2C_BUS_1, &Temperature );
         if ( Result == SUCCESS )
         {   
            /* Read the humidity */
            Result = Sensor_ReadHumidity( I2C_BUS_1, Temperature, &Humidity );
            if ( Result == SUCCESS )
            {
               /* Display the temperature and humidity */
               lcdPutTemperature( Temperature );
               lcdPutHumidity( Humidity );
            }   
         }

         if ( Result != SUCCESS )
            lcdPutString("ERROR");   

         /* Put the sensor back to sleep */
         Sensor_Sleep( I2C_BUS_1 );
      }
   }

   /* These routines exist, but are not needed by this application.  They */
   /* are called here to keep the linker happy.  The above "Forever" loop */
   /* never exits, so these routines are never really called.             */
   RTC_SetTimer(0);
   RTC_CaptureTimer();
   RTC_ReadAlarm(0);
   LPM_Disable_Wakeup(0);
   I2C_SlavePresent(0,0);
   I2C_ReadByte(0,0,0,&Forever);
   I2C_ReadDataWrite2(0,0,0,0,&Forever,1);
}

