/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Demo
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Sensor_Si7005.c
*
* Description:    Read the temperature and humidity from a Si7005 device
*
* Revision History:
*
*   09/17/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Tick.h"
#include "I2C.h"
#include "Sensor.h"


/* Chip Select (CS) lines for bus 1 and bus 2 */
SBIT( CS1, SFR_P1, 7 );
SBIT( CS2, SFR_P1, 4 );

/* I2C slave address of Si7005 */
#define SI7005_ADDR        0x40

/* Si7005 Registers */
#define REG_STATUS         0x00
#define REG_DATA           0x01
#define REG_CONFIG         0x03
#define REG_ID             0x11

/* Status Register */
#define STATUS_NOT_READY   0x01

/* Config Register */
#define CONFIG_START       0x01
#define CONFIG_HEAT        0x02
#define CONFIG_HUMIDITY    0x00
#define CONFIG_TEMPERATURE 0x10
#define CONFIG_FAST        0x20

/* ID Register */
#define ID_Si7005          0x50
#define ID_SI7015          0xF0

/* Coefficients */
#define TEMPERATURE_OFFSET   50
#define TEMPERATURE_SLOPE    32
#define HUMIDITY_OFFSET      24
#define HUMIDITY_SLOPE       16
#define SCALAR            16384
#define A0              (-78388)    /* -4.7844  * SCALAR */
#define A1                 6567     /*  0.4008  * SCALAR */
#define A2                 (-64)    /* -0.00393 * SCALAR */
#define Q0                 3233     /*  0.1973  * SCALAR */
#define Q1                   39     /*  0.00237 * SCALAR */


/*****************************************************************************/
/* Sensor_Init                                                               */
/*****************************************************************************/

void Sensor_Init( void )
{
   /* Make CS1 and CS2 lines push-pull */
   P1MDOUT |= 0x90;

   /* Put the Si7005s to sleep */   
   CS1 = TRUE;
   CS2 = TRUE;
}


/*****************************************************************************/
/* Sensor_WakeUp                                                             */
/*****************************************************************************/

void Sensor_WakeUp( U8 Bus )
{
   /* Run the Si7005 by making the CS line low */
   if ( Bus == I2C_BUS_1 )
      CS1 = FALSE;
   else
      CS2 = FALSE;

   /* Give the Si7005 a chance to wake up */
   Delay( 20 );
}


/*****************************************************************************/
/* Sensor_Sleep                                                              */
/*****************************************************************************/

void Sensor_Sleep( U8 Bus )
{
   /* Put the Si7005 to sleep by making the CS line high */
   if ( Bus == I2C_BUS_1 )
      CS1 = TRUE;
   else
      CS2 = TRUE;
}


/*****************************************************************************/
/* Sensor_Measure                                                            */
/*****************************************************************************/

S8 Sensor_Measure( U8 Bus, U8 Config, U16 *Value )
{
   U16 StartTime;
   U8  Status;
   U8  Data[2];
   S8  Result;

   /* Start the conversion */
   Result = I2C_WriteByte( Bus, SI7005_ADDR, REG_CONFIG, CONFIG_START|Config );
   if ( Result != SUCCESS ) return Result;

   /* Initialize working variables */
   Status    = STATUS_NOT_READY;
   StartTime = TickCount();

   /* Poll until the conversion is ready */   
   while( Status & STATUS_NOT_READY )
   {
      Result = I2C_ReadByte( Bus, SI7005_ADDR, REG_STATUS, &Status );
      if ( Result != SUCCESS ) return Result;
      if ( ElapsedTime(StartTime) > 500 ) return FAILURE;
   } 

   /* Stop the conversion */
   Result = I2C_WriteByte( Bus, SI7005_ADDR, REG_CONFIG, Config );
   if ( Result != SUCCESS ) return Result;

   /* Read the data */
   Result = I2C_ReadData( Bus, SI7005_ADDR, REG_DATA, Data, 2 );
   if ( Result != SUCCESS ) return Result;

   /* Return the data value */
   *Value = (Data[0]*256) + Data[1];
   
   return SUCCESS;
}


/*****************************************************************************/
/* Sensor_ReadTemperature                                                    */
/*                                                                           */
/* Temperature is returned in deci-degrees Celsius.                          */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadTemperature( U8 Bus, S16 *Temperature )
{
   U16 Value;
   S8  Result;

   /* Measure the temperature */
   Result = Sensor_Measure( Bus, CONFIG_TEMPERATURE, &Value );
   if ( Result != SUCCESS ) return Result;

   /* Convert the temperature to deci-degree Celsius */
   *Temperature = ((Value>>2)*10)/TEMPERATURE_SLOPE - TEMPERATURE_OFFSET*10;

   return SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ReadThermistor                                                     */
/*                                                                           */
/* Thermistor is returned in deci-degrees Celsius.                           */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadThermistor( U8 Bus, S16 *Temperature )
{
   /* The Si7005 does not have a thermistor, so just read the temperature */
   return Sensor_ReadTemperature( Bus, Temperature );
}


/*****************************************************************************/
/* Sensor_ReadHumidity                                                       */
/*                                                                           */
/* Humidity is returned in deci-percent relative humidity.                   */
/* For example: 48.2 percent relative humidity is returned as 482.           */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadHumidity( U8 Bus, S16 Temperature, S16 *Humidity )
{
   U16 Value;
   S32 Curve;
   S32 Linear;
   S8  Result;

   /* Measure the humidity */
   Result = Sensor_Measure( Bus, CONFIG_HUMIDITY, &Value );
   if ( Result != SUCCESS ) return Result;

   /* Convert the humidity to deci-percent */
   Curve = ((Value>>4)*10)/HUMIDITY_SLOPE - HUMIDITY_OFFSET*10;
   
   /* Linearization */
   Linear = (Curve*SCALAR - (Curve*Curve*A2)/10 - Curve*A1 - A0*10) / SCALAR;
   
   /* Temperature Compensation */
   Linear = (Linear*SCALAR + (Temperature-300)*((Linear*Q1)/10 + Q0)) / SCALAR;

   /* Limit the humidity to valid values */
   if ( Linear < 0 )
      *Humidity = 0;
   else if ( Linear > 1000 )
      *Humidity = 1000;
   else
      *Humidity = Linear;      

   return SUCCESS;   
}

