/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si70xx UDP Demo
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Sensor_Si70xx.c
*
* Description:    Read the temperature and humidity from a Si70xx device
*
* Revision History:
*
*   09/17/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Tick.h"
#include "I2C.h"
#include "Sensor.h"


/* I2C slave address of Si70xx */
#define SI70XX_ADDR                     0x41

/* Commands */
#define CMD_MEASURE_HUMIDITY_HOLD       0xE5
#define CMD_MEASURE_HUMIDITY_NO_HOLD    0xF5
#define CMD_MEASURE_TEMPERATURE_HOLD    0xE3
#define CMD_MEASURE_TEMPERATURE_NO_HOLD 0xF3
#define CMD_MEASURE_THERMISTOR_HOLD     0xEE
#define CMD_READ_PREVIOUS_TEMPERATURE   0xE0
#define CMD_RESET                       0xFE
#define CMD_WRITE_REGISTER_1            0xE6
#define CMD_READ_REGISTER_1             0xE7
#define CMD_WRITE_REGISTER_2            0x50
#define CMD_READ_REGISTER_2             0x10
#define CMD_WRITE_REGISTER_3            0x51
#define CMD_READ_REGISTER_3             0x11
#define CMD_WRITE_COEFFICIENT           0xC5
#define CMD_READ_COEFFICIENT            0x84

/* User Register 1 */
#define REG1_RESOLUTION_MASK            0x81
#define REG1_RESOLUTION_H12_T14         0x00
#define REG1_RESOLUTION_H08_T12         0x01
#define REG1_RESOLUTION_H10_T13         0x80
#define REG1_RESOLUTION_H11_T11         0x81
#define REG1_LOW_VOLTAGE                0x40
#define REG1_ENABLE_HEATER              0x04

/* User Register 2 */
#define REG2_VOUT                       0x01
#define REG2_VREF_VDD                   0x02
#define REG2_VIN_BUFFERED               0x04
#define REG2_RESERVED                   0x08
#define REG2_FAST_CONVERSION            0x10
#define REG2_MODE_CORRECTION            0x20
#define REG2_MODE_NO_HOLD               0x40

/* Device Identification */
#define ID_SAMPLE                       0xFF
#define ID_SI7013                       0x0D
#define ID_SI7020                       0x14
#define ID_SI7021                       0x15

/* Coefficients */
#define COEFFICIENT_BASE                0x82
#define COEFFICIENT_COUNT               9

/* Thermistor Correction Coefficients */
code struct
{
   S16 Input[COEFFICIENT_COUNT];
   U16 Output[COEFFICIENT_COUNT];
   S16 Slope[COEFFICIENT_COUNT];
} CoefficientTable =
{
   /* Input  */{ 19535, 15154, 11187,  7982,  5592,  3895,  2721,  1916,  1367 },
   /* Output */{ 11879, 15608, 19338, 23067, 26797, 30527, 34256, 37986, 41715 },
   /* Slope  */{  -218,  -241,  -298,  -400,  -563,  -813, -1186, -1739, -2513 }
}; 


/*****************************************************************************/
/* Sensor_ReadCoefficient                                                    */
/*****************************************************************************/

S8 Sensor_ReadCoefficient( U8 Bus, U8 Slave, U8 Address, S16 *Coefficient )
{
   U8 Data[2];
   S8 Result;

   /* Read the MSB from the coefficient address */
   Result = I2C_ReadDataWrite2( Bus, Slave, CMD_READ_COEFFICIENT, Address, 
      &Data[0], 1 );
   if ( Result != SUCCESS ) return Result;

   /* Read the LSB from the following coefficient address */   
   Result = I2C_ReadDataWrite2( Bus, Slave, CMD_READ_COEFFICIENT, Address+1, 
      &Data[1], 1 );
   if ( Result != SUCCESS ) return Result;

   /* Combine the MSB and LSB */
   *Coefficient = (Data[0]*256) + Data[1];
   
   return SUCCESS;
}


/*****************************************************************************/
/* Sensor_WriteCoefficient                                                   */
/*****************************************************************************/

S8 Sensor_WriteCoefficient( U8 Bus, U8 Slave, U8 Address, S16 Coefficient )
{
   U8 Data[2];
   S8 Result;

   /* Write the MSB to the coefficient address */
   Data[0] = Address;
   Data[1] = Coefficient >> 8;
   Result = I2C_WriteData( Bus, Slave, CMD_WRITE_COEFFICIENT, Data, 2 );
   if ( Result != SUCCESS ) return Result;

   /* Write the LSB to the following coefficient address */
   Data[0] = Address + 1;
   Data[1] = Coefficient & 0xFF;
   Result = I2C_WriteData( Bus, Slave, CMD_WRITE_COEFFICIENT, Data, 2 );
   if ( Result != SUCCESS ) return Result;

   return SUCCESS;   
}


/*****************************************************************************/
/* Sensor_Init                                                               */
/*****************************************************************************/

void Sensor_Init( void )
{
   S16 Coefficient;
   U8  Address;
   S8  Result;
   U8  x;

   /* Wait for the Si7013 to power up */
   Delay( 25 );
   
   /* If the thermistor correction coefficients are not programmed into the Si7013 */
   Result = Sensor_ReadCoefficient( I2C_BUS_1, SI70XX_ADDR, COEFFICIENT_BASE, &Coefficient );
   if ( (Result == SUCCESS) && (Coefficient == -1) )
   {
      /* Start at the beginning of coefficient memory */
      Address = COEFFICIENT_BASE;

      /* Write the "input" coefficients */
      for ( x=0; x<COEFFICIENT_COUNT; x++ )
      {
         Result = Sensor_WriteCoefficient( I2C_BUS_1, SI70XX_ADDR, Address, 
            CoefficientTable.Input[x] );
         if ( Result != SUCCESS ) return;
         Address += 2;
      }

      /* Write the "output" coefficients */
      for ( x=0; x<COEFFICIENT_COUNT; x++ )
      {
         Result = Sensor_WriteCoefficient( I2C_BUS_1, SI70XX_ADDR, Address, 
            CoefficientTable.Output[x] );
         if ( Result != SUCCESS ) return;
         Address += 2;
      }

      /* Write the "slope" coefficients */
      for ( x=0; x<COEFFICIENT_COUNT; x++ )
      {
         Result = Sensor_WriteCoefficient( I2C_BUS_1, SI70XX_ADDR, Address, 
            CoefficientTable.Slope[x] );
         if ( Result != SUCCESS ) return;
         Address += 2;
      }
      
      /* Reset the Si7013 to activate the coefficients */
      I2C_WriteData( I2C_BUS_1, SI70XX_ADDR, CMD_RESET, 0, 0 );

      /* Wait for the reset to complete */
      Delay( 10 );
   }
}


/*****************************************************************************/
/* Sensor_WakeUp                                                             */
/*****************************************************************************/

void Sensor_WakeUp( U8 Bus )
{
   /* The Si70xx automatically wakes up when you give it a command */
   Bus = I2C_BUS_1;
}


/*****************************************************************************/
/* Sensor_Sleep                                                              */
/*****************************************************************************/

void Sensor_Sleep( U8 Bus )
{
   /* The Si70xx automatically sleeps between commands */
   Bus = I2C_BUS_1;
}


/*****************************************************************************/
/* Sensor_Measure                                                            */
/*****************************************************************************/

S8 Sensor_Measure( U8 Bus, U8 Command, U16 *Value )
{
   U8 Data[2];
   S8 Result;

   /* Measure the humidity or temperature value */
   Result = I2C_ReadData( Bus, SI70XX_ADDR, Command, Data, 2 );
   if ( Result != SUCCESS ) return Result;

   /* Swap the bytes and clear the status bits */
   *Value = ((Data[0]*256) + Data[1]) & ~3;
   
   return SUCCESS;
}


/*****************************************************************************/
/* Sensor_ReadTemperature                                                    */
/*                                                                           */
/* Temperature is returned in deci-degrees Celsius.                          */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadTemperature( U8 Bus, S16 *Temperature )
{
   U16 Value;
   S8  Result;

   /* Measure the temperature */
   Result = Sensor_Measure( Bus, CMD_MEASURE_TEMPERATURE_HOLD, &Value );
   if ( Result != SUCCESS ) return Result;

   /* Convert the temperature to deci-degree Celsius */
   *Temperature = ((((S32)Value)*1757)>>16) - 469;

   return SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ReadThermistor                                                     */
/*                                                                           */
/* Thermistor is returned in deci-degrees Celsius.                           */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadThermistor( U8 Bus, S16 *Temperature )
{
   U16 Value;
   S8  Result;

   /* Bias the thermistor and use the correction coefficients */
   Result = I2C_WriteByte( Bus, SI70XX_ADDR, CMD_WRITE_REGISTER_2, 
      REG2_VREF_VDD|REG2_VIN_BUFFERED|REG2_RESERVED|REG2_MODE_CORRECTION );
   if ( Result != SUCCESS ) return Result;

   /* Give the bias voltage a chance to settle */
   Delay( 10 );

   /* Measure the temperature */
   Result = Sensor_Measure( Bus, CMD_MEASURE_THERMISTOR_HOLD, &Value );
   if ( Result != SUCCESS ) return Result;

   /* Unbias the thermistor and stop using the correction coefficients */
   Result = I2C_WriteByte( Bus, SI70XX_ADDR, CMD_WRITE_REGISTER_2, 
      REG2_VOUT|REG2_VREF_VDD|REG2_VIN_BUFFERED|REG2_RESERVED ); 
   if ( Result != SUCCESS ) return Result;

   /* Convert the temperature to deci-degree Celsius */
   *Temperature = ((((S32)Value)*1757)>>16) - 469;

   return SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ReadHumidity                                                       */
/*                                                                           */
/* Humidity is returned in deci-percent relative humidity.                   */
/* For example: 48.2 percent relative humidity is returned as 482.           */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadHumidity( U8 Bus, S16 Temperature, S16 *Humidity )
{
   U16 Value;
   S32 Decipercent;
   S8  Result;
   
   /* Keep the compiler happy */
   Temperature = 0;

   /* Measure the humidity */
   Result = Sensor_Measure( Bus, CMD_MEASURE_HUMIDITY_HOLD, &Value );
   if ( Result != SUCCESS ) return Result;

   /* Convert the humidity to deci-percent */
   Decipercent = ((((S32)Value)*625)>>15) - 60;

   /* Limit the humidity to valid values */
   if ( Decipercent < 0 )
      *Humidity = 0;
   else if ( Decipercent > 1000 )
      *Humidity = 1000;
   else
      *Humidity = Decipercent;      

   return SUCCESS;   
}

