/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Demo
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      lcdPutNumber.c
*
* Description:    Display a number, temperature or humidity on the LCD
*
* Revision History:
*
*   09/17/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "lcdPutChar.h"
#include "lcdPutNumber.h"
#include "I2C.h"
#include "PortMatch.h"


/*****************************************************************************/
/* lcdPutNumber                                                              */
/*                                                                           */
/* lcdPutNumber() displays an unsigned 8-bit number on the LCD in decimal.   */
/* If the number has fewer digits than the minimum width then the LCD is     */
/* padded on the left with spaces. The number is right aligned within the    */
/* the minimum width.  If the number has more digits than the minimum width, */
/* then the all the digits are displayed and the minimum width is exceeded.  */
/*                                                                           */
/*****************************************************************************/

void lcdPutNumber( U8 Number, U8 MinWidth )
{
   char Digit;
   U8   Place;
   U8   Width;
   U8   Leading = TRUE;

   /* For each decimal place (greater than one) */
   for ( Place=100, Width=3; Place>1; Width--,Place/=10 )
   {
      /* Get the digit for the place */
      Digit = '0' + (Number/Place);
      
      /* If it's a leading zero */
      if ( Digit=='0' && Leading )
      {
         /* If less than the minimum field width */
         if ( Width <= MinWidth )
            /* Display a leading space */
            lcdPutChar(' ');
      }
      else /* It's not a leading zero */
      {
         /* Display the digit */
         lcdPutChar( Digit );
         Leading = FALSE;
      }

      /* Remove the place from the number */
      Number %= Place;
   }
   
   /* One's place */
   lcdPutChar( '0' + Number );
}


/*****************************************************************************/
/* lcdPutTemperature                                                         */
/*****************************************************************************/

void lcdPutTemperature( S16 Temperature )
{
   U8 WholeDegree;
   U8 DeciDegree;
   U8 Negative = FALSE;

   /* Clear the display */
   lcdPutChar('\r');
   lcdPutChar('\n');

   /* If the scale is Fahrenheit then do the conversion */
   if ( Scale == 'F' )
      Temperature = ((Temperature*9)/5) + 320;
   
   /* Normalize the temperature to a positive number */
   if ( Temperature < 0 )
   {
      Temperature = -Temperature;
      Negative = TRUE;
   }

   /* Extract the whole and fractional components */
   WholeDegree = Temperature / 10;
   DeciDegree  = Temperature % 10;

   /* If the temperature is negative */
   if ( Negative )
   {
      lcdPutChar('-');
      if ( WholeDegree >= 10 )
      {
         if ( DeciDegree >= 5 )
            WholeDegree++;
         lcdPutNumber( WholeDegree, 2 );
      }
      else /* Negative but greater than -10 */
      {
         lcdPutNumber( WholeDegree, 1 );
         lcdPutChar('.');
         lcdPutNumber( DeciDegree, 1 );
      }
   }
   /* If the temperature is a triple digit whole number */
   else if ( WholeDegree >= 100 )
   {
      if ( DeciDegree >= 5 )
         WholeDegree++;
      lcdPutNumber( WholeDegree, 3 );
   }
   else /* Positive but less than a hundred */
   {
      lcdPutNumber( WholeDegree, 2 );
      lcdPutChar('.');
      lcdPutNumber( DeciDegree, 1 );
   }

   lcdPutChar('\'');
   lcdPutChar( Scale ); 
   
   if ( Thermistor )
      lcdPutChar('.');
}


/*****************************************************************************/
/* lcdPutHumidity                                                            */
/*****************************************************************************/

void lcdPutHumidity( S16 Humidity )
{
   U8 WholePercent;
   U8 DeciPercent;

   /* Extract the whole and fractional components */
   WholePercent = Humidity / 10;
   DeciPercent  = Humidity % 10;

   /* Round up to the nearest percent */   
   if ( DeciPercent >= 5 )
      WholePercent++;

   lcdPutNumber( WholePercent, 3 );
   lcdPutChar('H');
}

