/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si70xx UDP Demo
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Sensor_Si70xx.c
*
* Description:    Read the temperature and humidity from a Si70xx device
*
* Revision History:
*
*   01/14/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include "myCpu.h"
#include "mySensor.h"


/* I2C slave address of Si70xx */
#define SI70XX_ADDR                     0x41

/* Commands */
#define CMD_MEASURE_HUMIDITY_HOLD       0xE5
#define CMD_MEASURE_HUMIDITY_NO_HOLD    0xF5
#define CMD_MEASURE_TEMPERATURE_HOLD    0xE3
#define CMD_MEASURE_TEMPERATURE_NO_HOLD 0xF3
#define CMD_MEASURE_THERMISTOR_HOLD     0xEE
#define CMD_READ_PREVIOUS_TEMPERATURE   0xE0
#define CMD_RESET                       0xFE
#define CMD_WRITE_REGISTER_1            0xE6
#define CMD_READ_REGISTER_1             0xE7
#define CMD_WRITE_REGISTER_2            0x50
#define CMD_READ_REGISTER_2             0x10
#define CMD_WRITE_REGISTER_3            0x51
#define CMD_READ_REGISTER_3             0x11
#define CMD_WRITE_COEFFICIENT           0xC5
#define CMD_READ_COEFFICIENT            0x84

/* User Register 1 */
#define REG1_RESOLUTION_MASK            0x81
#define REG1_RESOLUTION_H12_T14         0x00
#define REG1_RESOLUTION_H08_T12         0x01
#define REG1_RESOLUTION_H10_T13         0x80
#define REG1_RESOLUTION_H11_T11         0x81
#define REG1_LOW_VOLTAGE                0x40
#define REG1_ENABLE_HEATER              0x04

/* User Register 2 */
#define REG2_VOUT                       0x01
#define REG2_VREF_VDD                   0x02
#define REG2_VIN_BUFFERED               0x04
#define REG2_RESERVED                   0x08
#define REG2_FAST_CONVERSION            0x10
#define REG2_MODE_CORRECTION            0x20
#define REG2_MODE_NO_HOLD               0x40

/* Device Identification */
#define ID_SAMPLE                       0xFF
#define ID_SI7013                       0x0D
#define ID_SI7020                       0x14
#define ID_SI7021                       0x15

/* Coefficients */
#define COEFFICIENT_BASE                0x82
#define COEFFICIENT_COUNT               9

/* Thermistor Correction Coefficients */
struct
{
   int16_t  Input[COEFFICIENT_COUNT];
   uint16_t Output[COEFFICIENT_COUNT];
   int16_t  Slope[COEFFICIENT_COUNT];
} CoefficientTable =
{
   /* Input  */{ 19535, 15154, 11187,  7982,  5592,  3895,  2721,  1916,  1367 },
   /* Output */{ 11879, 15608, 19338, 23067, 26797, 30527, 34256, 37986, 41715 },
   /* Slope  */{  -218,  -241,  -298,  -400,  -563,  -813, -1186, -1739, -2513 }
}; 


/*****************************************************************************/
/* Sensor_ReadCoefficient                                                    */
/*****************************************************************************/

int Sensor_ReadCoefficient( int Bus, uint8_t Slave, uint8_t Address, int16_t *Coefficient )
{
   uint8_t Data[2];
   int     Result;

   /* Read the MSB from the coefficient address */
   Result = I2C0_ReadDataWrite2( Bus, Slave, CMD_READ_COEFFICIENT, Address, 
      &Data[0], 1 );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Read the LSB from the following coefficient address */   
   Result = I2C0_ReadDataWrite2( Bus, Slave, CMD_READ_COEFFICIENT, Address+1, 
      &Data[1], 1 );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Combine the MSB and LSB */
   *Coefficient = (Data[0]*256) + Data[1];
   
   return I2C0_STATUS_SUCCESS;
}


/*****************************************************************************/
/* Sensor_WriteCoefficient                                                   */
/*****************************************************************************/

int Sensor_WriteCoefficient( int Bus, uint8_t Slave, uint8_t Address, int16_t Coefficient )
{
   uint8_t Data[2];
   int     Result;

   /* Write the MSB to the coefficient address */
   Data[0] = Address;
   Data[1] = Coefficient >> 8;
   Result = I2C0_WriteData( Bus, Slave, CMD_WRITE_COEFFICIENT, Data, 2 );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Write the LSB to the following coefficient address */
   Data[0] = Address + 1;
   Data[1] = Coefficient & 0xFF;
   Result = I2C0_WriteData( Bus, Slave, CMD_WRITE_COEFFICIENT, Data, 2 );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   return I2C0_STATUS_SUCCESS;   
}


/*****************************************************************************/
/* Sensor_enter_default_mode_from_reset                                      */
/*****************************************************************************/

void Sensor_enter_default_mode_from_reset( void )
{
   int16_t Coefficient;
   uint8_t Address;
   int     Result;
   int     x;

   /* Wait for the Si7013 to power up */
   Delay( 25 );
   
   /* If the thermistor correction coefficients are not programmed into the Si7013 */
   Result = Sensor_ReadCoefficient( I2C0_BUS_1, SI70XX_ADDR, COEFFICIENT_BASE, &Coefficient );
   if ( (Result == I2C0_STATUS_SUCCESS) && (Coefficient == -1) )
   {
      /* Start at the beginning of coefficient memory */
      Address = COEFFICIENT_BASE;

      /* Write the "input" coefficients */
      for ( x=0; x<COEFFICIENT_COUNT; x++ )
      {
         Result = Sensor_WriteCoefficient( I2C0_BUS_1, SI70XX_ADDR, Address, 
            CoefficientTable.Input[x] );
         if ( Result != I2C0_STATUS_SUCCESS ) return;
         Address += 2;
      }

      /* Write the "output" coefficients */
      for ( x=0; x<COEFFICIENT_COUNT; x++ )
      {
         Result = Sensor_WriteCoefficient( I2C0_BUS_1, SI70XX_ADDR, Address, 
            CoefficientTable.Output[x] );
         if ( Result != I2C0_STATUS_SUCCESS ) return;
         Address += 2;
      }

      /* Write the "slope" coefficients */
      for ( x=0; x<COEFFICIENT_COUNT; x++ )
      {
         Result = Sensor_WriteCoefficient( I2C0_BUS_1, SI70XX_ADDR, Address, 
            CoefficientTable.Slope[x] );
         if ( Result != I2C0_STATUS_SUCCESS ) return;
         Address += 2;
      }
      
      /* Reset the Si7013 to activate the coefficients */
      I2C0_WriteData( I2C0_BUS_1, SI70XX_ADDR, CMD_RESET, 0, 0 );

      /* Wait for the reset to complete */
      Delay( 10 );
   }
}


/*****************************************************************************/
/* Sensor_WakeUp                                                             */
/*****************************************************************************/

void Sensor_WakeUp( int Bus )
{
   /* The Si70xx automatically wakes up when you give it a command */
}


/*****************************************************************************/
/* Sensor_Sleep                                                              */
/*****************************************************************************/

void Sensor_Sleep( int Bus )
{
   /* The Si70xx automatically sleeps between commands */
}


/*****************************************************************************/
/* Sensor_Measure                                                            */
/*****************************************************************************/

int Sensor_Measure( int Bus, uint8_t Command, int *Value )
{
   uint8_t  Data[2];
   int      Result;

   /* Measure the humidity or temperature value */
   Result = I2C0_ReadData( Bus, SI70XX_ADDR, Command, Data, 2 );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Swap the bytes and clear the status bits */
   *Value = ((Data[0]*256) + Data[1]) & ~3;
   
   return I2C0_STATUS_SUCCESS;
}


/*****************************************************************************/
/* Sensor_ReadTemperature                                                    */
/*                                                                           */
/* Temperature is returned in deci-degrees Celsius.                          */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

int Sensor_ReadTemperature( int Bus, int *Temperature )
{
   int Value;
   int Result;

   /* Measure the temperature */
   Result = Sensor_Measure( Bus, CMD_MEASURE_TEMPERATURE_HOLD, &Value );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Convert the temperature to deci-degree Celsius */
   *Temperature = ((Value*1757)>>16) - 469;

   return I2C0_STATUS_SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ReadThermistor                                                     */
/*                                                                           */
/* Thermistor is returned in deci-degrees Celsius.                           */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

int Sensor_ReadThermistor( int Bus, int *Temperature )
{
   int Value;
   int Result;

   /* Bias the thermistor and use the correction coefficients */
   Result = I2C0_WriteByte( Bus, SI70XX_ADDR, CMD_WRITE_REGISTER_2, 
      REG2_VREF_VDD|REG2_VIN_BUFFERED|REG2_RESERVED|REG2_MODE_CORRECTION );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Give the bias voltage a chance to settle */
   Delay( 10 );

   /* Measure the temperature */
   Result = Sensor_Measure( Bus, CMD_MEASURE_THERMISTOR_HOLD, &Value );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Unbias the thermistor and stop using the correction coefficients */
   Result = I2C0_WriteByte( Bus, SI70XX_ADDR, CMD_WRITE_REGISTER_2, 
      REG2_VOUT|REG2_VREF_VDD|REG2_VIN_BUFFERED|REG2_RESERVED ); 
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Convert the temperature to deci-degree Celsius */
   *Temperature = ((Value*1757)>>16) - 469;

   return I2C0_STATUS_SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ReadHumidity                                                       */
/*                                                                           */
/* Humidity is returned in deci-percent relative humidity.                   */
/* For example: 48.2 percent relative humidity is returned as 482.           */
/*                                                                           */
/*****************************************************************************/

int Sensor_ReadHumidity( int Bus, int Temperature, int *Humidity )
{
   int Value;
   int Decipercent;
   int Result;

   /* Measure the humidity */
   Result = Sensor_Measure( Bus, CMD_MEASURE_HUMIDITY_HOLD, &Value );
   if ( Result != I2C0_STATUS_SUCCESS ) return Result;

   /* Convert the humidity to deci-percent */
   Decipercent = ((Value*625)>>15) - 60;

   /* Limit the humidity to valid values */
   if ( Decipercent < 0 )
      *Humidity = 0;
   else if ( Decipercent > 1000 )
      *Humidity = 1000;
   else
      *Humidity = Decipercent;      

   return I2C0_STATUS_SUCCESS;   
}

