/***************************************************************************//**
 * @file
 * @brief Top level application functions
 *******************************************************************************
 * # License
 * <b>Copyright 2020 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/
#include <stdint.h>
#include <stdbool.h>

#if defined(SL_COMPONENT_CATALOG_PRESENT)
#include "sl_component_catalog.h"
#endif

#include "em_core.h"
#include "em_cmu.h"
#include "em_gpio.h"

#include "sl_status.h"
#include "sl_atomic.h"

#include "app.h"
#include "sl_sensor_rht.h"

#include "sl_cli_command.h"
#include "sl_cli_handles.h"

#if !defined(SL_CATALOG_KERNEL_PRESENT)
#include "sl_sleeptimer.h"

static volatile bool measurement_flag = true;
static sl_sleeptimer_timer_handle_t measurement_timer;
static void measurement_callback(sl_sleeptimer_timer_handle_t *handle, void *data);
#else
#include "cmsis_os2.h"
#include "sl_cmsis_os2_common.h"

#define STACK_SIZE      2048

__ALIGNED(8) static uint8_t task_stack[(STACK_SIZE * sizeof(void *)) & 0xFFFFFFF8u];
__ALIGNED(4) static uint8_t thread_cb[osThreadCbSize];

osThreadId_t thread_id;
static void task(void *arg);
#endif

/** Time (in ms) between periodic updates of the measurements. */
#define MEASUREMENT_INTERVAL_MS      2000

uint32_t rhData = 0;
int32_t tempData = 0;

extern const sl_cli_command_entry_t app_cli_command_table[];

extern sl_cli_handle_t sl_cli_vcom_handle;
extern sl_cli_command_group_t sl_cli_vcom_command_group;

sl_cli_command_group_t sl_cli_app_command_group =
{
  { NULL },
  false,
  app_cli_command_table
};

/***************************************************************************//**
 * Initialize application.
 ******************************************************************************/
void app_init(void)
{
  sl_sensor_rht_init();

#if !defined(SL_CATALOG_KERNEL_PRESENT)
  /* Set up periodic measurement timer. */
  sl_sleeptimer_start_periodic_timer_ms(&measurement_timer, MEASUREMENT_INTERVAL_MS, measurement_callback, NULL, 0, 0);
#else
  osThreadAttr_t attr;

  attr.name = "si7013 app task";
  attr.priority = 25;
  attr.stack_mem = task_stack;
  attr.stack_size = STACK_SIZE;
  attr.cb_mem = thread_cb;
  attr.cb_size = osThreadCbSize;
  attr.attr_bits = 0u;
  attr.tz_module = 0u;

  thread_id = osThreadNew(&task, NULL, &attr);
  EFM_ASSERT(thread_id != NULL);
#endif

  // Remove unused autogen CLI command group
  sl_cli_command_remove_command_group(sl_cli_vcom_handle, &sl_cli_vcom_command_group);

  // Add application CLI commands
  sl_cli_command_add_command_group(sl_cli_vcom_handle, &sl_cli_app_command_group);
}

#if !defined(SL_CATALOG_KERNEL_PRESENT)
/***************************************************************************//**
 * App ticking function.
 ******************************************************************************/
void app_process_action(void)
{
  CORE_DECLARE_IRQ_STATE;

  CORE_ENTER_CRITICAL();
  if (measurement_flag) {
    measurement_flag = false;
    CORE_EXIT_CRITICAL();
    sl_sensor_rht_get(&rhData, &tempData);
  } else {
  CORE_EXIT_CRITICAL();
  }
}

static void measurement_callback(sl_sleeptimer_timer_handle_t *handle, void *data)
{
  (void)handle;
  (void)data;
  sl_atomic_store(measurement_flag, true);
}

#else // SL_CATALOG_KERNEL_PRESENT

static void task(void *arg)
{
(void)arg;

while (1) {
  sl_sensor_rht_get(&rhData, &tempData);
  osDelay(((uint64_t)osKernelGetTickFreq() * MEASUREMENT_INTERVAL_MS) / 1000);
}
}
#endif

