/***************************************************************************//**
 * @file
 * @brief Top level application functions
 *******************************************************************************
 * # License
 * <b>Copyright 2020 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/
#include <stdint.h>
#include <stdbool.h>

#include "sl_app_assert.h"
#include "sl_bluetooth.h"
#include "sl_simple_timer.h"
#include "gatt_db.h"

extern uint32_t rhData;
extern int32_t tempData;
extern uint32_t adcData;

// Connection handle.
static uint8_t app_connection = 0;

// Periodic timer handle.
static sl_simple_timer_t app_periodic_timer;

// The advertising set handle allocated from Bluetooth stack.
static uint8_t advertising_set_handle = 0xff;

// The Measurement Interval is the time interval between two measurement indications.
#define SL_BT_HT_MEASUREMENT_INTERVAL_SEC   1

// Periodic timer callback.
static void app_periodic_relative_humidity_timer_cb(sl_simple_timer_t *timer, void *data);

// Periodic timer callback.
static void app_periodic_temperature_timer_cb(sl_simple_timer_t *timer, void *data);

// Periodic timer callback.
static void app_periodic_adc_voltage_timer_cb(sl_simple_timer_t *timer, void *data);

/**************************************************************************//**
 * Callback function of connection close event.
 *
 * @param[in] reason Unused parameter required by the health_thermometer component
 * @param[in] connection Unused parameter required by the health_thermometer component
 *****************************************************************************/
void sl_bt_connection_closed_cb(uint16_t reason, uint8_t connection)
{
  (void)reason;
  (void)connection;
  sl_status_t sc;

  // Stop timer.
  sc = sl_simple_timer_stop(&app_periodic_timer);
  sl_app_assert(sc == SL_STATUS_OK,
                "[E: 0x%04x] Failed to stop periodic timer\n",
                (int)sc);
}

/**************************************************************************//**
 * Relative Humidity - Humidity Measurement
 * Indication changed callback
 *
 * Called when indication of humidity measurement is enabled/disabled by
 * the client.
 *****************************************************************************/
void sl_bt_relative_humidity_indication(uint8_t connection,
                                  gatt_client_config_flag_t client_config)
{
  sl_status_t sc;
  app_connection = connection;
  // Indication or notification enabled.
  if (gatt_disable != client_config) {
    // Start timer used for periodic indications.
    sc = sl_simple_timer_start(&app_periodic_timer,
                               SL_BT_HT_MEASUREMENT_INTERVAL_SEC * 1000,
                               app_periodic_relative_humidity_timer_cb,
                               NULL,
                               true);
    sl_app_assert(sc == SL_STATUS_OK,
                  "[E: 0x%04x] Failed to start periodic timer\n",
                  (int)sc);
    // Send first indication.
    app_periodic_relative_humidity_timer_cb(&app_periodic_timer, NULL);
  }
  // Indications disabled.
  else {
    // Stop timer used for periodic indications.
    (void)sl_simple_timer_stop(&app_periodic_timer);
  }
}

/**************************************************************************//**
 * Timer callback
 * Called periodically to time periodic humidity indications.
 *****************************************************************************/
static void app_periodic_relative_humidity_timer_cb(sl_simple_timer_t *timer, void *data)
{
  (void)data;
  (void)timer;

  uint32_t relative_humidity = rhData / 1000;

  uint8_t buf[4] = { 0 };

  sprintf((char*) buf, "%d%%", relative_humidity);

  sl_bt_gatt_server_send_indication(
    app_connection,
    gattdb_humidity,
    sizeof(buf),
    buf);
}

/**************************************************************************//**
 * Temperature - Temperature Measurement
 * Indication changed callback
 *
 * Called when indication of temperature measurement is enabled/disabled by
 * the client.
 *****************************************************************************/
void sl_bt_temperature_indication(uint8_t connection,
                                  gatt_client_config_flag_t client_config)
{
  sl_status_t sc;
  app_connection = connection;
  // Indication or notification enabled.
  if (gatt_disable != client_config) {
    // Start timer used for periodic indications.
    sc = sl_simple_timer_start(&app_periodic_timer,
                               SL_BT_HT_MEASUREMENT_INTERVAL_SEC * 1000,
                               app_periodic_temperature_timer_cb,
                               NULL,
                               true);
    sl_app_assert(sc == SL_STATUS_OK,
                  "[E: 0x%04x] Failed to start periodic timer\n",
                  (int)sc);
    // Send first indication.
    app_periodic_temperature_timer_cb(&app_periodic_timer, NULL);
  }
  // Indications disabled.
  else {
    // Stop timer used for periodic indications.
    (void)sl_simple_timer_stop(&app_periodic_timer);
  }
}

/**************************************************************************//**
 * Timer callback
 * Called periodically to time periodic temperature indications.
 *****************************************************************************/
static void app_periodic_temperature_timer_cb(sl_simple_timer_t *timer, void *data)
{
  (void)data;
  (void)timer;

  uint32_t temperature = tempData / 1000;
  uint8_t buf[5] = { 0 };

  sprintf((char*) buf, "%d C", temperature);

  sl_bt_gatt_server_send_indication(
    app_connection,
    gattdb_temperature,
    sizeof(buf),
    buf);
}

/**************************************************************************//**
 * Analog - ADC Voltage Measurement
 * Indication changed callback
 *
 * Called when indication of ADC measurement is enabled/disabled by
 * the client.
 *****************************************************************************/
void sl_bt_adc_voltage_indication(uint8_t connection,
                                  gatt_client_config_flag_t client_config)
{
  sl_status_t sc;
  app_connection = connection;
  // Indication or notification enabled.
  if (gatt_disable != client_config) {
    // Start timer used for periodic indications.
    sc = sl_simple_timer_start(&app_periodic_timer,
                               SL_BT_HT_MEASUREMENT_INTERVAL_SEC * 1000,
                               app_periodic_adc_voltage_timer_cb,
                               NULL,
                               true);
    sl_app_assert(sc == SL_STATUS_OK,
                  "[E: 0x%04x] Failed to start periodic timer\n",
                  (int)sc);
    // Send first indication.
    app_periodic_adc_voltage_timer_cb(&app_periodic_timer, NULL);
  }
  // Indications disabled.
  else {
    // Stop timer used for periodic indications.
    (void)sl_simple_timer_stop(&app_periodic_timer);
  }
}

/**************************************************************************//**
 * Timer callback
 * Called periodically to time periodic ADC indications.
 *****************************************************************************/
static void app_periodic_adc_voltage_timer_cb(sl_simple_timer_t *timer, void *data)
{
  (void)data;
  (void)timer;

  uint32_t adc_voltage = (adcData * 2500) / 4096;
  uint8_t buf[7] = { 0 };

  sprintf((char*) buf, "%d mV", adc_voltage);

  sl_bt_gatt_server_send_indication(
    app_connection,
    gattdb_analog,
    sizeof(buf),
    buf);
}

/**************************************************************************//**
 * Bluetooth stack event handler.
 * This overrides the dummy weak implementation.
 *
 * @param[in] evt Event coming from the Bluetooth stack.
 *****************************************************************************/
void sl_bt_on_event(sl_bt_msg_t *evt)
{
  sl_status_t sc;
  bd_addr address;
  uint8_t address_type;
  uint8_t system_id[8];

  switch (SL_BT_MSG_ID(evt->header)) {
    // -------------------------------
    // This event indicates the device has started and the radio is ready.
    // Do not call any stack command before receiving this boot event!
    case sl_bt_evt_system_boot_id:

      // Extract unique ID from BT Address.
      sc = sl_bt_system_get_identity_address(&address, &address_type);
      sl_app_assert(sc == SL_STATUS_OK,
                    "[E: 0x%04x] Failed to get Bluetooth address\n",
                    (int)sc);

      // Pad and reverse unique ID to get System ID.
      system_id[0] = address.addr[5];
      system_id[1] = address.addr[4];
      system_id[2] = address.addr[3];
      system_id[3] = 0xFF;
      system_id[4] = 0xFE;
      system_id[5] = address.addr[2];
      system_id[6] = address.addr[1];
      system_id[7] = address.addr[0];

      sc = sl_bt_gatt_server_write_attribute_value(gattdb_system_id,
                                                   0,
                                                   sizeof(system_id),
                                                   system_id);
      sl_app_assert(sc == SL_STATUS_OK,
                    "[E: 0x%04x] Failed to write attribute\n",
                    (int)sc);

      // Create an advertising set.
      sc = sl_bt_advertiser_create_set(&advertising_set_handle);
      sl_app_assert(sc == SL_STATUS_OK,
                    "[E: 0x%04x] Failed to create advertising set\n",
                    (int)sc);

      // Set advertising interval to 100ms.
      sc = sl_bt_advertiser_set_timing(
        advertising_set_handle,
        160, // min. adv. interval (milliseconds * 1.6)
        160, // max. adv. interval (milliseconds * 1.6)
        0,   // adv. duration
        0);  // max. num. adv. events
      sl_app_assert(sc == SL_STATUS_OK,
                    "[E: 0x%04x] Failed to set advertising timing\n",
                    (int)sc);
      // Start general advertising and enable connections.
      sc = sl_bt_advertiser_start(
        advertising_set_handle,
        advertiser_general_discoverable,
        advertiser_connectable_scannable);
      sl_app_assert(sc == SL_STATUS_OK,
                    "[E: 0x%04x] Failed to start advertising\n",
                    (int)sc);
      break;

    // -------------------------------
    // This event indicates that a new connection was opened.
    case sl_bt_evt_connection_opened_id:
      break;

    // -------------------------------
    // This event indicates that a connection was closed.
    case sl_bt_evt_connection_closed_id:
      // Restart advertising after client has disconnected.
      sc = sl_bt_advertiser_start(
        advertising_set_handle,
        advertiser_general_discoverable,
        advertiser_connectable_scannable);
      sl_app_assert(sc == SL_STATUS_OK,
                    "[E: 0x%04x] Failed to start advertising\n",
                    (int)sc);
      break;

    ///////////////////////////////////////////////////////////////////////////
    // Add additional event handlers here as your application requires!      //
    ///////////////////////////////////////////////////////////////////////////
    case sl_bt_evt_gatt_server_characteristic_status_id:
          if (gattdb_humidity == evt->data.evt_gatt_server_characteristic_status.characteristic) {
              if (gatt_server_client_config == (gatt_server_characteristic_status_flag_t)evt->data.evt_gatt_server_characteristic_status.status_flags) {
                  sl_bt_relative_humidity_indication(
                  evt->data.evt_gatt_server_characteristic_status.connection,
                  evt->data.evt_gatt_server_characteristic_status.client_config_flags);
              }
          }
          else if (gattdb_temperature == evt->data.evt_gatt_server_characteristic_status.characteristic) {
                  if (gatt_server_client_config == (gatt_server_characteristic_status_flag_t)evt->data.evt_gatt_server_characteristic_status.status_flags) {
                    sl_bt_temperature_indication(
                      evt->data.evt_gatt_server_characteristic_status.connection,
                      evt->data.evt_gatt_server_characteristic_status.client_config_flags);
                  }
          }
          else if (gattdb_analog == evt->data.evt_gatt_server_characteristic_status.characteristic) {
                  if (gatt_server_client_config == (gatt_server_characteristic_status_flag_t)evt->data.evt_gatt_server_characteristic_status.status_flags) {
                      sl_bt_adc_voltage_indication(
                      evt->data.evt_gatt_server_characteristic_status.connection,
                      evt->data.evt_gatt_server_characteristic_status.client_config_flags);
                  }
          }
          break;

    // -------------------------------
    // Default event handler.
    default:
      break;
  }
}
